/*************************************************************************
*
* ADOBE CONFIDENTIAL
* ___________________
*
*  Copyright 1997 Adobe Systems Incorporated
*  All Rights Reserved.
*
* NOTICE:  All information contained herein is, and remains
* the property of Adobe Systems Incorporated and its suppliers,
* if any.  The intellectual and technical concepts contained
* herein are proprietary to Adobe Systems Incorporated and its
* suppliers and are protected by trade secret or copyright law.
* Dissemination of this information or reproduction of this material
* is strictly forbidden unless prior written permission is obtained
* from Adobe Systems Incorporated.
**************************************************************************/
package com.day.crx.statistics.result;

import com.day.crx.statistics.Report;

import javax.jcr.Session;
import javax.jcr.RepositoryException;
import javax.jcr.Node;
import javax.jcr.Item;
import java.util.Iterator;
import java.util.Calendar;
import java.util.List;
import java.util.ArrayList;

import org.apache.jackrabbit.util.Text;

/**
 * <code>ResultHistoryReport</code> implements a report that returns historical
 * data about a given result.
 *
 * @author mreutegg
 */
public class ResultHistoryReport extends Report {

    /**
     * Default period is 12 months
     */
    private int period = 12;

    /**
     * The path of the result node.
     */
    private final String path;

    /**
     * Creates a new report.
     *
     * @param dataPath the location where the result statistics are stored.
     * @param path     the path of the result node.
     */
    public ResultHistoryReport(String dataPath, String path) {
        super(dataPath);
        this.path = path;
    }

    /**
     * {@inheritDoc}
     * Returns result rows with the following objects:
     * <ul>
     * <li>Path <code>String</code>, which indicates the month</li>
     * <li><code>Long</code> count (how many times the result was selected)</li>
     * </ul>
     */
    public Iterator getResult(Session session) throws RepositoryException {
        ResultSelected rs = new ResultSelected(getDataPath(), path, 1, "");
        Calendar cal = Calendar.getInstance();
        cal.set(Calendar.DAY_OF_MONTH, 1);
        cal.add(Calendar.MONTH, -(getPeriod() - 1));
        List data = new ArrayList();
        for (int i = 0; i < getPeriod(); i++) {
            rs.setTimestamp(cal.getTimeInMillis());
            String p = Text.getRelativeParent(rs.getPath(), 1);
            long count = 0;
            if (session.itemExists(p)) {
                Item item = session.getItem(p);
                if (item.isNode()) {
                    count = ((Node) item).getProperty("count").getLong();
                }
            }
            data.add(new Object[]{p, new Long(count)});
            cal.add(Calendar.MONTH, 1);
        }
        return data.iterator();
    }

    /**
     * @return the report period in number of days.
     */
    public int getPeriod() {
        return period;
    }

    /**
     * @param period the report period in number of days.
     */
    public void setPeriod(int period) {
        this.period = period;
    }
}
