/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.text;

import java.util.StringTokenizer;
import java.util.List;
import java.util.LinkedList;
import java.util.Iterator;

/**
 * The <code>StringTable</code> class easy handling of string tables, especially
 * creating tables from strings and lists. The table constructed is guaranteed
 * to be complete in that each row has the exact same number of columns and
 * each table cell is guaranteed to not be <code>null</code>.
 */
public class StringTable {

    /* default logger */
    //private static final FmtLogger log =
    //        (FmtLogger) FmtLogger.getLogger(StringTable.class);

    /** The empty string to fill sparse cells */
    protected static final String EMPTY_STRING = "";

    /** Constant denoting last field was not a delimiter */
    protected static final int DELIM_TYPE_NONE = 0x00;

    /** Constant denoting last field was a column delimiter */
    protected static final int DELIM_TYPE_COL  = 0x01;

    /** Constant denoting last field was a row delimiter */
    protected static final int DELIM_TYPE_ROW  = 0x02;

    /** Constant denoting last field was a CR row delimiter */
    protected static final int DELIM_TYPE_CR   = 0x04;

    /** The table itself */
    private  String[][] table;

    /** The number of rows in the table */
    private int numRows;

    /** The max number of cols in any one row */
    private int numCols;

    /**
     * Constructs a <code>StringTable</code> object from the list of lists.
     *
     * @param numRows The number of rows of the table
     * @param numCols The number of columns of the table
     * @param rowList The <code>List</code> of lists to get the table data
     * 		from.
     */
    protected StringTable(int numRows, int numCols, List rowList) {
	this.numRows = numRows;
	this.numCols = numCols;
	this.table = new String[numRows][numCols];

	Iterator rowIter = rowList.iterator();
	for (int row=0; rowIter.hasNext(); row++) {
            List colList = (List) rowIter.next();

            // copy the columns to the (beginning of the) array
            colList.toArray(table[row]);

	    // fill rest of array entries with the empty string
	    for (int col=colList.size(); col<numCols; col++) {
                table[row][col] = EMPTY_STRING;
            }
	}
    }

    /**
     * Constructor does the hard work converting the string
     * Creates a new table from the string, where the fields are separated
     * by any one of the delimiters. Consecutive delimiters define empty
     * field values. The table rows in the string are separated by either
     * CR, LF or both. So neither CR nor LF may be used as a field delimiter.
     * <p>
     * The string is logically split on CR/LF to get the rows. Each row is
     * then split on the delimiters to get the cells.
     * <p>
     * Any empty rows in the string table are preserved as empty rows in the
     * resulting table.
     *
     * @param srcString The string from which to construct the table
     * @param delims The delimiters for columns.
     *
     * @return The table from the string.
     */
    public static StringTable fromString(String srcString, String delims) {
        return fromString(srcString, delims, true);
    }

    /**
     * Constructor does the hard work converting the string
     * Creates a new table from the string, where the fields are separated
     * by any one of the delimiters. Consecutive delimiters define empty
     * field values. The table rows in the string are separated by either
     * CR, LF or both. So neither CR nor LF may be used as a field delimiter.
     * <p>
     * The string is logically split on CR/LF to get the rows. Each row is
     * then split on the delimiters to get the cells.
     *
     * @param srcString The string from which to construct the table
     * @param delims The delimiters for columns.
     * @param preserveEmptyRows If <code>true</code> empty rows in the input
     *      string are added as empty rows in the table. Otherwise empty rows
     *      in the input are ignored.
     *
     * @return The table from the string.
     */
    public static StringTable fromString(String srcString, String delims,
            boolean preserveEmptyRows) {
	int maxCols = -1;

	delims += "\r\n";
	StringTokenizer tokener = new StringTokenizer(srcString, delims, true);
	int delim = DELIM_TYPE_ROW;

	// prepare lists
	List rowList = new LinkedList();
	List colList = new LinkedList();

	while (tokener.hasMoreTokens()) {
	    String tok = tokener.nextToken();

	    if (tok.length() == 1 && delims.indexOf(tok.charAt(0)) >= 0) {
		// delimiter
		char d = tok.charAt(0);
		if (d == '\r' || d == '\n') {
		    // row end

		    // check for empty row
		    if (d != '\n' || delim != DELIM_TYPE_CR) {
			if (colList.size() > 0 || preserveEmptyRows) {
                            rowList.add(colList);
                            if (colList.size() > maxCols) {
                                maxCols = colList.size();
                            }
                        }
			colList = new LinkedList();
		    }

		    // set current delimiter
		    delim = (d == '\r') ? DELIM_TYPE_CR : DELIM_TYPE_ROW;
		} else {
		    // assume column delimiter

		    // check for empty column
		    if (delim != DELIM_TYPE_NONE) {
			colList.add(EMPTY_STRING);
		    }

		    // set current delimiter
		    delim = DELIM_TYPE_COL;
		}
	    } else {
		// element
		if (colList == null) {
		    colList = new LinkedList();
		}
		colList.add(tok);
		delim = DELIM_TYPE_NONE;
	    }
	}

	// we might have to add the last column list
	if (delim != DELIM_TYPE_ROW && delim != DELIM_TYPE_CR) {
	    rowList.add(colList);
	    if (colList.size() > maxCols) {
		maxCols = colList.size();
	    }
	}

	return new StringTable(rowList.size(), maxCols, rowList);
    }

    /**
     * Transposes the table - cols will be rows and rows will be cols. The
     * operation is that the table is mirrored on the diagonal from 0/0 to n/n.
     */
    public StringTable transpose() {
	String[][] tmpTab = new String[numCols][numRows];

	// Copy the table
	for (int row=0; row<numRows; row++) {
	    for (int col = 0; col<numCols; col++) {
		tmpTab[col][row] = table[row][col];
	    }
	}

	// Replace the existing table
	table = tmpTab;

	// Swap the etries
	int t = numRows;
	numRows = numCols;
	numCols = t;

	return this;
    }

    /**
     * Returns the string table created. This is the original table, so
     * modifying entries in this object, also modifies the table. Especially
     * modifying array elements of the first dimension may invalidate the
     * class invariant, in that the table is not complete anymore. User beware.
     *
     * @return The string table
     */
    public String[][] getTable() {
	return table;
    }

    /**
     * Returns the number of rows in the table.
     */
    public int getRows() {
	return numRows;
    }

    /**
     * Returns the number of columns in the table.
     */
    public int getCols() {
	return numCols;
    }

}
