/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 2014 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.adobe.granite.socketio;

import java.io.IOException;
import java.security.Principal;
import java.util.Set;

import javax.annotation.Nonnull;

import org.apache.sling.commons.json.JSONArray;

import org.osgi.annotation.versioning.ProviderType;

/**
 * A Socket is the fundamental class for interacting with browser clients.
 * A Socket belongs to a certain Namespace (by default /) and uses an underlying Client to communicate.
 */
@ProviderType
public interface SocketIOSocket extends SocketIOEmitter {

    /**
     * Sets a listener to this socket that will receive the socket events.
     *
     * @param listener the listener
     * @return this
     * @throws java.lang.IllegalArgumentException if this socket already has a listener
     */
    @Nonnull
    SocketIOSocket setListener(@Nonnull SocketIOSocketListener listener);

    /**
     * Removes a listener from this socket. If this socket has no listener defined, this method has no effect.
     *
     * @param listener the listener
     * @return this
     * @throws java.lang.IllegalArgumentException if this socket already has a listener defined that is not the given one.
     */
    @Nonnull
    SocketIOSocket removeListener(@Nonnull SocketIOSocketListener listener);

    /**
     * A unique identifier for the socket session, that comes from the underlying Client.
     * @return the id
     */
    @Nonnull
    String getId();

    /**
     * Client namespace
     *
     * @return - namespace
     */
    @Nonnull
    SocketIONamespace getNamespace();

    /**
     * A list of strings identifying the rooms this socket is in.
     * @return set of rooms
     */
    @Nonnull
    Set<String> getRooms();

    /**
     * Disconnects this socket.
     *
     * @param close {@code true} to also close the underlying connection
     * @return this
     */
    @Nonnull
    SocketIOSocket disconnect(boolean close);

    /**
     * Adds the socket to the room, and fires optionally a callback fn with err signature (if any).
     *
     * The socket is automatically a member of a room identified with its
     * session id (see Socket#id).
     *
     * The mechanics of joining rooms are handled by the Adapter
     * that has been configured (see Server#adapter above), defaulting to
     * socket.io-adapter.
     *
     * @param name room name
     * @return this
     */
    @Nonnull
    SocketIOSocket join(@Nonnull String name);

    /**
     * Removes the socket from room, and fires optionally a callback fn
     * with err signature (if any).
     *
     * Rooms are left automatically upon disconnection.
     *
     * The mechanics of leaving rooms are handled by the Adapter
     * that has been configured (see Server#adapter above), defaulting to
     * socket.io-adapter.
     *
     * @param name room name
     * @return this
     */
    @Nonnull
    SocketIOSocket leave(@Nonnull String name);

    /**
     * Sets a modifier for a subsequent event emission that the event will
     * sent to all sockets except to itself.
     * @return an emitter
     */
    @Nonnull
    SocketIOEmitter broadcast();

    /**
     * Emits an event to the socket identified by the string name. Any other parameters can be included.
     * The called needs to specify an {@link SocketIOAckListener} that will be called after the client sent the
     * ack response.
     *
     * @param eventName name of the event
     * @param ackListener the acknowledge listener
     * @param arguments optional arguments
     * @return this
     * @throws IOException error
     */
    @Nonnull
    SocketIOSocket emit(@Nonnull String eventName, @Nonnull SocketIOAckListener ackListener,
                        @Nonnull Object... arguments) throws IOException;

    /**
     * Emits an event to the socket identified by the string name. Any other parameters can be included.
     * The called needs to specify an {@link SocketIOAckListener} that will be called after the client sent the
     * ack response.
     *
     * @param eventName name of the event
     * @param ackListener the acknowledge listener
     * @param arguments optional arguments
     * @return this
     * @throws IOException error
     */
    @Nonnull
    SocketIOSocket emit(@Nonnull String eventName, @Nonnull SocketIOAckListener ackListener,
                        @Nonnull JSONArray arguments) throws IOException;

    /**
     * @return the {@link java.security.Principal} of the current authenticated user.
     */
    Principal getUserPrincipal();

}