/***************************************************************************/
/*                                                                         */
/*                      ADOBE CONFIDENTIAL                                 */
/*                      _ _ _ _ _ _ _ _ _ _                                */
/*                                                                         */
/*  Copyright 2001-2002, Adobe Systems Incorporated                        */
/*  All Rights Reserved.                                                   */
/*                                                                         */
/*  NOTICE: All information contained herein is, and remains the property  */
/*  of Adobe Systems Incorporated and its suppliers, if any.  The          */
/*  intellectual and technical concepts contained herein are proprietary   */
/*  to Adobe Systems Incorporated and its suppliers and may be covered by  */
/*  U.S. and Foreign Patents, patents in process, and are protected by     */
/*  trade secret or copyright law.  Dissemination of this information or   */
/*  reproduction of this material is strictly forbidden unless prior       */
/*  written permission is obtained from Adobe Systems Incorporated.        */
/*                                                                         */
/***************************************************************************/
package com.adobe.versioncue.nativecomm.msg;

import java.nio.ByteBuffer;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

/**
 * A Dictionary mapping <code>NCType</code> keys to <code>NCType</code> objects.
 * 
 * Note that this class is **NOT** Thread-safe in its current form.
 * 
 * @author <a href="mailto:tnaroska@adobe.com">Timo Naroska</a>
 * @version $Revision: #1 $
 */
public final class NCMap extends NCType
{
	// --------------------------------------------------------------------------- Private Variables

	/** The underlying Map object */
	private final HashMap<NCType, NCType> map;

	// -------------------------------------------------------------------------- Public Constructor

	/** Default constructor */
	public NCMap()
	{
		map = new HashMap<NCType, NCType>();
	}

	// ------------------------------------------------------------------------------ Public Methods

	/**
	 * Returns the number of entries in the NCMap.
	 * 
	 * @return number of entries in the NCMap.
	 */
	public int size()
	{
		return map.size();
	}

	/**
	 * Returns true if the NCMap is empty
	 * 
	 * @return true if the NCMap is empty
	 */
	public boolean isEmtpy()
	{
		return map.isEmpty();
	}

	/**
	 * Returns an iterator over the mappings in this IDicitonary. The elements
	 * are returned in no particular order.
	 * 
	 * @return a <code>Iterator</code> containing
	 *         <code>java.util.Map.Entry</code> Objects
	 */
	public Iterator<Map.Entry<NCType, NCType>> entries()
	{
		return map.entrySet().iterator();
	}

	/**
	 * Returns a set of Map.Entry objects.
	 * 
	 * @return set of Map.Entry objects.
	 */
	public Set<Map.Entry<NCType, NCType>> entrySet()
	{
		return map.entrySet();
	}

	/**
	 * Returns true if this NCMap contains a mapping for the specified key.
	 * 
	 * @param key - key to check
	 * @return true if this NCMap contains a mapping for the specified key.
	 */
	public boolean containsKey(NCType key)
	{
		return map.containsKey(key);
	}

	/**
	 * Returns true if this NCMap contains a mapping for the specified key.
	 * 
	 * @param key - key to check
	 * @return true if this NCMap contains a mapping for the specified key.
	 */
	public boolean containsKey(String key)
	{
		return map.containsKey(new NCString(key));
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(NCType key, NCType value)
	{
		// == PRE-CONDITION ========================================================================
		if (key == null)
			throw new NullPointerException("key can't be null"); //$NON-NLS-1$
		if (value == null)
			throw new NullPointerException("value can't be null"); //$NON-NLS-1$
		// == PRE-CONDITION ========================================================================

		map.put(key, value);
		return this;
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, NCType value)
	{
		return put(new NCString(key), value);
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, INCExternalizable value)
	{
		return put(new NCString(key), value.externalize());
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, String value)
	{
		return put(new NCString(key), new NCString(value));
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, boolean value)
	{
		return put(new NCString(key), value ? NCBool.TRUE : NCBool.FALSE);
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, int value)
	{
		return put(new NCString(key), new NCInt(value));
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, long value)
	{
		return put(new NCString(key), new NCLong(value));
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, double value)
	{
		return put(new NCString(key), new NCDouble(value));
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, Date value)
	{
		return put(new NCString(key), new NCDate(value));
    }

	/** Add a key/value pair to this map.
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, byte[] value)
	{
		return put(new NCString(key), new NCData(value));
	}

	/**
	 * Add a key/value pair to this map.
	 * 
	 * @param key the key
	 * @param value the value
	 * @return the NCMap
	 */
	public NCMap put(String key, ByteBuffer value)
	{
		return put(new NCString(key), new NCData(value));
	}

	/**
	 * Returns the NCType object associated with the specified key. Returns null
	 * if the NCMap contains no entry for this key.
	 * 
	 * @param key key to retrieve
	 * @return value associated with the specified key; null if the key has no
	 *         value associated
	 */
	public NCType get(NCType key)
	{
		return map.get(key);
	}

	/**
	 * Returns the NCType object associated with the specified key. Returns null
	 * if the NCMap contains no entry for this key.
	 * 
	 * @param key key to retrieve
	 * @return value associated with the specified key; null if the key has no
	 *         value associated
	 */
	public NCType get(String key)
	{
		return get(new NCString(key));
	}

	/**
	 * Returns the String value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return String value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public String getString(String key) throws BadMessageException
	{
		NCString value = (NCString) checkedGet(key, NCString.class);
		return value.string();
	}

	/**
	 * Returns the boolean value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return boolean value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public boolean getBool(String key) throws BadMessageException
	{
		NCBool value = (NCBool) checkedGet(key, NCBool.class);
		return value.booleanValue();
	}

	/**
	 * Returns the double value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return double value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public double getDouble(String key) throws BadMessageException
	{
		NCNumber value = (NCNumber) checkedGet(key, NCNumber.class);
		return value.doubleValue();
	}

	/**
	 * Returns the int value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return int value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public int getInt(String key) throws BadMessageException
	{
		NCNumber value = (NCNumber) checkedGet(key, NCNumber.class);
		return value.intValue();
	}

	/**
	 * Returns the long value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return long value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public long getLong(String key) throws BadMessageException
	{
		NCNumber value = (NCNumber) checkedGet(key, NCNumber.class);
		return value.longValue();
	}

 	/** Returns the date value associated with the specified key.
	 * @param key the key String
	 * @return date value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public Date getDate(String key) throws BadMessageException
	{
		NCDate value = (NCDate) checkedGet(key, NCDate.class);
		return value.date();
	}

 	/** Returns the byte[] value associated with the specified key.
	 * @param key the key String
	 * @return byte[] value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public byte[] getByteArray(String key) throws BadMessageException
	{
		NCData value = (NCData) checkedGet(key, NCData.class);
		return value.byteArray();
	}

	/**
	 * Returns the ByteBuffer value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return ByteBuffer value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public ByteBuffer getBytes(String key) throws BadMessageException
	{
		NCData value = (NCData) checkedGet(key, NCData.class);
		return value.bytes();
	}

	/**
	 * Returns the NCList value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return NCList value
	 * @throws BadMessageException if the requested value is not present or of a bad type
	 */
	public NCList getList(String key) throws BadMessageException
	{
		return (NCList) checkedGet(key, NCList.class);
	}

	/**
	 * Returns the NCMap value associated with the specified key.
	 * 
	 * @param key the key String
	 * @return NCMap value
	 * @throws BadMessageException
	 *             if the associated value is not present or not of the correct
	 *             type
	 */
	public NCMap getMap(String key) throws BadMessageException
	{
		return (NCMap) checkedGet(key, NCMap.class);
	}

	/**
	 * Removes the mapping for this key from this <code>NCMap</code> if it is present.
	 * 
	 * @param key key whose associated <code>NCType</code> object is to be removed
	 * @return previous <code>NCType</code> object associated with the specified
	 *         key, or null if there was no mapping for the key
	 */
	public NCType remove(NCType key)
	{
		return map.remove(key);
	}

	/**
	 * Removes the mapping for this key from this <code>NCMap</code> if it is present.
	 * 
	 * @param key key whose associated <code>NCType</code> object is to be removed
	 * @return previous <code>NCType</code> object associated with the specified
	 *         key, or null if there was no mapping for the key
	 */
	public NCType remove(String key)
	{
		return remove(new NCString(key));
	}

	/**
	 * Removes all entries from this NCMap <code>object</code>.
	 * 
	 * @return the NCMap
	 */
	public NCMap clear()
	{
		map.clear();
		return this;
	}

	// ---------------------------------------------------------------------------- NCType Overrides

	@Override
	public int getType()
	{
		return TYPE_MAP;
	}

	// ------------------------------------------------------------------ java.lang.Object Overrides

	@Override
	public boolean equals(Object other)
	{
		if (other == this)
		{
			return true;
		}

		if (other instanceof NCMap)
		{
			final NCMap otherDict = (NCMap) other;
			return map.equals(otherDict.map);
		}
		return false;
	}

	@Override
	public String toString()
	{
		StringBuilder buf = new StringBuilder(128);

		String nl = System.getProperty("line.separator");

		buf.append(getClass().getSimpleName()).append(nl).append('{').append(nl);

		for (Entry<NCType, NCType> entry : map.entrySet())
		{
			buf.append("    ").append(entry.getKey()).append(": ")
					.append(entry.getValue()).append(nl);
		}

		buf.append('}');
		return buf.toString();
	}

	@Override
	public int hashCode()
	{
		return map.hashCode();
	}

	// ----------------------------------------------------------------------------- Private Methods

	/**
	 * Checks the type of and returns the value associated with key.
	 * 
	 * @param key the key
	 * @param expectedType expected value type
	 * @return checked value
	 * @throws BadMessageException if no value is present or if a value is of a bad type
	 */
	private NCType checkedGet(String key, Class<?> expectedType) throws BadMessageException
	{
		NCType value = get(key);
		if (!expectedType.isInstance(value))
		{
			throw new BadMessageException("Expected <" + expectedType + "> for key '" + key +
					"'. Actual value: " + value);
		}
		return value;
	}
}