/*******************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ******************************************************************************/
package com.adobe.cq.sightly;

import javax.script.Bindings;

import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.SlingHttpServletResponse;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.scripting.SlingBindings;
import org.apache.sling.api.scripting.SlingScriptHelper;
import org.apache.sling.scripting.sightly.pojo.Use;
import org.osgi.annotation.versioning.ConsumerType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.adobe.granite.xss.XSSAPI;
import com.day.cq.wcm.api.Page;
import com.day.cq.wcm.api.PageManager;
import com.day.cq.wcm.api.components.Component;
import com.day.cq.wcm.api.components.ComponentContext;
import com.day.cq.wcm.api.components.EditContext;
import com.day.cq.wcm.api.designer.Design;
import com.day.cq.wcm.api.designer.Designer;
import com.day.cq.wcm.api.designer.Style;
import com.day.cq.wcm.scripting.WCMBindingsConstants;

/**
 * Abstract implementation of {@link Use} interface. This could be extended to provide custom Java Use-API objects.
 * @see <a href="https://docs.adobe.com/docs/en/htl/docs/use-api/java.html">HTL Use-API Documentation</a>
 **/
@ConsumerType
@SuppressWarnings("unused")
public abstract class WCMUsePojo implements Use {

    private static final Logger LOG = LoggerFactory.getLogger(WCMUsePojo.class);

    private Bindings bindings;
    private SightlyWCMMode wcmMode;
    private PageManager pageManager;
    private Page currentPage;
    private Page resourcePage;
    private ValueMap pageProperties;
    private ValueMap properties;
    private Designer designer;
    private Design currentDesign;
    private Style currentStyle;
    private Component component;
    private ValueMap inheritedPageProperties;
    private Resource resource;
    private ResourceResolver resourceResolver;
    private SlingHttpServletRequest slingHttpServletRequest;
    private SlingHttpServletResponse slingHttpServletResponse;
    private SlingScriptHelper slingScriptHelper;
    private ComponentContext componentContext;
    private EditContext editContext;
    private Design resourceDesign;
    private XSSAPI xssapi;

    /**
     * Initialize bindings and calls #activate()
     *
     * @param scriptBindings Bindings to be used, there is no guarantee of having any particular bindings available.
     *
     * */
    public final void init(Bindings scriptBindings) {
        bindings = scriptBindings;
        try {
            activate();
        } catch (Exception e) {
            LOG.error("Failed to activate Use class", e);
        }
    }

    /**
     * Implement this method to perform post initialization tasks. This method is called by {@link WCMUsePojo#init(Bindings)}.
     *
     * @throws Exception in case of any error during activation
     *
     * */
    public abstract void activate() throws Exception;

    /**
     * Get an object associated with the given name
     *
     * @param name  Object property name
     * @param type  Expected object type
     *
     * @return Object or null if Object cannot be found or typed.
     *
     * */
    public final <T> T get(String name, Class<T> type) {
        Object obj = bindings.get(name);
        try {
            return type.cast(obj);
        } catch (ClassCastException e) {
            LOG.error("Failed to cast value", e);
        }
        return null;
    }

    /**
     * @return {@link WCMBindings#WCM_MODE} binding if available, otherwise null is returned
     * */
    public final SightlyWCMMode getWcmMode() {
        if (wcmMode == null) {
            wcmMode = get(WCMBindings.WCM_MODE, SightlyWCMMode.class);
        }
        return wcmMode;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_PAGE_MANAGER} binding if available, otherwise null is returned
     * */
    public final PageManager getPageManager() {
        if (pageManager == null) {
            pageManager = get(WCMBindingsConstants.NAME_PAGE_MANAGER, PageManager.class);
        }
        return pageManager;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_CURRENT_PAGE} binding if available, otherwise null is returned
     * */
    public final Page getCurrentPage() {
        if (currentPage == null) {
            currentPage = get(WCMBindingsConstants.NAME_CURRENT_PAGE, Page.class);
        }
        return currentPage;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_RESOURCE_PAGE} binding if available, otherwise null is returned
     * */
    public final Page getResourcePage() {
        if (resourcePage == null) {
            resourcePage = get(WCMBindingsConstants.NAME_RESOURCE_PAGE, Page.class);
        }
        return resourcePage;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_PAGE_PROPERTIES} binding if available, otherwise null is returned
     * */
    public final ValueMap getPageProperties() {
        if (pageProperties == null) {
            pageProperties = get(WCMBindingsConstants.NAME_PAGE_PROPERTIES, ValueMap.class);
        }
        return pageProperties;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_PROPERTIES} binding if available, otherwise null is returned
     * */
    public final ValueMap getProperties() {
        if (properties == null) {
            properties = get(WCMBindingsConstants.NAME_PROPERTIES, ValueMap.class);
        }
        return properties;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_DESIGNER} binding if available, otherwise null is returned
     * */
    public final Designer getDesigner() {
        if (designer == null) {
            designer = get(WCMBindingsConstants.NAME_DESIGNER, Designer.class);
        }
        return designer;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_CURRENT_DESIGN} binding if available, otherwise null is returned
     * */
    public final Design getCurrentDesign() {
        if (currentDesign == null) {
            currentDesign = get(WCMBindingsConstants.NAME_CURRENT_DESIGN, Design.class);
        }
        return currentDesign;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_CURRENT_STYLE} binding if available, otherwise null is returned
     * */
    public final Style getCurrentStyle() {
        if (currentStyle == null) {
            currentStyle = get(WCMBindingsConstants.NAME_CURRENT_STYLE, Style.class);
        }
        return currentStyle;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_COMPONENT} binding if available, otherwise null is returned
     * */
    public final Component getComponent() {
        if (component == null) {
            component = get(WCMBindingsConstants.NAME_COMPONENT, Component.class);
        }
        return component;
    }

    /**
     * @return {@link WCMBindings#INHERITED_PAGE_PROPERTIES} binding if available, otherwise null is returned
     * @deprecated since 3.1.0; please switch to {@link #getInheritedPageProperties()}
     **/
    @Deprecated
    public final ValueMap getInheritedProperties() {
        return getInheritedPageProperties();
    }

    /**
     * @return {@link WCMBindings#INHERITED_PAGE_PROPERTIES} binding if available, otherwise null is returned
     **/
    public final ValueMap getInheritedPageProperties() {
        if (inheritedPageProperties == null) {
            inheritedPageProperties = get(WCMBindings.INHERITED_PAGE_PROPERTIES, ValueMap.class);
        }
        return inheritedPageProperties;
    }

    /**
     * @return Sling {@link SlingBindings#RESOURCE} binding if available, otherwise null is returned
     * */
    public final Resource getResource() {
        if (resource == null) {
            resource = get(SlingBindings.RESOURCE, Resource.class);
        }
        return resource;
    }

    /**
     * @return ResourceResolver associated with the current request
     * */
    public final ResourceResolver getResourceResolver() {
        if (resourceResolver == null) {
            resourceResolver = getRequest().getResourceResolver();
        }
        return resourceResolver;
    }

    /**
     * @return Sling {@link SlingBindings#REQUEST} binding if available, otherwise null is returned
     * */
    public final SlingHttpServletRequest getRequest() {
        if (slingHttpServletRequest == null) {
            slingHttpServletRequest = get(SlingBindings.REQUEST, SlingHttpServletRequest.class);
        }
        return slingHttpServletRequest;
    }

    /**
     * @return Sling {@link SlingBindings#RESPONSE} binding if available, otherwise null is returned
     * */
    public final SlingHttpServletResponse getResponse() {
        if (slingHttpServletResponse == null) {
            slingHttpServletResponse = get(SlingBindings.RESPONSE, SlingHttpServletResponse.class);
        }
        return slingHttpServletResponse;
    }

    /**
     * @return sling script helper binding {@link SlingBindings#SLING} if available, otherwise null is returned
     * */
    public final SlingScriptHelper getSlingScriptHelper() {
        if (slingScriptHelper == null) {
            slingScriptHelper = get(SlingBindings.SLING, SlingScriptHelper.class);
        }
        return slingScriptHelper;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_COMPONENT_CONTEXT} binding if available, otherwise null is returned
     */
    public final ComponentContext getComponentContext() {
        if (componentContext == null) {
            componentContext = get(WCMBindingsConstants.NAME_COMPONENT_CONTEXT, ComponentContext.class);
        }
        return componentContext;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_EDIT_CONTEXT} binding if available, otherwise null is returned
     */
    public final EditContext getEditContext() {
        if (editContext == null) {
            editContext = get(WCMBindingsConstants.NAME_EDIT_CONTEXT, EditContext.class);
        }
        return editContext;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_RESOURCE_DESIGN} binding if available, otherwise null is returned
     */
    public final Design getResourceDesign() {
        if (resourceDesign == null) {
            resourceDesign = get(WCMBindingsConstants.NAME_RESOURCE_DESIGN, Design.class);
        }
        return resourceDesign;
    }

    /**
     * @return {@link WCMBindingsConstants#NAME_XSSAPI} binding if available, otherwise null is returned
     */
    public final XSSAPI getXSSAPI() {
        if (xssapi == null) {
            xssapi = get(WCMBindingsConstants.NAME_XSSAPI, XSSAPI.class);
        }
        return xssapi;
    }
}
