/*
*
*	File: Gpos.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;
import java.util.TreeMap;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;
import com.adobe.fontengine.inlineformatting.AttributedRun;
import com.adobe.fontengine.inlineformatting.ElementAttribute;

/** Gives access to the 'GPOS' table. 
 * 
 * <h4>Applying features.</h4>
 * 
 * <p>To apply features, the client first resolves a feature tag to a
 * set of lookups, using the <code>resolveFeatureTag</code> method. Then,
 * it invokes the GSUB engine to apply those lookups to the glyph run,
 * using the <code>applyLookups</code> method. The client passes two objects
 * to <code>applyLookups</code>, implementing the <code>Shaper</code> 
 * and <code>Selector</code> interfaces respectively. The <code>Shaper</code>
 * object is used by the GPOS engine to access the glyphs as well as to 
 * report transformations to be done on the glyph run. The 
 * <code>Selector</code> object is used by the GPOS engine to learn if the 
 * feature should be applied at some positions. Thus, the <code>Shaper</code>
 * and <code>Selector</code> objects are the bulk of the interface between
 * the client and the GPOS engine.
 */
final public class Gpos extends LookupTable {
    
  protected Gpos (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);
  }



  /** Return the offset of the script list. */
  protected int getScriptListOffset () throws InvalidFontException {
    return this.data.getOffset (0, 4); 
  }

  /** Return the offset of the feature list. */
  protected int getFeatureListOffset () throws InvalidFontException {
    return this.data.getOffset (0, 6);
  }

  /** Return the offset of the lookup list. */
  protected int getLookupListOffset () throws InvalidFontException {
    return this.data.getOffset (0, 8);
  }


  
  /** Apply a lookup subtable at a position in the glyph run.
   * @param lookupType the type of the lookup
   * @param lookupFlag the value of lookupFlag for this lookup
   * @param stOffset the offset of the subtable
   * @param run the AttributedRun to process
   * @param start the first accessible element in the run
   * @param limit the element after the last accessible element in the run
   * @param curGlyph the glyph occurrence to process
   * @param selector to determine whether the lookup applies at some positions
   * @param gdef the GDEF table
   * @return the result the of applying the subtable
   */
  protected LookupResult applyLookupSubtable 
         (int lookupType, int lookupFlag, int stOffset,
          AttributedRun run, int start, int limit, int curGlyph, 
          OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
       
    switch (lookupType) {
      case 1: { 
        return applySinglePos (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 2: { 
        return applyPairPos (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 3: { 
        return applyCursiveAttachment (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 4: { 
        return applyMarkToBase (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 5: { 
        return applyMarkToLigature (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 6: { 
        return applyMarkToMark (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 7: { 
        return applyContextualSubtable (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 8: { 
        return applyChainingContextualSubtable (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 9: { 
        return applyExtensionSubtable (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      default: {
        throw new InvalidFontException ("Invalid GPOS lookup type (" + lookupType + ")"); }}
  }

  /** Compute the size of a ValueRecord.
   * @param valueFormat the format of the record
   * @return the size of the record
   */
  protected int getValueRecordSize (int valueFormat) {
    int size = 0;
    if ((valueFormat & 0x001) != 0) {
      size += 2; }
    if ((valueFormat & 0x002) != 0) {
      size += 2; }
    if ((valueFormat & 0x004) != 0) {
      size += 2; }
    if ((valueFormat & 0x008) != 0) {
      size += 2; }
    if ((valueFormat & 0x010) != 0) {
      size += 2; }
    if ((valueFormat & 0x020) != 0) {
      size += 2; }
    if ((valueFormat & 0x040) != 0) {
      size += 2; }
    if ((valueFormat & 0x080) != 0) {
      size += 2; }
    
    return size;
  }
  
  protected void applyValueRecord (AttributedRun run, int pos, int offset, 
                                   int valueFormat)
  throws InvalidFontException {
    // note that we do not apply device table correction, as our layout
    // model is not aware of the ppem
    
    int xPlacement = 0;
    int yPlacement = 0;
    int xAdvance = 0;
    int yAdvance = 0;
    
    if ((valueFormat & 0x001) != 0) {
      xPlacement = this.data.getint16 (offset);
      offset += 2; }
    
    if ((valueFormat & 0x002) != 0) {
      yPlacement = this.data.getint16 (offset);
      offset += 2; }
    
    if ((valueFormat & 0x004) != 0) {
      xAdvance = this.data.getint16 (offset);
      offset += 2; }
    
    if ((valueFormat & 0x008) != 0) {
      yAdvance = this.data.getint16 (offset);
      offset += 2; }
    
    run.adjustPlacementAndAdvance (pos, xPlacement, yPlacement, xAdvance, yAdvance);
  }

  protected LookupResult applySinglePos 
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applySinglePosFormat1 (lookupFlag, stOffset,  run, start, limit, curGlyph,
             selector, gdef); }
      
      case 2: { 
        return applySinglePosFormat2 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GPOS single pos format (" + format + ")"); }}
  }

  protected LookupResult applySinglePosFormat1 
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
        
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    if (getCoverageIndex (run.elementAt (inPos), coverageOffset) == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, inPos)) {
      return lookupNotApplied; }
    
    int valueFormat = this.data.getuint16 (stOffset + 4);
    applyValueRecord (run, inPos, stOffset + 6, valueFormat);
        
    return new LookupResult (true, inPos + 1, 0);
  }

  protected LookupResult applySinglePosFormat2 
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, inPos)) {
      return lookupNotApplied; }
    
    int valueFormat = this.data.getuint16 (stOffset + 4);
    int valueRecordOffset = stOffset + 8 + getValueRecordSize (valueFormat) * ci;
    
    applyValueRecord (run, inPos, valueRecordOffset, valueFormat);
    
    return new LookupResult (true, inPos + 1, 0);
  }
  
  protected LookupResult applyPairPos 
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyPairPosFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      case 2: { 
        return applyPairPosFormat2 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GPOS pair pos format (" + format + ")"); }}
  }

  protected LookupResult applyPairPosFormat1
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci == -1) {
      return lookupNotApplied; }
    
    int firstGlyphPos = inPos++;
    
    while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
      inPos++; }
    
    if (limit <= inPos || ! selector.isApplied (run, firstGlyphPos, inPos)) {
      return lookupNotApplied; }
    
    int valueFormat1 = this.data.getuint16 (stOffset + 4);
    int valueFormat2 = this.data.getuint16 (stOffset + 6);
    int valueRecord1Size = getValueRecordSize (valueFormat1);
    int valueRecord2Size = getValueRecordSize (valueFormat2);
    int pairValueRecordSize = 2 + valueRecord1Size + valueRecord2Size;
    
    int pairSetOffset = this.data.getOffset (stOffset, 10 + 2*ci);
    int pairValueCount = this.data.getuint16 (pairSetOffset);
    for (int p = 0; p < pairValueCount; p++) {
      int pairValueRecordOffset = pairSetOffset + 2 + pairValueRecordSize * p;
      int secondGlyph = this.data.getuint16 (pairValueRecordOffset);
      
      if (secondGlyph == run.elementAt (inPos)) {
        
        if (valueFormat1 != 0) {
          applyValueRecord (run, firstGlyphPos, pairValueRecordOffset + 2, valueFormat1);}
        if (valueFormat2 != 0) {
          applyValueRecord (run, inPos, pairValueRecordOffset + 2 + valueRecord1Size, valueFormat2); }
        
        if (valueFormat2 == 0) {
          return new LookupResult (true, inPos, 0); }
        else {
          return new LookupResult (true, inPos + 1, 0); }}}
    
    return lookupNotApplied; 
  }

  protected LookupResult applyPairPosFormat2 
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    if (getCoverageIndex (run.elementAt (inPos), coverageOffset) == -1) {
      return lookupNotApplied; }
    
    int firstGlyphPos = inPos++;
    
    while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
      inPos++; }
    
    if (limit <= inPos || ! selector.isApplied (run, firstGlyphPos, inPos)) {
      return lookupNotApplied; }
    
    
    int firstGlyphClass = getClassIndex (run.elementAt (firstGlyphPos),
                                         this.data.getOffset (stOffset, 8));
    int secondGlyphClass = getClassIndex (run.elementAt (inPos),
                                          this.data.getOffset (stOffset, 10));
    int class2Count = this.data.getuint16 (stOffset + 14);
    int valueFormat1 = this.data.getuint16 (stOffset + 4);
    int valueFormat2 = this.data.getuint16 (stOffset + 6);
    int valueRecord1Size = getValueRecordSize (valueFormat1);
    int valueRecord2Size = getValueRecordSize (valueFormat2);
    int class2RecordSize = valueRecord1Size + valueRecord2Size;
    int class1RecordSize = class2RecordSize * class2Count;
    
    int class2RecordOffset = stOffset + 16
                               + firstGlyphClass * class1RecordSize
                               + secondGlyphClass * class2RecordSize;
    
    if (valueFormat1 != 0) {
      applyValueRecord (run, firstGlyphPos, class2RecordOffset, valueFormat1); }
    if (valueFormat2 != 0) {
      applyValueRecord (run, inPos, class2RecordOffset + valueRecord1Size, valueFormat2); }
    
    if (valueFormat2 == 0) {
      return new LookupResult (true, inPos, 0); }
    else {
      return new LookupResult (true, inPos + 1, 0); }
  }
  
  protected LookupResult applyCursiveAttachment 
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyCursiveAttachmentFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GPOS cursive attachment format (" + format + ")"); }}
  }

  protected LookupResult applyCursiveAttachmentFormat1 
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {   
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int firstPos = curGlyph;
    
    int firstCoverageIndex = getCoverageIndex (run.elementAt (firstPos), coverageOffset);
    if (firstCoverageIndex == -1) {
      return lookupNotApplied; }
    
    int secondPos = curGlyph + 1;
    
    while (secondPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (secondPos))) {
      secondPos++; }
    
    if (limit <= secondPos || ! selector.isApplied (run, firstPos, secondPos)) {
      return lookupNotApplied; }
    
    int secondCoverageIndex = getCoverageIndex (run.elementAt (secondPos), coverageOffset);
    if (secondCoverageIndex == -1) { 
      return lookupNotApplied; }
    
    int firstAnchorOffset = this.data.getOffset (stOffset, 6 + firstCoverageIndex * 4 + 2);
    int secondAnchorOffset = this.data.getOffset (stOffset, 6 + secondCoverageIndex * 4);
    
    if (firstAnchorOffset == 0 || secondAnchorOffset == 0) {
      return lookupNotApplied; }
    
    mergeAnchors (run, firstPos, firstAnchorOffset, secondPos, secondAnchorOffset);
    
    return new LookupResult (true, firstPos + 1, 0);
  }
  
  protected LookupResult applyMarkToBase 
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyMarkToBaseFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("invalid GPOS mark to base format (" + format + ")"); }}
  }

  protected LookupResult applyMarkToBaseFormat1
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int markCoverageOffset = this.data.getOffset (stOffset, 2);
    int baseCoverageOffset = this.data.getOffset (stOffset, 4);
    int inPos = curGlyph;
    
    if (getCoverageIndex (run.elementAt (inPos), markCoverageOffset) == -1) {
      return lookupNotApplied; }
    
    int basePos = curGlyph - 1;
    
    while (start <= basePos 
           && (   lookupFlagCovers (lookupFlag, gdef, run.elementAt (basePos))
               || (   gdef != null
                   && gdef.getGlyphClass (run.elementAt (basePos)) == 3))) {
      basePos--; }
    
    if (basePos < start 
        || getCoverageIndex (run.elementAt (basePos), baseCoverageOffset) == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, basePos, inPos)) {
      return lookupNotApplied; }
    
    int classCount = this.data.getuint16 (stOffset + 6);
    int markCoverageIndex = getCoverageIndex (run.elementAt (inPos), markCoverageOffset);
    int markArrayOffset = this.data.getOffset (stOffset, 8);
    int markClass = this.data.getuint16 (markArrayOffset + 2 + 4 * markCoverageIndex);
    int markAnchorOffset = this.data.getOffset (markArrayOffset, 2 + 4 * markCoverageIndex + 2);
    
    int baseCoverageIndex = getCoverageIndex (run.elementAt (basePos), baseCoverageOffset);
    int baseArrayOffset = this.data.getOffset (stOffset, 10);
    int baseAnchorOffset = this.data.getOffset (baseArrayOffset, 2 + 2 * (classCount * baseCoverageIndex + markClass));
    
    mergeAnchors (run, basePos, baseAnchorOffset, inPos, markAnchorOffset);
    
    return new LookupResult (true, inPos + 1, 0);
  }

  protected LookupResult applyMarkToLigature
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyMarkToLigatureFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      default: {
        throw new InvalidFontException ("invalid GPOS mark to ligature format (" + format + ")"); }}
  }

  protected LookupResult applyMarkToLigatureFormat1 
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {

    int markCoverageOffset = this.data.getOffset (stOffset, 2);
    int ligCoverageOffset = this.data.getOffset (stOffset, 4);
    int inPos = curGlyph;
    
    if (getCoverageIndex (run.elementAt (inPos), markCoverageOffset) == -1) {
      return lookupNotApplied; }
    
    int ligPos = curGlyph - 1;
    
    while (start <= ligPos
           && (   lookupFlagCovers (lookupFlag, gdef, run.elementAt (ligPos))
               || (   gdef != null
                   && gdef.getGlyphClass (run.elementAt (ligPos)) == 3))) {
      ligPos--; }
    
    if (ligPos < start 
        || getCoverageIndex (run.elementAt (ligPos), ligCoverageOffset) == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, ligPos, inPos)) {
      return lookupNotApplied; }
    
    
    int classCount = this.data.getuint16 (stOffset + 6);
    int markCoverageIndex = getCoverageIndex (run.elementAt (inPos), markCoverageOffset);
    int markArrayOffset = this.data.getOffset (stOffset, 8);
    int markClass = this.data.getuint16 (markArrayOffset + 2 + 4 * markCoverageIndex);
    int markAnchorOffset = this.data.getOffset (markArrayOffset, 2 + 4 * markCoverageIndex + 2);
    
    int ligCoverageIndex = getCoverageIndex (run.elementAt (ligPos), ligCoverageOffset);
    int ligArrayOffset = this.data.getOffset (stOffset, 10);
    int ligAttachOffset = this.data.getOffset (ligArrayOffset, 2+2*ligCoverageIndex);
    
    Integer ligatureComponent = (Integer) run.getElementStyle (inPos, Gsub.ligatureComponentAttribute);
    int lc;
    if (ligatureComponent == null) {
      lc = 0; }
    else {
      lc = ligatureComponent.intValue (); }
    
    int ligAnchorOffset = this.data.getOffset (ligAttachOffset, 2 + 2 * (classCount * lc + markClass));
    
    mergeAnchors (run, ligPos, ligAnchorOffset, inPos, markAnchorOffset);
    
    return new LookupResult (true, inPos + 1, 0);
  }

  protected LookupResult applyMarkToMark
         (int lookupFlag, int stOffset,
          AttributedRun run, int start, int limit, int curGlyph,
          OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    switch (format) {
      case 1: { 
        return applyMarkToMarkFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      default: {
        throw new InvalidFontException ("invalid GPOS cursive attachment format (" + format + ")"); }}
  }

  protected LookupResult applyMarkToMarkFormat1
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {

    int markCoverageOffset = this.data.getOffset (stOffset, 2);
    int mark2CoverageOffset = this.data.getOffset (stOffset, 4);
    int inPos = curGlyph;
    
    if (getCoverageIndex (run.elementAt (inPos), markCoverageOffset) == -1) {
      return lookupNotApplied; }
    
    int mark2Pos = curGlyph - 1;
    
    while (start <= mark2Pos 
            && lookupFlagCovers (lookupFlag, gdef, run.elementAt (mark2Pos))) {
      mark2Pos--; }
    
    if (mark2Pos < start
        || getCoverageIndex (run.elementAt (mark2Pos), mark2CoverageOffset) == -1) {
      return lookupNotApplied; }
    
    if (! selector.isApplied (run, mark2Pos, inPos)) {
      return lookupNotApplied; }
    
    // The two marks must be on the same ligature component
    Integer inPosLigatureComponent = (Integer) run.getElementStyle (inPos, Gsub.ligatureComponentAttribute); 
    Integer mark2LigatureComponent = (Integer) run.getElementStyle (mark2Pos, Gsub.ligatureComponentAttribute);   
    if (   (inPosLigatureComponent == null ? 0 : inPosLigatureComponent.intValue ()) 
        != (mark2LigatureComponent == null ? 0 : mark2LigatureComponent.intValue ())) {
      return lookupNotApplied; }

    int classCount = this.data.getuint16 (stOffset + 6);
    int markCoverageIndex = getCoverageIndex (run.elementAt (inPos), markCoverageOffset);
    int markArrayOffset = this.data.getOffset (stOffset, 8);
    int markClass = this.data.getuint16 (markArrayOffset + 2 + 4 * markCoverageIndex);
    int markAnchorOffset = this.data.getOffset (markArrayOffset, 2 + 4 * markCoverageIndex + 2);
    
    int mark2CoverageIndex = getCoverageIndex (run.elementAt (mark2Pos), mark2CoverageOffset);
    int mark2ArrayOffset = this.data.getOffset (stOffset, 10);
    int mark2AnchorOffset =  this.data.getOffset (mark2ArrayOffset, 2 + 2 * (classCount * mark2CoverageIndex + markClass));
    
    mergeAnchors (run, mark2Pos, mark2AnchorOffset, inPos, markAnchorOffset);
    
    return new LookupResult (true, inPos + 1, 0); 
  }

  
  
  protected void mergeAnchors (AttributedRun run,
                               int pos1, int offset1, 
                               int pos2, int offset2)
  throws InvalidFontException {

    int a1x = this.data.getint16 (offset1 + 2);
    int a1y = this.data.getint16 (offset1 + 4);
    int a2x = this.data.getint16 (offset2 + 2);
    int a2y = this.data.getint16 (offset2 + 4);
    
    double deltaX = a1x + run.getElementXPlacement (pos1) - a2x - run.getElementXPlacement (pos2);
    double deltaY = a1y + run.getElementYPlacement (pos1) - a2y - run.getElementYPlacement (pos2);
    
    if (((Integer) run.getElementStyle (pos1, ElementAttribute.bidiLevel)).intValue () % 2 == 0) {
      for (int i = pos1; i < pos2; i++) {
        deltaX -= run.getElementXAdvance (i);
        deltaY -= run.getElementYAdvance (i); }}
    
    else {
      for (int i = pos1 + 1; i <= pos2; i++) {
        deltaX += run.getElementXAdvance (i);
        deltaY += run.getElementYAdvance (i); }}
      
    run.adjustPlacementAndAdvance (pos2, deltaX, deltaY, 0, 0); 
  }

  //-------------------------------------------------------- 'size' feature ---
  
  /** Return the arguments of the 'size' feature.
   * 
   * A number of fonts have been incorrectly constructed; it is not possible
   * to recognize those fonts just from their GPOS table, so this method 
   * takes an argument that controls whether to compensate for the bug.
   */
  public OpticalSizeData getOpticalSizeData (boolean compensateForFontBug)
  throws InvalidFontException {
    
    int featureListOffset = getFeatureListOffset ();
    int featureCount = this.data.getuint16 (featureListOffset);
    
    for (int i = 0; i < featureCount; i++) {
      if (this.data.getint32 (featureListOffset + 2 + 6*i) == Tag.feature_size) {
        int featureOffset = this.data.getOffset (featureListOffset, 2 + 6*i + 4);
        int featureParamsOffset = this.data.getOffset (0, featureOffset);
        
        if (compensateForFontBug) {
          featureParamsOffset += featureListOffset; }
        else {
          featureParamsOffset += featureOffset; }
        
        return new OpticalSizeData (
            this.data.getuint16 (featureParamsOffset),
            this.data.getuint16 (featureParamsOffset + 2),
            this.data.getuint16 (featureParamsOffset + 4),
            this.data.getuint16 (featureParamsOffset + 6),
            this.data.getuint16 (featureParamsOffset + 8)); }}

    return null;
  }
  
  void subsetAndStream(Subset subset, TreeMap lookups, Map tables, int numGlyphs, boolean preserveNonEmptyKernFeatures) 
  throws InvalidFontException, UnsupportedFontException 
  {
	  LookupTableSubsetter subsetter = new GposSubsetter(this, numGlyphs, preserveNonEmptyKernFeatures);
	  OTByteArrayBuilder newData = subsetter.subsetAndStream(lookups, subset);
	  tables.put(new Integer(Tag.table_GPOS), newData);
  }
  
  public void stream(Map tables) {
	  OTByteArrayBuilder newData = this.getDataAsByteArray();
	  tables.put (new Integer (Tag.table_GPOS), newData);
  }
}
