/*
*
*       File: Os2.java
*
*
*       ADOBE CONFIDENTIAL
*       ___________________
*
*       Copyright 2004-2005 Adobe Systems Incorporated
*       All Rights Reserved.
*
*       NOTICE: All information contained herein is, and remains the property of
*       Adobe Systems Incorporated and its suppliers, if any. The intellectual
*       and technical concepts contained herein are proprietary to Adobe Systems
*       Incorporated and its suppliers and may be covered by U.S. and Foreign
*       Patents, patents in process, and are protected by trade secret or
*       copyright law. Dissemination of this information or reproduction of this
*       material is strictly forbidden unless prior written permission is obtained
*       from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.util.Map;

import com.adobe.fontengine.font.EmbeddingPermission;
import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.LineMetrics;
import com.adobe.fontengine.font.Permission;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/** Gives access to the 'OS/2' table.
 * 
 * <h4>Version handling</h4>
 * 
 * <p>'OS/2' tables have only a minor version number.
 * This implementation:
 * <ul> 
 * <li>fully supports version 0, 1, 2 and 3 tables,</li>
 * <li>interprets version x > 3  tables as version 3 tables,</li>
 * </ul> 
 * 
 * <p>When some data is not present in the actual version of the table,
 * the accessors return identified values.</p>
 * 
 * <h4>Synchronization</h4>
 * 
 * <p>Like all tables, these objects are immutable.</p> 
 */

final public class Os2 extends Table {

  /*
   * Bit selectors for the fsSelection flags returned by getSelection()
   */
  public interface SelectionBits {
    public static final int italic      = 0x0001;
    public static final int underscore  = 0x0002;
    public static final int negative    = 0x0004;
    public static final int outlined    = 0x0008;
    public static final int strikeout   = 0x0010;
    public static final int bold        = 0x0020;
    public static final int regular     = 0x0040;
  }
  

  protected Os2 (FontByteArray buffer) 
      throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer);
  }
  
  /** Get the version of this table. */
  public int getTableVersion () throws InvalidFontException {
    return this.data.getuint16 (0);
  }
  
  /** Get the usWeight of this table. */
  public int getWeightClass () throws InvalidFontException {
    return this.data.getuint16 (4);
  }
  
  /** Get the usWidth of this table. */
  public int getWidthClass () throws InvalidFontException {
    return this.data.getuint16 (6);
  }
  
  /**
   * @return 0 if panose indicates the font is fixed pitch; 1 if it indicates proportional; -1 if don't know
   * @throws InvalidFontException
   */
  public int panoseIndicatesProportional () throws InvalidFontException {
    int family = this.data.getint8 (32);
    int data = this.data.getint8 (35);

    if (family == 0 || data == 0) {
      return -1; }

    switch (family) {
      case 2:
      case 4: {
        return data == 9 ? 0 : 1; }
      case 3:
      case 5: {
        return data == 3 ? 0 : 1; }}
    return -1;
  }
  
  /** Get the fsSelection of this table. */
  public int getSelection () throws InvalidFontException {
    return this.data.getuint16 (62);
  }
  
  public int getFirstChar() throws InvalidFontException {
	  return this.data.getuint16(64);
  }
  
  public int getLastChar() throws InvalidFontException {
	  return this.data.getuint16(66);
  }
  
  /** Get the sTypoAscender of this table. */
  public int getTypoAscender () throws InvalidFontException {
    return this.data.getint16 (68);
  }
  
  /** Get the sTypoDescender of this table. */
  public int getTypoDescender () throws InvalidFontException {
    return this.data.getint16 (70); 
  }
  
  /** Get the sTypoLineGap of this table. */
  public int getTypoLineGap () throws InvalidFontException {
    return this.data.getint16 (72); 
  }
  
  /** Get the usWinAscent of this table. */
  public int getWinAscent() throws InvalidFontException {
	  return this.data.getuint16(74);
  }
  
  /** Get the usWinDescent of this table. */
  public int getWinDescent() throws InvalidFontException {
	  return this.data.getuint16(76);
  }
  
  /** Return the line metrics for this font. 
   * 
   * The metrics are expressed in the design space of the font, 
   * i.e. they need to be converted through the metrics matrix.
   * 
   * Some font formats do not support the notion of line metrics,
   * and in those cases, this method returns null.
   * 
   * @throws UnsupportedFontException
   * @throws InvalidFontException
   */
  public LineMetrics getLineMetrics ()
  throws UnsupportedFontException, InvalidFontException {
    return new LineMetrics (getTypoAscender (), getTypoDescender (), getTypoLineGap ());
  }

  /** Get the vendor. */
  public long getVendor () throws InvalidFontException {
    return this.data.getuint32 (58);
  }
  
  /** Get the sxHeight. 
   * @return the value of the sCapHeight field if present, Integer.MAX_VALUE
   * otherwise.
   */
  public int getxHeight () throws InvalidFontException {
    if (getTableVersion() < 2) {
      return Integer.MAX_VALUE; }     
    return this.data.getint16 (86);
  }
  
  /** Get the sCapHeight. 
   * @return the value of the sCapHeight field if present, Integer.MAX_VALUE
   * otherwise.
   */
  public int getCapHeight () throws InvalidFontException {
    if (getTableVersion() < 2) {
      return Integer.MAX_VALUE; }     
    return this.data.getint16 (88);
  }
  
  //--------------------------------------------------------- Unicode ranges ---

  /** Identifies a Unicode range.
   */
  final public static class UnicodeRange {
    public final int offset;
    public final int mask;

    public final int firstVersion;
    public final int lastVersion;

    private UnicodeRange (int bit, int firstVersion, int lastVersion) {
      this.offset = 4 * (bit / 32);
      this.mask = 1 << (bit % 32);

      this.firstVersion = firstVersion;
      this.lastVersion = lastVersion;
    }

    private UnicodeRange (int bit, int firstVersion) {
      this (bit, firstVersion, Integer.MAX_VALUE);
    }

    public static final UnicodeRange BASIC_LATIN                             = new UnicodeRange ( 0, 1); 
    public static final UnicodeRange LATIN_1_SUPPLEMENT                      = new UnicodeRange ( 1, 1); 
    public static final UnicodeRange LATIN_EXTENDED_A                        = new UnicodeRange ( 2, 1); 
    public static final UnicodeRange LATIN_EXTENDED_B                        = new UnicodeRange ( 3, 1); 
    public static final UnicodeRange IPA_EXTENSIONS                          = new UnicodeRange ( 4, 1); 
    public static final UnicodeRange SPACING_MODIFIER_LETTERS                = new UnicodeRange ( 5, 1); 
    public static final UnicodeRange COMBINING_DIACRITICAL_MARKS             = new UnicodeRange ( 6, 1); 
    public static final UnicodeRange GREEK                                   = new UnicodeRange ( 7, 1); 
    public static final UnicodeRange GREEK_SYMBOLS_AND_COPTIC                = new UnicodeRange ( 8, 1, 1); 
    public static final UnicodeRange CYRILLIC                                = new UnicodeRange ( 9, 1); 
    public static final UnicodeRange ARMENIAN                                = new UnicodeRange (10, 1); 
    public static final UnicodeRange HEBREW                                  = new UnicodeRange (10, 1); 
    public static final UnicodeRange HEBREW_EXTENDED                         = new UnicodeRange (12, 1, 1); 
    public static final UnicodeRange ARABIC                                  = new UnicodeRange (13, 1); 
    public static final UnicodeRange ARABIC_EXTENDED                         = new UnicodeRange (14, 1, 1); 
    public static final UnicodeRange DEVANAGARI                              = new UnicodeRange (15, 1); 
    public static final UnicodeRange BENGALI                                 = new UnicodeRange (16, 1); 
    public static final UnicodeRange GURMUKHI                                = new UnicodeRange (17, 1); 
    public static final UnicodeRange GUJARATI                                = new UnicodeRange (18, 1); 
    public static final UnicodeRange ORIYA                                   = new UnicodeRange (19, 1); 
    public static final UnicodeRange TAMIL                                   = new UnicodeRange (20, 1); 
    public static final UnicodeRange TELUGU                                  = new UnicodeRange (21, 1); 
    public static final UnicodeRange KANNADA                                 = new UnicodeRange (22, 1); 
    public static final UnicodeRange MALAYALAM                               = new UnicodeRange (23, 1); 
    public static final UnicodeRange THAI                                    = new UnicodeRange (24, 1); 
    public static final UnicodeRange LAO                                     = new UnicodeRange (25, 1); 
    public static final UnicodeRange GEORGIAN                                = new UnicodeRange (26, 1); 
    public static final UnicodeRange GEORGIAN_EXTENDED                       = new UnicodeRange (27, 1, 1); 
    public static final UnicodeRange HANGUL_JAMO                             = new UnicodeRange (28, 1); 
    public static final UnicodeRange LATIN_EXTENDED_ADDITIONAL               = new UnicodeRange (29, 1); 
    public static final UnicodeRange GREEK_EXTENDED                          = new UnicodeRange (30, 1); 
    public static final UnicodeRange GENERAL_PUNCTUATION                     = new UnicodeRange (31, 1); 
    public static final UnicodeRange SUPERSCRIPTS_AND_SUSCRIPTS              = new UnicodeRange (32, 1); 
    public static final UnicodeRange CURRENCY_SYMBOLS                        = new UnicodeRange (33, 1); 
    public static final UnicodeRange COMBINING_DIACRITICAL_MARKS_FOR_SYMBOLS = new UnicodeRange (34, 1); 
    public static final UnicodeRange LETTERLIKE_SYMBOLS                      = new UnicodeRange (35, 1); 
    public static final UnicodeRange NUMBER_FORMS                            = new UnicodeRange (36, 1); 
    public static final UnicodeRange ARROWS                                  = new UnicodeRange (37, 1); 
    public static final UnicodeRange MATHEMATICAL_OPERATORS                  = new UnicodeRange (38, 1); 
    public static final UnicodeRange MISCELLANEOUS_TECHNICAL                 = new UnicodeRange (39, 1); 
    public static final UnicodeRange CONTROL_PICTURES                        = new UnicodeRange (40, 1); 
    public static final UnicodeRange OPTICAL_CHARACTER_RECOGNITION           = new UnicodeRange (41, 1); 
    public static final UnicodeRange ENCLOSED_ALPHANUMERICS                  = new UnicodeRange (42, 1); 
    public static final UnicodeRange BOX_DRAWING                             = new UnicodeRange (43, 1); 
    public static final UnicodeRange BLOCK_ELEMENTS                          = new UnicodeRange (44, 1); 
    public static final UnicodeRange GEOMETRIC_SHAPES                        = new UnicodeRange (45, 1); 
    public static final UnicodeRange MISCELLANEOUS_SYMBOLS                   = new UnicodeRange (46, 1); 
    public static final UnicodeRange DINGBATS                                = new UnicodeRange (47, 1); 
    public static final UnicodeRange CJK_SYMBOLS_AND_PUNCTUATION             = new UnicodeRange (48, 1); 
    public static final UnicodeRange HIRAGANA                                = new UnicodeRange (49, 1); 
    public static final UnicodeRange KATAKANA                                = new UnicodeRange (50, 1); 
    public static final UnicodeRange BOPOMOFO                                = new UnicodeRange (51, 1); 
    public static final UnicodeRange HANGUL_COMPATIBILITY_JAMO               = new UnicodeRange (52, 1); 
    public static final UnicodeRange CJK_MISCELLANEOUS                       = new UnicodeRange (53, 1, 2); 
    public static final UnicodeRange ENCLOSED_CJK_LETTERS_AND_MONTHS         = new UnicodeRange (54, 1); 
    public static final UnicodeRange CJK_COMPATIBILITY                       = new UnicodeRange (55, 1); 
    public static final UnicodeRange HANGUL_SYLLABLES                        = new UnicodeRange (56, 1); 
    public static final UnicodeRange SURROGATES                              = new UnicodeRange (57, 2); 
    public static final UnicodeRange CJK_UNIFIED_IDEOGRAPHS                  = new UnicodeRange (59, 1); 
    public static final UnicodeRange PRIVATE_USE_AREA                        = new UnicodeRange (60, 1); 
    public static final UnicodeRange CJK_COMPATIBILITY_IDEOGRAPHS            = new UnicodeRange (61, 1); 
    public static final UnicodeRange ALPHABETIC_PRESENTATION_FORMS           = new UnicodeRange (62, 1); 
    public static final UnicodeRange ARABIC_PRESENTATION_FORMS_A             = new UnicodeRange (63, 1); 
    public static final UnicodeRange COMBINING_HALF_MARKS                    = new UnicodeRange (64, 1); 
    public static final UnicodeRange CJK_COMPATIBILITY_FORMS                 = new UnicodeRange (65, 1); 
    public static final UnicodeRange SMALL_FORM_VARIANTS                     = new UnicodeRange (66, 1); 
    public static final UnicodeRange ARABIC_PRESENTATION_FORMS_B             = new UnicodeRange (67, 1); 
    public static final UnicodeRange HALFWIDTH_AND_FULLWIDTH_FORMS           = new UnicodeRange (68, 1); 
    public static final UnicodeRange SPECIALS                                = new UnicodeRange (69, 1); 
    public static final UnicodeRange TIBETAN                                 = new UnicodeRange (70, 2); 
    public static final UnicodeRange SYRIAC                                  = new UnicodeRange (71, 2); 
    public static final UnicodeRange THAANA                                  = new UnicodeRange (72, 2); 
    public static final UnicodeRange SINHALA                                 = new UnicodeRange (73, 2); 
    public static final UnicodeRange MYANMAR                                 = new UnicodeRange (74, 2); 
    public static final UnicodeRange ETHIOPIC                                = new UnicodeRange (75, 2); 
    public static final UnicodeRange CHEROKEE                                = new UnicodeRange (76, 2); 
    public static final UnicodeRange UNIFIED_CANADIAN_SYLLABICS              = new UnicodeRange (77, 2); 
    public static final UnicodeRange OGHAM                                   = new UnicodeRange (78, 2); 
    public static final UnicodeRange RUNIC                                   = new UnicodeRange (79, 2); 
    public static final UnicodeRange KHMER                                   = new UnicodeRange (80, 2); 
    public static final UnicodeRange MONGOLIAN                               = new UnicodeRange (81, 2); 
    public static final UnicodeRange BRAILLE                                 = new UnicodeRange (82, 2); 
    public static final UnicodeRange YI                                      = new UnicodeRange (83, 2); 
    public static final UnicodeRange TAGALOG_HANUNOO_BUHID_TAGBANWA          = new UnicodeRange (84, 3); 
    public static final UnicodeRange OLD_ITALIC                              = new UnicodeRange (85, 3); 
    public static final UnicodeRange GOTHIC                                  = new UnicodeRange (86, 3); 
    public static final UnicodeRange DESERET                                 = new UnicodeRange (87, 3); 
    public static final UnicodeRange MUSICAL_SYMBOLS                         = new UnicodeRange (88, 3); 
    public static final UnicodeRange MATHEMATICAL_ALPHANUMERIC_SYMBOLS       = new UnicodeRange (89, 3); 
    public static final UnicodeRange PRIVATE_USE_SUPPLEMENTARY               = new UnicodeRange (90, 3); 
    public static final UnicodeRange VARIATION_SELECTORS                     = new UnicodeRange (91, 3); 
    public static final UnicodeRange TAGS                                    = new UnicodeRange (92, 3); 
  }

  /** Determine if a Unicode range is supported.
   * For some ranges, some versions of the OS/2 table do not expressed 
   * whether it is supported. 
   * @return true iff the Unicode range is expressed and is supported.
   */
  public boolean supportsUnicodeRange (UnicodeRange cp)
  throws InvalidFontException {
    if (getTableVersion () < cp.firstVersion) {
      return false; }

    if (getTableVersion () > cp.lastVersion) {
      return false; }

    long l = this.data.getuint32 (42 + cp.offset);
    return ((l & cp.mask) != 0);
  }

  //------------------------------------------------------------- code pages ---

  /** Identifies a code page.
   */
  final public static class CodePage {
    public final int offset;
    public final int mask;

    public final int firstVersion;
    
    private CodePage (int bit, int firstVersion) {
      this.offset = 4 * (bit / 32);
      this.mask =  1 << (bit % 32);

      this.firstVersion = firstVersion;
    }
    
    public static final CodePage CP_1252_LATIN_1            = new CodePage ( 0, 1);
    public static final CodePage CP_1250_LATIN_2_EE         = new CodePage ( 1, 1);
    public static final CodePage CP_1251_CYRILLIC           = new CodePage ( 2, 1);
    public static final CodePage CP_1253_GREEK              = new CodePage ( 3, 1);
    public static final CodePage CP_1254_TURKISH            = new CodePage ( 4, 1);
    public static final CodePage CP_1255_HEBREW             = new CodePage ( 5, 1);
    public static final CodePage CP_1256_ARABIC             = new CodePage ( 6, 1);
    public static final CodePage CP_1257_BALTIC             = new CodePage ( 7, 1);
    public static final CodePage CP_1258_VIETNAMESE         = new CodePage ( 8, 2);
    public static final CodePage CP_874_THAI                = new CodePage (16, 1);
    public static final CodePage CP_932_JAPANESE            = new CodePage (17, 1);
    public static final CodePage CP_936_CHINESE_SIMPLIFIED  = new CodePage (18, 1);
    public static final CodePage CP_949_KOREAN_WANSUNG      = new CodePage (19, 1);
    public static final CodePage CP_950_CHINESE_TRADITIONAL = new CodePage (20, 1);
    public static final CodePage CP_1361_KOREAN_JOHAB       = new CodePage (21, 1);
    public static final CodePage CP_MAC_ROMAN               = new CodePage (29, 2);
    public static final CodePage CP_OEM                     = new CodePage (30, 1);
    public static final CodePage CP_SYMBOL                  = new CodePage (31, 1);
    public static final CodePage CP_869_IBM_GREEK           = new CodePage (48, 1);
    public static final CodePage CP_866_MSDOS_RUSSIAN       = new CodePage (49, 1);
    public static final CodePage CP_865_MSDOS_NORDIC        = new CodePage (50, 1);
    public static final CodePage CP_864_ARABIC              = new CodePage (51, 1);
    public static final CodePage CP_863_MSDOS_CANADIAN      = new CodePage (52, 1);
    public static final CodePage CP_862_HEBREW              = new CodePage (53, 1);
    public static final CodePage CP_861_MSDOS_ICELANDIC     = new CodePage (54, 1);
    public static final CodePage CP_860_MSDOS_PORTUGUESE    = new CodePage (55, 1);
    public static final CodePage CP_857_IBM_TURKISH         = new CodePage (56, 1);
    public static final CodePage CP_855_IBM_CYRILLIC        = new CodePage (57, 1);
    public static final CodePage CP_852_LATIN_2             = new CodePage (58, 1);
    public static final CodePage CP_775_MSDOS_BALTIC        = new CodePage (59, 1);
    public static final CodePage CP_737_GREEK               = new CodePage (60, 1);
    public static final CodePage CP_708_ARABIC              = new CodePage (61, 1);
    public static final CodePage CP_850_WE_LATIN_1          = new CodePage (62, 1);
    public static final CodePage CP_437_US                  = new CodePage (63, 1);
  }

  /** Determine if a code page is supported.
   * For some code pages, some versions of the OS/2 table do not expressed 
   * whether it is supported. 
   * @return true iff the code page is expressed and is supported.
   */
  public boolean supportsCodePage (CodePage cp) throws InvalidFontException {
    if (getTableVersion () < cp.firstVersion) {
      return false; }

    long l = this.data.getuint32 (78 + cp.offset);
    return ((l & cp.mask) != 0);
  }

  //------------------------------------------------------------ permissions ---

  int getRawFSType () throws InvalidFontException {
      return this.data.getuint16 (8);
  }

  
  /** Get the embedding permission for the font, using the
   * constants in {@link com.adobe.fontengine.font.EmbeddingPermission}.
   * @return one of RESTRICTED, PREVIEW_AND_PRINT, 
   * EDITABLE, INSTALLABLE or ILLEGAL_VALUE
   */
  public Permission getEmbeddingPermission ()
      throws InvalidFontException {
    int fsType = getRawFSType();
    return EmbeddingPermission.interpretFSType(fsType);
  }
  
  /** Get the subsetting restriction for embedding, using the
   * constants in {@link com.adobe.fontengine.font.EmbeddingPermission}.
   * @return one of FULL_FONT_ONLY, SUBSETTING_OK or
   * UNKNOWN_SUBSETTING_RESTRICTION
   */
  public EmbeddingPermission.SubsettingRestriction getEmbeddingSubsettingRestriction ()
      throws InvalidFontException {
    if (getTableVersion () < 2) {
      return EmbeddingPermission.SubsettingRestriction.UNKNOWN_SUBSETTING_RESTRICTION; }
    else {
      int fsType = getRawFSType();
      if ((fsType & (1 << 8)) != 0) {
        return EmbeddingPermission.SubsettingRestriction.FULL_FONT_ONLY; }
      else {
        return EmbeddingPermission.SubsettingRestriction.SUBSETTING_OK; }}
  }
  
  /** Get the outline restriction for embedding, using the
   * constants in {@link com.adobe.fontengine.font.EmbeddingPermission}.
   * @return one of BITMAP_ONLY, OUTLINES_OK or
   * UNKNOWN_OUTLINE_RESTRICTION
   */
  public EmbeddingPermission.OutlineRestriction getEmbeddingOutlineRestriction ()
      throws InvalidFontException {
    if (getTableVersion () < 2) {
      return EmbeddingPermission.OutlineRestriction.UNKNOWN_OUTLINE_RESTRICTION; }
    else {
      int fsType = getRawFSType();
      if ((fsType & (1 << 9)) != 0) {
        return EmbeddingPermission.OutlineRestriction.BITMAP_ONLY; }
      else {
        return EmbeddingPermission.OutlineRestriction.OUTLINES_OK; }}
  }
  

  //------------------------------------------------------------- subsetting ---
  
  public void subsetAndStream (Subset subset, Map tables) {
    OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(this.data);
    tables.put (new Integer (Tag.table_os2), newData); 
  } 
  
	public void stream(Map tables) {
		  OTByteArrayBuilder newData = this.getDataAsByteArray();
		  tables.put (new Integer (Tag.table_os2), newData);
	}
}
