/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.comments;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Collections;
import java.util.Map;

import org.apache.commons.lang.StringUtils;
import org.apache.jackrabbit.JcrConstants;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.osgi.annotation.versioning.ConsumerType;

/**
 * The <code>AbstractComment</code> implements {@link Comment} and provides the API implementation, at the same time
 * offering abstract methods with which concrete implementations can control retrieval and storage of resources.
 */
@ConsumerType
public abstract class AbstractComment implements Comment {

    private final AbstractCommentCollection collection;
    private final Resource resource;
    private final AbstractCommentingProvider provider;
    private final ValueMap properties;

    protected AbstractComment(final AbstractCommentCollection collection,
                              final Resource resource,
                              final AbstractCommentingProvider provider) {
        this.collection = collection;
        this.resource = resource;
        this.provider = provider;
        properties = resource.adaptTo(ValueMap.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final Resource addAttachment(final String name, final InputStream inputStream, final String mimeType)
            throws CommentException {
        if (null == inputStream) {
            throw new IllegalArgumentException("input stream may not be null");
        }
        if (StringUtils.isBlank(name)) {
            throw new IllegalArgumentException("name may not be null or empty");
        }
        if (StringUtils.isBlank(mimeType)) {
            throw new IllegalArgumentException("mimetype may not be null or empty");
        }
        return provider.createAttachmentResource(getResource(), name, inputStream, mimeType);
    }

    @Override
    public final String getAnnotationData() {
        return properties.get(AbstractCommentingProvider.PN_ANNOTATIONDATA, String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final Resource getAttachment(final String name) {
        return provider.getAttachmentResource(getResource(), name);
    }

    @Override
    public final String getAuthorName() {
        final String author = properties.get(AbstractCommentingProvider.PN_AUTHOR, String.class);
        return null != author ? author : getCreatedBy();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final Map<String, Resource> getAttachmentMap() {
        return Collections.unmodifiableMap(provider.getAttachmentMap(getResource()));
    }

    /**
     * {@inheritDoc}
     */
    private final String getCreatedBy() {
        return properties.get("jcr:createdBy", String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final Calendar getCreated() {
        return properties.get(JcrConstants.JCR_CREATED, Calendar.class);
    }

    @Override
    public final Calendar getLastModified() {
        return properties.get(JcrConstants.JCR_LASTMODIFIED, Calendar.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String getMessage() {
        return properties.get(AbstractCommentingProvider.PN_MESSAGE, String.class);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final CommentCollection getCollection() {
        return collection;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final String getPath() {
        return getResource().getPath();
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final ValueMap getProperties() {
        return properties;
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final void removeAttachment(final String name) throws CommentException {
        provider.removeAttachmentResource(getResource(), name);
    }

    /**
     * {@inheritDoc}
     */
    @Override
    public final void remove() {
        collection.removeComment(this);
    }

    /**
     * Return the {@link Resource} representing this comment.
     *
     * @return The resource.
     */
    protected Resource getResource() {
        return resource;
    }
}
