/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.comments;

import java.io.InputStream;
import java.util.Calendar;
import java.util.Map;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.osgi.annotation.versioning.ConsumerType;

/**
 * <p>A <code>Comment</code> represents an annotation created by a user (author) on a specific target (e.g. a page,
 * asset or any node for that matter). The comments are contained within a {@link CommentCollection} (see {@link
 * #getCollection()}. The comment collection is specific to a target (see {@link CommentCollection#getTarget()}. </p>
 * <p>In its simplest form, a comment simply provides a comment text or "message" (see {@link #getMessage()} and access
 * to various meta data, such as creation date, who created it, etc. </p> <p>Comments support attachments, i.e. files
 * that are stored as part of a comment, e.g. a preview image if the comment is used to annotate an image.</p>
 */
@ConsumerType
public interface Comment {

    /**
     * Attach a file to this comment.
     *
     * @param name        The name of the file to attach.
     * @param inputStream The {@link InputStream} representing the contents of the file.
     * @param mimeType    The mime-type of this file.
     *
     * @return The {@link Resource} representing the attachment just added.
     *
     * @throws CommentException         Upon encountering an error during persistence of the attachment.
     * @throws IllegalArgumentException if name or inputStream or mimeType is null or empty.
     */
    Resource addAttachment(String name, InputStream inputStream, String mimeType);

    /**
     * Returns the string representing annotation data that is part of this comment and pertains to the target of the
     * associated comment collection.
     *
     * @return The string of the annotation data, or <code>null</code> if none is present.
     */
    String getAnnotationData();

    /**
     * Retrieve an attachment of this comment by its file name.
     *
     * @param name The file name of the attachment.
     *
     * @return A {@link Resource} representing the attachment, or <code>null</code> if no attachment with the given name
     *         exists.
     */
    Resource getAttachment(String name);

    /**
     * Returns the author's identifying name that represents the author of this comment. This may be a user ID, a
     * regular name, depending on the implementation.
     *
     * @return The string representing the author name.
     */
    String getAuthorName();

    /**
     * Retrieves a map, an entry of which represents a file-name/attachment pair. The map contains all the attachments
     * of the comment.
     *
     * @return A {@link Map} representing the available attachments, or an empty map if no attachments are present.
     */
    Map<String, Resource> getAttachmentMap();

    /**
     * The time stamp this comment was created.
     *
     * @return A <code>Calendar</code> representing creation time.
     */
    Calendar getCreated();

    /**
     * Returns the moment in time this collection was last modified.
     *
     * @return The calendar representing the last modification.
     */
    Calendar getLastModified();

    /**
     * Retrieves the message of this comment.
     *
     * @return A <code>String</code> representing the message.
     */
    String getMessage();

    /**
     * Returns the parent {@link CommentCollection} this comment is a part of.
     *
     * @return The parent {@link CommentCollection}.
     */
    CommentCollection getCollection();

    /**
     * Get the path of the resource representing this comment.
     *
     * @return A string representing the path.
     */
    String getPath();

    /**
     * Returns a {@link ValueMap} holding all properties of this comment.
     *
     * @return A {@link ValueMap} representing the properties.
     */
    ValueMap getProperties();

    /**
     * Removes an attachment from this comment, as identified by its name.
     *
     * @param name The name of the attachment to remove.
     *
     * @throws CommentException If an error occurs removing the attachment.
     */
    void removeAttachment(String name);

    /**
     * Removes (deletes) this comment from its collection. If the given comment has already been removed, nothing
     * happens. After removal, the given comment object must be considered invalid/stale and must not be used anymore.
     *
     * @throws CommentException Upon encountering an error while removing.
     */
    void remove();
}
