/* ****************************************************************************
 *
 *	File: ASHexString.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFInvalidParameterException;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;

/**
 * This class represents a non-cos hex string.
 * 
 */
public class ASHexString extends ASString
{
	/**
	 * Constructor.
	 */
	public ASHexString(byte[] bytes) 
	{
		super(bytes);
	}

	/**
	 * Construct an <code>ASHexString</code> from an array of integers.
	 * @param numbers numbers to use in constructing the ASHexString
	 * @param bytesToUse number of bytes to use from each number starting with the LSB. Must be from 1 to 4.
	 * @param bytesToPad number of bytes to pad in front of each number beyond the number of bytes taken from each number
	 * @throws PDFInvalidParameterException if the <i>bytesToUse</i> is not between 1 and 4
	 */
	public ASHexString(int[] numbers, int bytesToUse, int bytesToPad) 
	throws PDFInvalidParameterException
	{
		super(convertIntegersToBytes(numbers, 0, numbers.length-1, bytesToUse, bytesToPad));
	}
	
	public ASHexString(int[] numbers, int startIndex, int endIndex, int bytesToUse, int bytesToPad)
	throws PDFInvalidParameterException
	{
		super(convertIntegersToBytes(numbers, startIndex, endIndex, bytesToUse, bytesToPad));
	}
	
	
	
	/**
	 * Returns hex value of string as byte array.
	 */
	public byte[] getHex() 
	{
		return ByteOps.getHex(this.getBytes());
	}

	/**
	 * Writes the ASHexString in to the given OutputStream in the format expected by the PDF Spec.
	 * @see ASString
	 * @param outputStream OutputStream to write to.
	 * @throws PDFIOException
	 */
	public void write(OutputStream outputStream)
		throws PDFIOException
	{
		try {
			outputStream.write('<');
			outputStream.write( this.getHex() );
			outputStream.write('>');
		} catch (IOException e) {
			throw new PDFIOException(e);
		}
	}

	/**
	 * Writes the ASHexString in to the given OutputByteStream in the format expected by the PDF Spec.
	 * @see ASString
	 * @param outputStream OutputStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputStream)
	throws PDFIOException
	{
		try {
			outputStream.write('<');
			outputStream.write( this.getHex() );
			outputStream.write('>');
		} catch (IOException e) {
			throw new PDFIOException(e);
	}
}
	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.types.ASString#asString()
	 */
	@Override
	public String asString() 
	{
		byte[] hex = this.getBytes();
		try {
			// Determine if the CosString contains Unicode data as described
			// in section 3.8.1 of the PDF Reference Manual version 1.4.
			boolean isUnicode = hex.length >= 2 && hex[0] == -2 && hex[1] == -1;
			return new String(hex, isUnicode ? "UTF-16" : "ISO-8859-1");
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException("US-ASCII encoding not supported.",e);
		}
	}

	/* (non-Javadoc)
	 * @see com.adobe.internal.pdftoolkit.core.types.ASString#toString()
	 */
	@Override
	public String toString() 
	{
		return this.asString();
	}
}
