/* ****************************************************************************
 *
 *	File: ASName.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/

package com.adobe.internal.pdftoolkit.core.types;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.HashMap;
import java.util.Map;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.util.ByteOps;
import com.adobe.internal.pdftoolkit.core.util.UTF8Support;
import com.adobe.internal.pdftoolkit.core.util.UTFSupport;

/**
 * An ASName is the basic data type for a COS name. ASNames consist of
 * a sequence of bytes, not characters. In general, ASNames are either
 * statically defined or created by parsing new names from a PDF file.
 * The class provides the guarantee that for any two ASNames a and b, if
 * a.equals(b) is true, then a == b is true.
 */
public final class ASName extends ASObject implements Comparable
{
	/* The three following static objects are only referenced from getName() or
	 * methods called by getName() which is synchronized.  This prevents these
	 * class globals from being accessed by multiple threads concurrently.
	 */

	/** This Map contains the <code>NameEntry</code> objects that represent the 
	 * <code>ASName</code> objects.  The <code>NameEntry</code> objects allow the names
	 * to be weakly held and so unused ones can be garbage collected.  The Map will have
	 * <code>NameEntry</code> objects for all of the currently live ASName objects plus 
	 * ones for the garbage collected ASName objects that haven't yet been removed from the Map.
	 * These "dead" <code>NameEntry</code> objects get removed on every insertion into the Map.
	 */
	private static final Map nameMap = new HashMap();	

	/** Reference queue for NameEntry objects that have had their ASName GC'ed.
	 */
	private static final ReferenceQueue queue = new ReferenceQueue();

	/** This is a special one only search object used for looking into the HashMap.
	 */
	private static final NameSearchEntry searchEntry = new NameSearchEntry();

	public static final ASNameTrie TRIE = new ASNameTrie();
	
	
	


	/** The value of this ASName.
	 */
	private byte[] contents;

	/**
	 * The hash value used for hashCode().
	 */
	private int hash;

	

	/**
	 * The constructor for the ASName class is invoked indirectly via the
	 * <code>create</code> and <code>getASName</code> methods.
	 *
	 * @param b	an array containing the bytes for the ASName.
	 * @param start offset into the contents array where the ASName	data begins.
	 * @param length	number of bytes to read from the contents array.
	 *
	 * @see #create(String)
	 * @see #getName(byte[], int, int)
	 */
	private ASName(byte[] b, int start, int length)
	{
		this.contents = new byte[length];
		System.arraycopy(b, start, this.contents, 0, length);
		this.hash = generateHash(this.contents, 0, length);
	}

	private static final int generateHash(byte[] b, int start, int length)
	{
		int hash = 0;
		for (int i = 0; i < length; i++)
		{
			hash = 	hash * 31 + b[i + start];
		}
		return hash;
	}

	/** 
	 * This method should be called only from Debugger
	 * Gibson clients should call one of the two asString() methods
	 * 
	 * @see java.lang.Object#toString()
	 */
	@Override
	public String toString()
	{
		return '/' + asString(true);
	}

	/**
	 * Make a Java string out of an ASName. Creates UTF-8 representation of the name.
	 * @return UTF-8 encoded String representation of the name
	 */
	public String asString()
	{
		String nameStr;
		try {
			nameStr = new String(this.contents, UTFSupport.UTF8);
		} catch (UnsupportedEncodingException e) {
			throw new RuntimeException("Platform does not support UTF-8.", e);
		}
		return nameStr;
	}

	/**
	 * Make a Java string out of an ASName
	 * @param withoutEscapes indicator to produce UTF-8 representation and escaped 
	 * representation of the name when UTF-8 representation cannot be produced (true) 
	 * or escaped representation only (false)
	 * @return String representation of the name. Depending on the parameter this 
	 * representation contains only escaped version of the name or UTF-8 representation 
	 * when it can be produced and escaped representation of the name when UTF-8 representation 
	 * cannot be produced 
	 */
	public String asString(boolean withoutEscapes)
	{
		String nameStr = null;
		if (withoutEscapes)
			nameStr = asString();
		if (nameStr == null)
		{
			int maxEscapeByteSize = this.contents.length * 3;
			byte[] escapedBytes = new byte[maxEscapeByteSize];
			int len = outputString(escapedBytes);
			StringBuilder stringRep = new StringBuilder(maxEscapeByteSize);
			int byteInd;
			for (byteInd = 0; byteInd < len; byteInd++)
			{
				stringRep.append((char) escapedBytes[byteInd]);
			}
			nameStr = stringRep.toString();
		}
		return nameStr;
	}

	/**
	 * Get the bytes associated with this name. Clients must NOT alter the returned array.
	 */
	public byte[] getBytes()
	{
		return this.contents;
	}

	private int outputString(byte[] bytes)
	{
		int len = 0;
		int nameInd;
		for (nameInd = 0; nameInd < this.contents.length; nameInd++) {
			byte curByte = this.contents[nameInd];
			if ((curByte != 0) && ((curByte == '#') || !ByteOps.isRegular(curByte) || (curByte < 33) || (curByte > 126))) {
				String hexStr = Integer.toHexString(curByte & 0x000000ff).toUpperCase();
				byte[] hexBytes;
				try
				{
					// These are digits. In UTF-8 they look in bytes same as in chars 
					hexBytes = hexStr.getBytes(UTFSupport.UTF8);
				} catch (UnsupportedEncodingException e)
				{
					throw new RuntimeException("Platform does not support UTF-8", e);
				}
				bytes[len++] = '#';
				if (hexBytes.length == 1) {
					bytes[len++] = '0';
					bytes[len++] = hexBytes[0];
				} else {
					bytes[len++] = hexBytes[0];
					bytes[len++] = hexBytes[1];
				}
			} else {
				bytes[len++] = curByte;
			}
		}
		return len;
	}

	/**
	 * Writes the ASName in to the given OutputStream in the format expected by the PDF Spec.
	 * @see com.adobe.internal.pdftoolkit.core.types.ASObject#write(OutputByteStream)
	 *
	 * @param outputByteStream OutputByteStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
	throws PDFIOException
	{
		// Maximum size of array escapedBytes is thrice the num of bytes in contents
		byte[] escapedBytes = new byte[this.contents.length * 3];
		int len = outputString(escapedBytes);
		try {
			outputByteStream.write('/');
			outputByteStream.write(escapedBytes, 0, len);
		} catch (IOException e) {
			throw new PDFIOException(e);
		}

	}

	/**
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object otherObject)
	{
		if (this == otherObject)
		{
			return true;
		}
		return false;
	}

	@Override
	public int hashCode()
	{
		return this.hash;
	}

	/**
	 * @see java.lang.Comparable#compareTo(java.lang.Object)
	 */
	public int compareTo(Object anotherName)
	{
		if (!(anotherName instanceof ASName))
			throw new RuntimeException("Expected ASName");

		byte v1[] = this.contents;
		byte v2[] = ((ASName) anotherName).contents;
		int len1 = v1.length;
		int len2 = v2.length;
		int n = Math.min(len1, len2);
		int k = 0;
		while (k < n) 
		{
			if (v1[k] != v2[k]) 
			{
				return (v1[k]&0xff) - (v2[k]&0xff);
			}
			k++;
		}
		// If they are the same so far then the longer one is "greater".
		return len1 - len2;
	}

	/**
	 * Obtains the ASName corresponding to the specified portion of the
	 * specified byte array. If the ASName cannot be found, one is created.
	 * <p>
	 * Because the method modifies static data, it is synchronized for thread
	 * safety.
	 *
	 * @param b	data whose ASName is to be obtained
	 * @param start	starting offset into data
	 * @param length	length of the data to consider
	 * @return The ASName corresponding to the specified portion of the
	 * 			specified byte array.
	 */
	public static ASName getName(byte[] b, int start, int length)
	{
		// First check if this name is present in TRIE 
		ASName name = TRIE.get(b, start, length);
		if(name != null)
		{
			return name;
		}
		return getNameInternal(b, start, length);
	}
	
	public static synchronized ASName getNameInternal(byte[] b, int start, int length)
	{
		cleanValues();
		ASName.searchEntry.setBytes(b, start, length);
		ASName name = getNameFromMap(ASName.searchEntry);
		searchEntry.setBytes(null, 0, 0);

		if (name == null)
		{
			name = new ASName(b, start, length);
			addNameToMap(name);
		}		
		return name;
	}
	

	/**
	 * Obtains the ASName corresponding to the 
	 * specified byte array. If the ASName cannot be found, one is created.
	 * 
	 * @param b	data whose ASName is to be obtained
	 * @return The ASName corresponding to the specified portion of the
	 * 			specified byte array.
	 */
	public static ASName getName(byte[] b)
	{
		return ASName.getName(b, 0, b.length);
	}

	/**
	 * Use the given <code>NameSearchEntry</code> to look for a matching
	 * <code>ASName</code> in the singleton collection.
	 * @param search The search key to use when finding an ASName object in the singleton collection.
	 * @return The corresponding ASName object if found, null otherwise.
	 */
	private static ASName getNameFromMap(NameSearchEntry search)
	{
		NameEntry entry = (NameEntry) ASName.nameMap.get(search);
		ASName name = null;
		if (entry == null)
		{
			return null;
		}

		name = (ASName) entry.get();
		if (name == null)
		{
			/* Hmmm, we got the container but it had no name in it.
			 * I think this could only happen if garbage collection
			 * occured inside the HashMap after we found the NameEntry
			 * but before we looked inside and made a hard reference.
			 */
			ASName.cleanValues();
		}
		return name;
	}

	/**
	 * Add the ASName to the singleton collection.
	 * @param name ASName to add to the singleton collection.
	 */
	private static void addNameToMap(ASName name)
	{
		NameEntry entry = new NameEntry(ASName.queue, name);
		ASName.nameMap.put(entry, entry);
	}

	/**
	 * Clean any of the <code>NameEntry</code> objects in the queue from the
	 * HashMap.  These <code>NameEntry</code> objects contained <code>ASName</code> objects
	 * which have been garbage collected and need to removed from the HashMap as they are
	 * invalid.
	 */
	private static void cleanValues()
	{
		Object obj;
		while ((obj = ASName.queue.poll()) != null)
		{
			NameEntry entry = (NameEntry) obj;
			ASName.nameMap.remove(entry);
		}
	}

	/**
	 * Creates an ASName based on the specified string.
	 * <p>
	 * If this method throws an exception, it is due to the
	 * ASName table being full.
	 *
	 * @param strName     the string content for the ASName
	 * @return An ASName created from the specified string.
	 */
	public static ASName create(String strName)
	{
		// TODO: Handle name encoding issues per PDF Reference Manual
		//       version 1.4 section 3.2.4 and Appendix H.3 note 5.
		if (strName == null) 
		{
			return null;
		}
		byte[] bytes = UTF8Support.toUTF8(strName);
		return getName(bytes, 0, bytes.length);
	}

	/**
	 * This class is used to probe the HashMap for existing ASName objects that
	 * have the same bytes as the search parameter.  To allow this to work correctly
	 * the <code>hashCode()</code> and <code>equals()</code> methods of this class
	 * and the <code>NameEntry</code> class must return identical results for
	 * the same byte strings.
	 */
	private static final class NameSearchEntry
	{
		byte[] b;
		int start = 0;
		int length = 0;

		int hash = 0;

		private NameSearchEntry()
		{
		}

		private void setBytes(byte[] b, int start, int length)
		{
			this.b = b;
			this.hash = 0;
			this.start = start;
			this.length = length;
			this.hash = ASName.generateHash(b, start, length);
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj)
		{
			if (this == obj)
			{
				return true;
			}

			if (obj instanceof NameEntry)
			{
				ASName name = (ASName) ((NameEntry) obj).get();
				if (name == null)
				{
					return false;
				}
				return this.equals(name.contents, 0, name.contents.length);
			}
			if (obj instanceof NameSearchEntry)
			{
				NameSearchEntry searchEntry = (NameSearchEntry) obj;
				return this.equals(searchEntry.b, searchEntry.start, searchEntry.length);
			}
			return false;
		}

		private boolean equals(byte[] b, int start, int length)
		{
			if (this.length == length)
			{
				for (int i = 0; i < length; i++)
				{
					if (this.b[this.start + i] != b[start + i])
					{
						return false;
					}
				}
				return true;
			}
			return false;
		}

		/**
		 * @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode()
		{
			return this.hash;
		}
		
	}

	/**
	 * These are the weak entries in the HashMap that will hold the ASName.
	 * There will be a strict one to one mapping of these to ASName object.
	 * These <code>NameEntry</code> objects are then placed in the HashMap
	 * for both key and value.  This allows us to use the standard HashMap
	 * to retrieve these objects and ultimately the ASName and yet still
	 * allow them to be weakly referenced by the HashMap.
	 */
	private static final class NameEntry extends WeakReference
	{
		int hash = 0;

		NameEntry(ReferenceQueue queue, ASName name)
		{
			super(name, queue);
			this.hash = name.hashCode();
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#equals(java.lang.Object)
		 */
		@Override
		public boolean equals(Object obj)
		{
			if (this == obj)
			{
				return true;
			}

			if (obj instanceof NameEntry)
			{
				ASName name = (ASName) ((NameEntry) obj).get();
				if (name == null)
				{
					return false;
				}
				return this.equals(name.contents, 0, name.contents.length);
			}
			if (obj instanceof NameSearchEntry)
			{
				NameSearchEntry searchEntry = (NameSearchEntry) obj;
				return this.equals(searchEntry.b, searchEntry.start, searchEntry.length);
			}
			return false;
		}

		private boolean equals(byte[] b, int start, int length)
		{
			ASName name = (ASName) this.get();

			if (name == null)
			{
				return false;
			}

			if (name.contents.length == length)
			{
				for (int i = 0; i < length; i++)
				{
					if (name.contents[i] != b[start + i])
					{
						return false;
					}
				}
				return true;
			}
			return false;
		}

		/* (non-Javadoc)
		 * @see java.lang.Object#hashCode()
		 */
		@Override
		public int hashCode()
		{
			return this.hash;
		}

			
	}

	//-----------------------------------------------	Predefined PDF names ---

	public static final ASName k_BlackIs1 = create("BlackIs1");
	public static final ASName CM = create("cm");
	public static final ASName k_Columns = create("Columns");
	public static final ASName k_Colors = create("Colors");
	public static final ASName k_DamagedRowsBeforeError = create("DamagedRowsBeforeError");
	public static final ASName k_DecodeParms = create("DecodeParms");
	public static final ASName k_FDecodeParms = create("FDecodeParms");
	public static final ASName k_EarlyChange = create("EarlyChange");
	public static final ASName k_EncodedByteAlign = create("EncodedByteAlign");
	public static final ASName k_EndOfLine = create("EndOfLine");
	public static final ASName k_EndOfBlock = create("EndOfBlock");
	public static final ASName k_First = create("First");
	public static final ASName k_N = create("N");
	public static final ASName k_ObjStm = create("ObjStm");
	public static final ASName k_Open = create("Open");
	public static final ASName k_OpenAction = create("OpenAction");
	public static final ASName k_Predictor = create("Predictor");
	public static final ASName k_Rows = create("Rows");
	public static final ASName k_XRef = create("XRef");
	public static final ASName k_XRefStm = create("XRefStm");
	public static final ASName k_Client_Data_PDF = create("Client_Data_PDF");
	public static final ASName k_Client_Data_FDF = create("Client_Data_FDF");
	public static final ASName k_Client_Data_Java_Value = create("Client_Data_Java_Value");
	public static final ASName k_Cos_Object_Dirty = create("Cos_Object_Dirty");
	public static final ASName k_Cos_Object_Cloned = create("Cos_Object_Cloned");

	public static final ASName k_CL = create("CL");
	public static final ASName k_Btn = create("Btn");
	public static final ASName k_Opt = create("Opt");
	public static final ASName k_RV = create("RV");
	public static final ASName k_C = create("C");
	public static final ASName k_Text = create("Text");
	public static final ASName k_L = create("L");
	public static final ASName k_LE = create("LE");
	public static final ASName k_Cap = create("Cap");
	public static final ASName k_Link = create("Link");
	public static final ASName k_FreeText = create("FreeText");
	public static final ASName k_FreeTextCallout = create("FreeTextCallout");
	public static final ASName k_Line = create("Line");
	public static final ASName k_Linear = create("Linear");
	public static final ASName k_Square = create("Square");
	public static final ASName k_Circle = create("Circle");
	public static final ASName k_Highlight = create("Highlight");
	public static final ASName k_Underline = create("Underline");
	public static final ASName k_Squiggly = create("Squiggly");
	public static final ASName k_Projection = create("Projection");
	public static final ASName k_RichMedia = create("RichMedia");
	public static final ASName k_StickyNote = create("StickyNote");
	public static final ASName k_St = create("St");
	public static final ASName k_StrikeOut = create("StrikeOut");
	public static final ASName k_Stamp = create("Stamp");
	public static final ASName k_Ink = create("Ink");
	public static final ASName k_Pause = create("Pause");
	public static final ASName k_PA = create("PA");
	public static final ASName k_PC = create("PC");
	public static final ASName k_PI = create("PI");
	public static final ASName k_Play = create("Play");
	public static final ASName k_PO = create("PO");
	public static final ASName k_PV = create("PV");
	public static final ASName k_Popup = create("Popup");
	public static final ASName k_Polygon = create("Polygon");
	public static final ASName k_PolyLine = create("PolyLine");
	public static final ASName k_FileAttachment = create("FileAttachment");
	public static final ASName k_Filespec = create("Filespec");
	public static final ASName k_Sound = create("Sound");
	public static final ASName k_Movie = create("Movie");
	public static final ASName k_NM = create("NM");

	public static final ASName k_A = create("A");
	public static final ASName k_AA = create("AA");
	public static final ASName k_AbsoluteColorimetric = create("AbsoluteColorimetric");
	public static final ASName k_AcroForm = create("AcroForm");
	public static final ASName k_Action = create("Action");
	public static final ASName k_ADBE 	= ASName.create("ADBE");
	public static final ASName k_ADBE_AuthType = create("ADBE_AuthType");
	public static final ASName k_ADBE_Build = create("ADBE_Build");
	public static final ASName k_ADBE_PwdTime = create("ADBE_PwdTime");
	public static final ASName k_A85 = create("A85");
	public static final ASName k_AHx = create("AHx");
	public static final ASName k_AIS = create("AIS");
	public static final ASName k_All = create("All");
	public static final ASName k_AllPages = create("AllPages");
	public static final ASName k_AllOff = create("AllOff");
	public static final ASName k_AllOn = create("AllOn");
	public static final ASName k_Alternates = create("Alternates");
	public static final ASName k_And = create("And");
	public static final ASName k_AN = create("AN");
	public static final ASName k_Annot = create("Annot");
	public static final ASName k_Annotation = create("Annotation");
	public static final ASName k_Annots = create("Annots");
	public static final ASName k_AnnotStates = create("AnnotStates");
	public static final ASName k_AnyOff = create("AnyOff");
	public static final ASName k_AnyOn = create("AnyOn");
	public static final ASName k_AP = create("AP");
	public static final ASName k_App = create("App");
	public static final ASName k_ArtBox = create("ArtBox");
	public static final ASName k_AS = create("AS");
	public static final ASName k_Ascent = create("Ascent");
	public static final ASName k_ASCII85Decode = create("ASCII85Decode");
	public static final ASName k_ASCII85Encode = create("ASCII85Encode");
	public static final ASName k_ASCIIHexDecode = create("ASCIIHexDecode");
	public static final ASName k_ASCIIHexEncode = create("ASCIIHexEncode");
	public static final ASName k_AuthEvent = create("AuthEvent");
	public static final ASName k_Author = create("Author");
	public static final ASName k_auto = create("auto");
	public static final ASName k_AvgWidth = create("AvgWidth");
	public static final ASName k_b = create("b");
	public static final ASName k_B = create("B");
	public static final ASName k_BBox = create("BBox");
	public static final ASName k_BaseEncoding = create("BaseEncoding");
	public static final ASName k_BaseFont = create("BaseFont");
	public static final ASName k_BaseState = create("BaseState");
	public static final ASName k_BaseVersion 	= ASName.create("BaseVersion");
	public static final ASName k_BC = create("BC");
	public static final ASName k_BDC = create("BDC");
	public static final ASName k_BI = create("BI");
	public static final ASName k_BitsPerComponent = create("BitsPerComponent");
	public static final ASName k_BleedBox = create("BleedBox");
	public static final ASName k_Border = create("Border");
	public static final ASName k_BE = create("BE");
	public static final ASName k_BG = create("BG");
	public static final ASName k_BG2 = create("BG2");
	public static final ASName k_BM = create("BM");
	public static final ASName k_BMC = create("BMC");
	public static final ASName k_BoxColorInfo = create("BoxColorInfo");
	public static final ASName k_BPC = create("BPC");
	public static final ASName k_BS = create("BS");
	public static final ASName k_bStar = create("b*");
	public static final ASName k_BStar = create("B*");
	public static final ASName k_BT = create("BT");
	public static final ASName k_BX = create("BX");
	public static final ASName k_ByteRange = create("ByteRange");
	public static final ASName k_c = create("c");
	public static final ASName k_ca = create("ca");
	public static final ASName k_Caret = create("Caret");
	public static final ASName k_Catalog = create("Catalog");
	public static final ASName k_CCF = create("CCF");
	public static final ASName k_Cert = create("Cert");
	public static final ASName k_CF = create("CF");
	public static final ASName k_CFM = create("CFM");
	public static final ASName k_Ch = create("Ch");
	public static final ASName k_CharProcs = create("CharProcs");
	public static final ASName k_CheckSum = create("CheckSum");
	public static final ASName k_cm = create("cm");
	public static final ASName k_CMD = create("CMD");
	public static final ASName k_CA = create("CA");
	public static final ASName k_CalCMYK = create("CalCMYK");
	public static final ASName k_CalGray = create("CalGray");
	public static final ASName k_CalRGB = create("CalRGB");
	public static final ASName k_CapHeight = create("CapHeight");
	public static final ASName k_Category = create("Category");
	public static final ASName k_CCITTFaxDecode = create("CCITTFaxDecode");
	public static final ASName k_CharSet = create("CharSet");
	public static final ASName k_CIDFontType0 = create("CIDFontType0");
	public static final ASName k_CIDFontType2 = create("CIDFontType2");
	public static final ASName k_CIDSet = create("CIDSet");
	public static final ASName k_CIDSystemInfo = create("CIDSystemInfo");
	public static final ASName k_CIDToGIDMap = create("CIDToGIDMap");
	public static final ASName k_ClassMap = create("ClassMap");
	public static final ASName k_CMap = create("CMap");
	public static final ASName k_CMapName = create("CMapName");
	public static final ASName k_CMYK = create("CMYK");
	public static final ASName k_CO = create("CO");
	public static final ASName k_Colorants = create("Colorants");
	public static final ASName k_ColorBurn = create("ColorBurn");
	public static final ASName k_ColorDodge = create("ColorDodge");
	public static final ASName k_ColorSpace = create("ColorSpace");
	public static final ASName k_Comments = create("Comments");
	public static final ASName k_Compatible = create("Compatible");
	public static final ASName k_Condensed = create("Condensed");
	public static final ASName k_Configs = create("Configs");
	public static final ASName k_ContactInfo = create("ContactInfo");
	public static final ASName k_Contents = create("Contents");
	public static final ASName k_Count = create("Count");
	public static final ASName k_CreationDate = create("CreationDate");
	public static final ASName k_Creator = create("Creator");
	public static final ASName k_CreatorInfo = create("CreatorInfo");
	public static final ASName k_CropBox = create("CropBox");
	public static final ASName k_Crypt = create("Crypt");
	public static final ASName k_Identity = create("Identity");
	public static final ASName k_CryptFilter = create("CryptFilter");
	public static final ASName k_cs = create("cs");
	public static final ASName k_CS = create("CS");
	public static final ASName k_d = create("d");
	public static final ASName k_D = create("D");
	public static final ASName k_d0 = create("d0");
	public static final ASName k_d1 = create("d1");
	public static final ASName k_DA = create("DA");
	public static final ASName k_Darken = create("Darken");
	public static final ASName k_DCT = create("DCT");
	public static final ASName k_DCTDecode = create("DCTDecode");
	public static final ASName k_Default = create("Default");
	public static final ASName k_DefaultForPrinting = create("DefaultForPrinting");
	public static final ASName k_DescendantFonts = create("DescendantFonts");
	public static final ASName k_Descent = create("Descent");
	public static final ASName k_Difference = create("Difference");
	public static final ASName k_Differences = create("Differences");
	public static final ASName k_Do = create("Do");
	public static final ASName k_DocOpen = create("DocOpen");
	public static final ASName k_DocMDP = create("DocMDP");
	public static final ASName k_UR = create("UR");
	public static final ASName k_UR3 = create("UR3");
	public static final ASName k_DoubleQuote = create("\"");
	public static final ASName k_Date = create("Date");
	public static final ASName k_Data = create("Data");
	public static final ASName k_Desc = create("Desc");
	public static final ASName k_Design = create("Design");
	public static final ASName k_Dest = create("Dest");
	public static final ASName k_Dests = create("Dests");
	public static final ASName k_DeviceCMYK = create("DeviceCMYK");
	public static final ASName k_DeviceGray = create("DeviceGray");
	public static final ASName k_DeviceN = create("DeviceN");
	public static final ASName k_DeviceRGB = create("DeviceRGB");
	public static final ASName k_DigestValue = create("DigestValue");
	public static final ASName k_DOS = create("DOS");
	public static final ASName k_DP = create("DP");
	public static final ASName k_DR = create("DR");
	public static final ASName k_DS = create("DS");
	public static final ASName k_DV = create("DV");
	public static final ASName k_DW = create("DW");
	public static final ASName k_DW2 = create("DW2");
	public static final ASName k_E = create("E");
	public static final ASName k_EF = create("EF");
	public static final ASName k_EFF = create("EFF");
	public static final ASName k_EFOpen = create("EFOpen");
	public static final ASName k_EI = create("EI");
	public static final ASName k_EMC = create("EMC");
	public static final ASName k_Encrypt = create("Encrypt");
	public static final ASName k_EncryptMetadata = create("EncryptMetadata");
	public static final ASName k_EmbeddedFiles = create("EmbeddedFiles");
	public static final ASName k_Encoding = create("Encoding");
	public static final ASName k_ET = create("ET");
	public static final ASName k_Event = create("Event");
	public static final ASName k_EX = create("EX");
	public static final ASName k_Exclusion = create("Exclusion");
	public static final ASName k_Expanded = create("Expanded");
	public static final ASName k_Export = create("Export");
	public static final ASName k_ExportState = create("ExportState");
	public static final ASName k_Extends = create("Extends");
	public static final ASName k_ExtensionLevel = ASName.create("ExtensionLevel");
	public static final ASName k_ExtGState = create("ExtGState");
	public static final ASName k_ExtraCondensed = create("ExtraConsdensed");
	public static final ASName k_ExtraExpanded = create("ExtraExpanded");
	public static final ASName k_f = create("f");
	public static final ASName k_F = create("F");
	public static final ASName k_F1 = create("F1");
	public static final ASName k_false = create("false");
	public static final ASName k_FD = create("FD");
	public static final ASName k_FDF = create("FDF");
	public static final ASName k_Ff = create("Ff");
	public static final ASName k_FG = create("FG");
	public static final ASName k_FT = create("FT");
	public static final ASName k_FS = create("FS");
	public static final ASName k_fStar = create("f*");
	public static final ASName k_Fields = create("Fields");
	public static final ASName k_Fit = create("Fit");
	public static final ASName k_FitH = create("FitH");
	public static final ASName k_FitV = create("FitV");
	public static final ASName k_FitR = create("FitR");
	public static final ASName k_FitB = create("FitB");
	public static final ASName k_FitBH = create("FitBH");
	public static final ASName k_FitBV = create("FitBV");
	public static final ASName k_Filter = create("Filter");
	public static final ASName k_FFilter = create("FFilter");
	public static final ASName k_FirstChar = create("FirstChar");
	public static final ASName k_FixedPrint = create("FixedPrint");
	public static final ASName k_Fl = create("Fl");
	public static final ASName k_FL = create("FL");
	public static final ASName k_Flags = create("Flags");
	public static final ASName k_FlateDecode = create("FlateDecode");
	public static final ASName k_Fo = create("Fo");
	public static final ASName k_Font = create("Font");
	public static final ASName k_FontBBox = create("FontBBox");
	public static final ASName k_FontDescriptor = create("FontDescriptor");
	public static final ASName k_FontFamily = create("FontFamily");
	public static final ASName k_FontFauxing = create("FontFauxing");
	public static final ASName k_FontFile = create("FontFile");
	public static final ASName k_FontFile2 = create("FontFile2");
	public static final ASName k_FontFile3 = create("FontFile3");
	public static final ASName k_FontMatrix = create("FontMatrix");
	public static final ASName k_FontName = create("FontName");
	public static final ASName k_FontStretch = create("FontStretch");
	public static final ASName k_FontWeight = create("FontWeight");
	public static final ASName k_g = create("g");
	public static final ASName k_G = create("G");
	public static final ASName k_GoTo3DView = create("GoTo3DView");
	public static final ASName k_GoTo = create("GoTo");
	public static final ASName k_GoToR = create("GoToR");
	public static final ASName k_GoToE = create("GoToE");
	public static final ASName k_Group = create("Group");
	public static final ASName k_gs = create("gs");
	public static final ASName k_h = create("h");
	public static final ASName k_H = create("H");
	public static final ASName k_HardLight = create("HardLight");
	public static final ASName k_HF = create("HF");
	public static final ASName k_Height = create("Height");
	public static final ASName k_HT = create("HT");
	public static final ASName k_Hide = create("Hide");
	public static final ASName k_HideToolbar = create("HideToolbar");
	public static final ASName k_HideMenubar = create("HideMenubar");
	public static final ASName k_HideWindowUI = create("HideWindowUI");
	public static final ASName k_Hue = create("Hue");
	public static final ASName k_FitWindow = create("FitWindow");
	public static final ASName k_CenterWindow = create("CenterWindow");
	public static final ASName k_DisplayDocTitle = create("DisplayDocTitle");
	public static final ASName k_NonFullScreenPageMode = create("NonFullScreenPageMode");
	public static final ASName k_Direction = create("Direction");
	public static final ASName k_ViewArea = create("ViewArea");
	public static final ASName k_ViewClip = create("ViewClip");
	public static final ASName k_PrintArea = create("PrintArea");
	public static final ASName k_PrintClip = create("PrintClip");
	public static final ASName k_i = create("i");
	public static final ASName k_I = create("I");
	public static final ASName k_IC = create("IC");
	public static final ASName k_ICCBased = create("ICCBased");
	public static final ASName k_ID = create("ID");
	public static final ASName k_IDS = create("IDS");
	public static final ASName k_IDTree = create("IDTree");
	public static final ASName k_IM = create("IM");
	public static final ASName k_ImageMask = create("ImageMask");
	public static final ASName k_Ind = create("Ind");
	public static final ASName k_Index = create("Index");
	public static final ASName k_Indexed = create("Indexed");
	public static final ASName k_Intent = create("Intent");
	public static final ASName k_Interpolate = create("Interpolate");
	public static final ASName k_IsMap = create("IsMap");
	public static final ASName k_Info = create("Info");
	public static final ASName k_IT = create("IT");
	public static final ASName k_ItalicAngle = create("ItalicAngle");
	public static final ASName k_j = create("j");
	public static final ASName k_J = create("J");
	public static final ASName k_JavaScript = create("JavaScript");
	public static final ASName k_JBIG2Decode = create("JBIG2Decode");
	public static final ASName k_JPXDecode = create("JPXDecode");
	public static final ASName k_JS = create("JS");
	public static final ASName k_InkList = create("InkList");
	public static final ASName k_IRT = create("IRT");
	public static final ASName k_k = create("k");
	public static final ASName k_K = create("K");
	public static final ASName k_Kids = create("Kids");
	public static final ASName k_Keywords = create("Keywords");
	public static final ASName k_l = create("l");
	public static final ASName k_Lab = create("Lab");
	public static final ASName k_Lang = create("Lang");
	public static final ASName k_Language = create("Language");
	public static final ASName k_Last = create("Last");
	public static final ASName k_LastChar = create("LastChar");
	public static final ASName k_LastModified = create("LastModified");
	public static final ASName k_Launch = create("Launch");
	public static final ASName k_LC = create("LC");
	public static final ASName k_Leading = create("Leading");
	public static final ASName k_Legal = create("Legal");
	public static final ASName k_Length = create("Length");
	public static final ASName k_Length1 = create("Length1");
	public static final ASName k_Length2 = create("Length2");
	public static final ASName k_Length3 = create("Length3");
	public static final ASName k_Lighten = create("Lighten");
	public static final ASName k_Limits = create("Limits");
	public static final ASName k_Linearized = create("Linearized");
	public static final ASName k_ListMode = create("ListMode");
	public static final ASName k_LJ = create("LJ");
	public static final ASName k_Location = create("Location");
	public static final ASName k_Lock = create("Lock");
	public static final ASName k_Locked = create("Locked");
	public static final ASName k_LW = create("LW");
	public static final ASName k_LZW = create("LZW");
	public static final ASName k_LZWDecode = create("LZWDecode");
	public static final ASName k_LZWEncode = create("LZWEncode");
	public static final ASName k_m = create("m");
	public static final ASName k_max = create("max");
	public static final ASName k_min = create("min");
	public static final ASName k_M = create("M");
	public static final ASName k_Mac = create("Mac");
	public static final ASName k_Marked = create("Marked");
	public static final ASName k_MarkInfo = create("MarkInfo");
	public static final ASName k_MarkStyle = create("MarkStyle");
	public static final ASName k_Mask = create("Mask");
	public static final ASName k_Matrix = create("Matrix");
	public static final ASName k_MaxLen = create("MaxLen");
	public static final ASName k_Multiline = create("Multiline");
	public static final ASName k_MaxWidth = create("MaxWidth");
	public static final ASName k_MCID = create("MCID");
	public static final ASName k_MCR = create("MCR");
	public static final ASName k_MediaBox = create("MediaBox");
	public static final ASName k_MediaClips = create("MediaClips");
	public static final ASName k_Metadata = create("Metadata");
	public static final ASName k_MissingWidth = create("MissingWidth");
	public static final ASName k_MK = create("MK");
	public static final ASName k_ML = create("ML");
	public static final ASName k_MMType1 = create("MMType1");
	public static final ASName k_ModDate = create("ModDate");
	public static final ASName k_MP = create("MP");
	public static final ASName k_Multiply = create("Multiply");
	public static final ASName k_n = create("n");
	public static final ASName k_Name = create("Name");
	public static final ASName k_Named = create("Named");
	public static final ASName k_Names = create("Names");
	public static final ASName k_Next = create("Next");
	public static final ASName k_NewWindow = create("NewWindow");
	public static final ASName k_NeedAppearances = create("NeedAppearances");
	public static final ASName k_NeedsRendering = create("NeedsRendering");
	public static final ASName k_None = create("None");
	public static final ASName k_NonEFontNoWarn = create("NonEFontNoWarn");
	public static final ASName k_Normal = create("Normal");
	public static final ASName k_Note = create("Note");
	public static final ASName k_Not = create("Not");
	public static final ASName k_Nums = create("Nums");
	public static final ASName k_null = create("null");
	public static final ASName k_O = create("O");
	public static final ASName k_OBJR = create("OBJR");
	public static final ASName k_Obj = create("Obj");
	public static final ASName k_OC = create("OC");
	public static final ASName k_OCG = create("OCG");
	public static final ASName k_OCGs = create("OCGs");
	public static final ASName k_OCMD = create("OCMD");
	public static final ASName k_OCProperties = create("OCProperties");
	public static final ASName k_OFF = create("OFF");
	public static final ASName k_ON = create("ON");
	public static final ASName k_op = create("op");
	public static final ASName k_OP = create("OP");
	public static final ASName k_OPI = create("OPI");
	public static final ASName k_OPM = create("OPM");
	public static final ASName k_Or = create("Or");
	public static final ASName k_Order = create("Order");
	public static final ASName k_Ordering = create("Ordering");
	public static final ASName k_Org = create("Org");
	public static final ASName k_OS = create("OS");
	public static final ASName k_Oscillating = create("Oscillating");
	public static final ASName k_Outlines = create("Outlines");
	public static final ASName k_Overlay = create("Overlay");
	public static final ASName k_P = create("P");
	public static final ASName k_Password = create("Password");
	public static final ASName k_Page = create("Page");
	public static final ASName k_Pg = create("Pg");
	public static final ASName k_PageElement = create("PageElement");
	public static final ASName k_PageLabel = create("PageLabel");
	public static final ASName k_PageLabels = create("PageLabels");
	public static final ASName k_PageLayout = create("PageLayout");
	public static final ASName k_Pages = create("Pages");
	public static final ASName k_PageMode = create("PageMode");
	public static final ASName k_Params = create("Params");
	public static final ASName k_Parent = create("Parent");
	public static final ASName k_ParentTree = create("ParentTree");
	public static final ASName k_ParentTreeNextKey = create("ParentTreeNextKey");
	public static final ASName k_Pattern = create("Pattern");
	public static final ASName k_PatternType = create("PatternType");
	public static final ASName k_PCM = create("PCM");
	public static final ASName k_PDF = create("PDF");
	public static final ASName k_Perceptual = create("Perceptual");
	public static final ASName k_Perms = create("Perms");
	public static final ASName k_PieceInfo = create("PieceInfo");
	public static final ASName k_PMD = create("PMD");
	public static final ASName k_Poster = create("Poster");
	public static final ASName k_PS = create("PS");	
	public static final ASName k_Preferred = create("Preferred");
	public static final ASName k_Prev = create("Prev");
	public static final ASName k_PreRelease = create("PreRelease");
	public static final ASName k_Preview = create("Preview");
	public static final ASName k_Print = create("Print");
	public static final ASName k_PrintState = create("PrintState");
	public static final ASName k_PrinterMark = create("PrinterMark");
	public static final ASName k_Private = create("Private");
	public static final ASName k_ProcSet = create("ProcSet");
	public static final ASName k_Producer = create("Producer");
	public static final ASName k_Properties = create("Properties");
	public static final ASName k_Prop_Build = create("Prop_Build");
	public static final ASName k_PubSec = create("PubSec");
	public static final ASName k_Q = create("Q");
	public static final ASName k_q = create("q");
	public static final ASName k_QuadPoints = create("QuadPoints");
	public static final ASName k_R = create("R");
	public static final ASName k_RC = create("RC");
	public static final ASName k_RD = create("RD");
	public static final ASName k_re = create("re");
	public static final ASName k_ReaderRequirement = create("ReaderRequirement");
	public static final ASName k_ReaderRequirements = create("ReaderRequirements");
	public static final ASName k_Ref = create("Ref");
	public static final ASName k_RelativeColorimetric = create("RelativeColorimetric");
	public static final ASName k_Resources = create("Resources");
	public static final ASName k_Reason = create("Reason");
	public static final ASName k_Rect = create("Rect");
	public static final ASName k_Recipients = create("Recipients");
	public static final ASName k_Reference = create("Reference");
	public static final ASName k_Registry = create("Registry");
	public static final ASName k_Rendition = create("Rendition");
	public static final ASName k_Renditions = create("Renditions");
	public static final ASName k_Resume = create("Resume");
	public static final ASName k_REx = create("REx");
	public static final ASName k_RBGroups = create("RBGroups");
	public static final ASName k_rg = create("rg");
	public static final ASName k_RG = create("RG");
	public static final ASName k_RGB = create("RGB");
	public static final ASName k_ri = create("ri");
	public static final ASName k_RI = create("RI");
	public static final ASName k_RichMediaCommand = create("RichMediaCommand");
	public static final ASName k_RichMediaExecute = create("RichMediaExecute");
	public static final ASName k_Rights = create("Rights");
	public static final ASName k_RL = create("RL");
	public static final ASName k_RoleMap = create("RoleMap");
	public static final ASName k_Root = create("Root");
	public static final ASName k_Rotate = create("Rotate");
	public static final ASName k_RT = create("RT");
	public static final ASName k_RunLengthDecode = create("RunLengthDecode");
	public static final ASName k_s = create("s");
	public static final ASName k_S = create("S");
	public static final ASName k_SA = create("SA");
	public static final ASName k_Saturation = create("Saturation");
	public static final ASName k_SearchIndex = create("SearchIndex");
	public static final ASName k_Separation = create("Separation");
	public static final ASName k_SetOCGState = create("SetOCGState");
	public static final ASName k_sc = create("sc");
	public static final ASName k_SC = create("SC");
	public static final ASName k_scn = create("scn");
	public static final ASName k_SCN = create("SCN");
	public static final ASName k_SE = create("SE");
	public static final ASName k_SemiCondensed = create("SemiCondensed");
	public static final ASName k_SemiExpanded = create("SemiExpanded");
	public static final ASName k_sh = create("sh");
	public static final ASName k_Shading = create("Shading");
	public static final ASName k_ShadingType = create("ShadingType");
	public static final ASName k_Sig = create("Sig");
	public static final ASName k_DocTimeStamp = create("DocTimeStamp");
	public static final ASName k_SigQ = create("SigQ");
	public static final ASName k_SingleQuote = create("\'");
	public static final ASName k_Size = create("Size");
	public static final ASName k_CompressedSize = create("CompressedSize");
	public static final ASName k_SM = create("SM");
	public static final ASName k_SMask = create("SMask");
	public static final ASName k_SoftLight = create("SoftLight");
	public static final ASName k_Standard = create("Standard");
	public static final ASName k_State = create("State");
	public static final ASName k_StateModel = create("StateModel");
	public static final ASName k_StemH = create("StemH");
	public static final ASName k_StemV = create("StemV");
	public static final ASName k_StmF = create("StmF");
	public static final ASName k_Stm = create("Stm");
	public static final ASName k_StmOwn = create("StmOwn");
	public static final ASName k_Stop = create("Stop");
	public static final ASName k_StrF = create("StrF");
	public static final ASName k_StructElem = create("StructElem");
	public static final ASName k_StructParent = create("StructParent");
	public static final ASName k_StructParents = create("StructParents");
	public static final ASName k_StructTreeRoot = create("StructTreeRoot");
	public static final ASName k_Style = create("Style");
	public static final ASName k_Subj = create("Subj");
	public static final ASName k_Subject = create("Subject");
	public static final ASName k_SubFilter = create("SubFilter");
	public static final ASName k_Subtype = create("Subtype");
	public static final ASName k_Subtype2 = create("Subtype2");
	public static final ASName k_Supplement = create("Supplement");
	public static final ASName k_SV = create("SV");
	public static final ASName k_Sy = create("Sy");
	public static final ASName k_T = create("T");
	public static final ASName k_Tc = create("Tc");
	public static final ASName k_Td = create("Td");
	public static final ASName k_TD = create("TD");
	public static final ASName k_Templates = create("Templates");
	public static final ASName k_Tf = create("Tf");
	public static final ASName k_Threads = create("Threads");
	public static final ASName k_Title = create("Title");
	public static final ASName k_TI = create("TI");
	public static final ASName k_Tj = create("Tj");
	public static final ASName k_TJ = create("TJ");
	public static final ASName k_TK = create("TK");
	public static final ASName k_TL = create("TL");
	public static final ASName k_Tm = create("Tm");
	public static final ASName k_TM = create("TM");
	public static final ASName k_ToUnicode = create("ToUnicode");
	public static final ASName k_Tr = create("Tr");
	public static final ASName k_TR = create("TR");
	public static final ASName k_TR2 = create("TR2");
	public static final ASName k_Transparency = create("Transparency");
	public static final ASName k_TrapNet = create("TrapNet");	
	public static final ASName k_Trapped = create("Trapped");	
	public static final ASName k_TrimBox = create("TrimBox");
	public static final ASName k_TrustedMode = create("TrustedMode");
	public static final ASName k_true = create("true");
	public static final ASName k_TrueType = create("TrueType");
	public static final ASName k_Thread = create("Thread");
	public static final ASName k_Thumb = create("Thumb");
	public static final ASName k_Trans = create("Trans");
	public static final ASName k_Ts = create("Ts");
	public static final ASName k_TStar = create("T*");
	public static final ASName k_Ttl = create("Ttl");
	public static final ASName k_Tw = create("Tw");
	public static final ASName k_TU = create("TU");
	public static final ASName k_Tx = create("Tx");
	public static final ASName k_Type = create("Type");
	public static final ASName k_Type0 = create("Type0");
	public static final ASName k_Type1 = create("Type1");
	public static final ASName k_Type3 = create("Type3");
	public static final ASName k_Tz = create("Tz");
	public static final ASName k_U = create("U");
	public static final ASName k_UCR = create("UCR");
	public static final ASName k_UCR2 = create("UCR2");
	public static final ASName k_UltraCondensed = create("UltraCondensed");
	public static final ASName k_UltraExpanded = create("UltraExpanded");
	public static final ASName k_Unix = create("Unix");
	public static final ASName k_Unchanged = create("Unchanged");
	public static final ASName k_Usage = create("Usage");
	public static final ASName k_UseOutlines = create("UseOutlines");
	public static final ASName k_User = create("User");
	public static final ASName k_URI = create("URI");
	public static final ASName k_URL = create("URL");
	public static final ASName k_URLS = create("URLS");
	public static final ASName k_UseCMap = create("UseCMap");
	public static final ASName k_UserUnit = create("UserUnit");
	public static final ASName k_v = create("v");
	public static final ASName k_V = create("V");
	public static final ASName k_VE = create("VE");
	public static final ASName k_AESV2 = create("AESV2");
	public static final ASName k_AESV3 = create("AESV3");
	public static final ASName k_V2 = create("V2");
	public static final ASName k_View = create("View");
	public static final ASName k_ViewState = create("ViewState");
	public static final ASName k_ViewerPreferences = create("ViewerPreferences");
	public static final ASName k_Version = create("Version");
	public static final ASName k_Vertices = create("Vertices");
	public static final ASName k_VisiblePages = create("VisiblePages");
	public static final ASName k_w = create("w");
	public static final ASName k_W = create("W");
	public static final ASName k_W2 = create("W2");
	public static final ASName k_Watermark = create("Watermark");
	public static final ASName k_WStar = create("W*");
	public static final ASName k_Widget = create("Widget");
	public static final ASName k_Width = create("Width");
	public static final ASName k_Widths = create("Widths");
	public static final ASName k_Win = create("Win");
	public static final ASName k_WMode = create("WMode");
	public static final ASName k_XFA = create("XFA");
	public static final ASName k_XFAResources = create("XFAResources");
	public static final ASName k_XFAImages = create("XFAImages");
	public static final ASName k_XFAFlashAssets = create("XFAFlashAssets");
	public static final ASName k_UseBlackPTComp = create("UseBlackPTComp");
	public static final ASName k_XHeight = create("XHeight");
	public static final ASName k_XML = create("XML");
	public static final ASName k_XYZ = create("XYZ");
	public static final ASName k_XObject = create("XObject");
	public static final ASName k_y = create("y");
	public static final ASName k_Form = create("Form");
	public static final ASName k_FormType = create("FormType");
	public static final ASName k_Image = create("Image");
	public static final ASName k_X = create("X");
	public static final ASName k_XA = create("XA");
	public static final ASName k_XD = create("XD");
	public static final ASName k_Zoom = create("Zoom");

	public static final ASName k_SubmitForm = create("SubmitForm");
	public static final ASName k_ResetForm = create("ResetForm");
	public static final ASName k_ImportData = create("ImportData");

	public static final ASName k_Times_Roman = ASName.create("Times-Roman");
	public static final ASName k_TimesNewRomanPSMT = ASName.create("TimesNewRomanPSMT");
	public static final ASName k_TimesNewRoman = ASName.create("TimesNewRoman");
	public static final ASName k_TimesNewRomanPS = ASName.create("TimesNewRomanPS");

	public static final ASName k_Times_Bold = ASName.create("Times-Bold");
	public static final ASName k_TimesNewRomanPS_BoldMT = ASName.create("TimesNewRomanPS-BoldMT");
	public static final ASName k_TimesNewRomanCommaBold = ASName.create("TimesNewRoman,Bold");
	public static final ASName k_TimesNewRomanPS_Bold = ASName.create("TimesNewRomanPS-Bold");
	public static final ASName k_TimesNewRoman_Bold = ASName.create("TimesNewRoman-Bold");

	public static final ASName k_Times_Italic = ASName.create("Times-Italic");
	public static final ASName k_TimesNewRomanPS_ItalicMT = ASName.create("TimesNewRomanPS-ItalicMT");
	public static final ASName k_TimesNewRomanCommaItalic = ASName.create("TimesNewRoman,Italic");
	public static final ASName k_TimesNewRomanPS_Italic = ASName.create("TimesNewRomanPS-Italic");
	public static final ASName k_TimesNewRoman_Italic = ASName.create("TimesNewRoman-Italic");

	public static final ASName k_Times_BoldItalic = ASName.create("Times-BoldItalic");
	public static final ASName k_TimesNewRomanPS_BoldItalicMT = ASName.create("TimesNewRomanPS-BoldItalicMT");
	public static final ASName k_TimesNewRomanCommaBoldItalic = ASName.create("TimesNewRoman,BoldItalic");
	public static final ASName k_TimesNewRomanPS_BoldItalic = ASName.create("TimesNewRomanPS-BoldItalic");
	public static final ASName k_TimesNewRoman_BoldItalic = ASName.create("TimesNewRoman-BoldItalic");

	public static final ASName k_Helvetica = ASName.create("Helvetica");
	public static final ASName k_ArialMT = ASName.create("ArialMT");
	public static final ASName k_Arial = ASName.create("Arial");

	public static final ASName k_Helvetica_Bold = ASName.create("Helvetica-Bold");
	public static final ASName k_Arial_BoldMT = ASName.create("Arial-BoldMT");
	public static final ASName k_ArialCommaBold = ASName.create("Arial,Bold");
	public static final ASName k_Arial_Bold = ASName.create("Arial-Bold");
	public static final ASName k_HelveticaCommaBold = ASName.create("Helvetica,Bold");

	public static final ASName k_Helvetica_Oblique = ASName.create("Helvetica-Oblique");
	public static final ASName k_Arial_ItalicMT = ASName.create("Arial-ItalicMT");
	public static final ASName k_ArialCommaItalic = ASName.create("Arial,Italic");
	public static final ASName k_Arial_Italic = ASName.create("Arial-Italic");
	public static final ASName k_HelveticaCommaItalic = ASName.create("Helvetica,Italic");
	public static final ASName k_Helvetica_Italic = ASName.create("Helvetica-Italic");

	public static final ASName k_Helvetica_BoldOblique = ASName.create("Helvetica-BoldOblique");
	public static final ASName k_Arial_BoldItalicMT = ASName.create("Arial-BoldItalicMT");
	public static final ASName k_ArialCommaBoldItalic = ASName.create("Arial,BoldItalic");
	public static final ASName k_Arial_BoldItalic = ASName.create("Arial-BoldItalic");
	public static final ASName k_HelveticaCommaBoldItalic = ASName.create("Helvetica,BoldItalic");
	public static final ASName k_Helvetica_BoldItalic = ASName.create("Helvetica-BoldItalic");

	public static final ASName k_Courier = ASName.create("Courier");
	public static final ASName k_CourierNew = ASName.create("CourierNew");
	public static final ASName k_CourierNewPSMT = ASName.create("CourierNewPSMT");

	public static final ASName k_Courier_Bold = ASName.create("Courier-Bold");
	public static final ASName k_CourierNewCommaBold = ASName.create("CourierNew,Bold");
	public static final ASName k_CourierCommaBold = ASName.create("Courier,Bold");
	public static final ASName k_CourierNewPS_BoldMT = ASName.create("CourierNewPS-BoldMT");
	public static final ASName k_CourierNew_Bold = ASName.create("CourierNew-Bold");

	public static final ASName k_Courier_Oblique = ASName.create("Courier-Oblique");
	public static final ASName k_CourierNewCommaItalic = ASName.create("CourierNew,Italic");
	public static final ASName k_CourierCommaItalic = ASName.create("Courier,Italic");
	public static final ASName k_CourierNewPS_ItalicMT = ASName.create("CourierNewPS-ItalicMT");
	public static final ASName k_CourierNew_Italic = ASName.create("CourierNew-Italic");

	public static final ASName k_Courier_BoldOblique = ASName.create("Courier-BoldOblique");
	public static final ASName k_CourierNewCommaBoldItalic = ASName.create("CourierNew,BoldItalic");
	public static final ASName k_CourierCommaBoldItalic = ASName.create("Courier,BoldItalic");
	public static final ASName k_CourierNewPS_BoldItalicMT = ASName.create("CourierNewPS-BoldItalicMT");
	public static final ASName k_CourierNew_BoldItalic = ASName.create("CourierNew-BoldItalic");

	public static final ASName k_Symbol = ASName.create("Symbol");
	public static final ASName k_ZapfDingbats = ASName.create("ZapfDingbats");

	public static final ASName k_StandardEncoding = ASName.create("StandardEncoding");
	public static final ASName k_MacRomanEncoding = ASName.create("MacRomanEncoding");
	public static final ASName k_WinAnsiEncoding = ASName.create("WinAnsiEncoding");
	public static final ASName k_PDFDocEncoding = ASName.create("PDFDocEncoding");
	public static final ASName k_MacExpertEncoding = ASName.create("MacExpertEncoding");
	public static final ASName k_SymbolEncoding = ASName.create("SymbolEncoding");
	public static final ASName k_ZapfDingbatsEncoding = ASName.create("ZapfDingbatsEncoding");

	public static final ASName k_Identity_H = create("Identity-H");
	public static final ASName k_Identity_V = create("Identity-V");

	public static final ASName k_begincmap = ASName.create("begincmap");
	public static final ASName k_endcmap = ASName.create("endcmap");
	public static final ASName k_usecmap = ASName.create("usecmap");
	public static final ASName k_begincodespacerange = ASName.create("begincodespacerange");
	public static final ASName k_endcodespacerange = ASName.create("endcodespacerange");
	public static final ASName k_beginbfchar = ASName.create("beginbfchar");
	public static final ASName k_endbfchar = ASName.create("endbfchar");
	public static final ASName k_beginbfrange = ASName.create("beginbfrange");
	public static final ASName k_endbfrange = ASName.create("endbfrange");
	public static final ASName k_begincidchar = ASName.create("begincidchar");
	public static final ASName k_endcidchar = ASName.create("endcidchar");
	public static final ASName k_begincidrange = ASName.create("begincidrange");
	public static final ASName k_endcidrange = ASName.create("endcidrange");
	public static final ASName K_LLO = ASName.create("LLO");
	public static final ASName k_CST = ASName.create("CS");
	public static final ASName k_CSTYLE = ASName.create("CP");
	public static final ASName k_MEASURE = ASName.create("Measure");
	public static final ASName k_COFF = ASName.create("CO");
	public static final ASName k_SubType = ASName.create("SubType");
	public static final ASName k_Redact = ASName.create("Redact");
	
	public static final ASName k_BatesN = ASName.create("BatesN");

	public static final ASName k_UserProperties = create("UserProperties");
	public static final ASName k_OutputIntents = create("OutputIntents");
	public static final ASName k_SpiderInfo = create("SpiderInfo");
	public static final ASName k_DefaultGray = create("DefaultGray");
	public static final ASName k_DefaultRGB = create("DefaultRGB");
	public static final ASName k_DefaultCMYK = create("DefaultCMYK");
	public static final ASName k_Requirements = create("Requirements");
	public static final ASName k_Collection = create("Collection");
	public static final ASName k_RH = create("RH");
	public static final ASName k_PZ = create("PZ");
	public static final ASName k_Dur = create("Dur");
	public static final ASName k_Hid = create("Hid");
	public static final ASName k_SeparationInfo = create("SeparationInfo");
	public static final ASName k_Tabs = create("Tabs");
	public static final ASName k_TemplateInstantiated = create("TemplateInstantiated");
	public static final ASName k_PresSteps = create("PresSteps");
	public static final ASName k_Measure = create("Measure");
	public static final ASName k_VP = create("VP");
	public static final ASName k_Y = create("Y");
	public static final ASName k_CYX = create("CYX");
	public static final ASName k_SS = create("SS");
	public static final ASName k_ExData = create("ExData");
	public static final ASName k_Screen = create("Screen");
	public static final ASName k_3D = create("3D");
	public static final ASName k_Markup3D = create("Markup3D");
	public static final ASName k_AlternatePresentations = create("AlternatePresentations");
	public static final ASName k_EnableJavaScripts = ASName.create("EnableJavaScripts");
	public static final ASName k_NoOp = create("NoOp");
	public static final ASName k_ReqHandler = ASName.create("ReqHandler");
	public static final ASName k_Requirement = ASName.create("Requirement");
	public static final ASName k_Script = ASName.create("Script");
	public static final ASName k_StartResource = create("StartResource");
	public static final ASName k_Aspect = create("Aspect");
	public static final ASName k_CP = create("CP");
	public static final ASName k_Decode = create("Decode");
	public static final ASName k_HTP = create("HTP");
	public static final ASName k_FunctionType = create("FunctionType");
	public static final ASName k_Background = create("Background");
	public static final ASName k_AntiAlias = create("AntiAlias");
	public static final ASName k_HalftoneType = create("HalftoneType");
	public static final ASName k_HalftoneName = create("HalftoneName");
	public static final ASName k_SigFlags = create("SigFlags");
	public static final ASName k_DummyVar = create("DummyVar");
	public static final ASName k_Di = create("Di");
	public static final ASName k_Dm = create("Dm");
	public static final ASName k_PrintScaling = create("PrintScaling");
	public static final ASName k_Duplex = create("Duplex");
	public static final ASName k_PickTrayByPDFSize = create("PickTrayByPDFSize");
	public static final ASName k_PrintPageRange = create("PrintPageRange");
	public static final ASName k_NumCopies = create("NumCopies");
	public static final ASName k_Alt = create("Alt");
	public static final ASName k_ActualText = create("ActualText");
	public static final ASName k_DeviceColorant = create("DeviceColorant");
	public static final ASName k_RF = create("RF");
	public static final ASName k_UF = create("UF");
	public static final ASName k_Filters = create("Filters");
	public static final ASName k_DestOutputProfile = create("DestOutputProfile");
	public static final ASName k_OutputConditionIdentifier = create("OutputConditionIdentifier");
	public static final ASName k_RegistryName = create("RegistryName");
	public static final ASName k_OutputCondition = create("OutputCondition");
	public static final ASName k_PaintType = create("PaintType");
	public static final ASName k_TilingType = create("TilingType");
	public static final ASName k_XStep = create("XStep");
	public static final ASName k_YStep = create("YStep");
	public static final ASName k_DL = create("DL");
	public static final ASName k_StreamData = create("StreamData");
	public static final ASName k_JBIG2Globals = create("JBIG2Globals");
	public static final ASName k_ColorTransform = create("ColorTransform");
	public static final ASName k_Range = create("Range");
	public static final ASName k_BitsPerSample = create("BitsPerSample");
	public static final ASName k_Domain = create("Domain");
	public static final ASName k_Encode = create("Encode");
	public static final ASName k_C0 = create("C0");
	public static final ASName k_C1 = create("C1");
	public static final ASName k_Functions = create("Functions");
	public static final ASName k_Bounds = create("Bounds");
	public static final ASName k_ResFork = create("ResFork");
	public static final ASName k_Gamma = create("Gamma");
	public static final ASName k_WhitePoint = create("WhitePoint");
	public static final ASName k_BlackPoint = create("BlackPoint");
	public static final ASName k_Alternate = create("Alternate");
	public static final ASName k_Process = create("Process");
	public static final ASName k_MixingHints = create("MixingHints");
	public static final ASName k_Components = create("Components");
	public static final ASName k_Solidities = create("Solidities");
	public static final ASName k_DotGain = create("DotGain");
	public static final ASName k_PrintingOrder = create("PrintingOrder");
	public static final ASName k_Function = create("Function");
	public static final ASName k_Extend = create("Extend");
	public static final ASName k_Coords = create("Coords");
	public static final ASName k_BitsPerCoordinate = create("BitsPerCoordinate");
	public static final ASName k_BitsPerFlag = create("BitsPerFlag");
	public static final ASName k_VerticesPerRow = create("VerticesPerRow");
	public static final ASName k_SMaskInData = create("SMaskInData");
	public static final ASName k_Level1 = create("Level1");
	public static final ASName k_Panose = create("Panose");
	public static final ASName k_SpotFunction = create("SpotFunction");
	public static final ASName k_TransferFunction = create("TransferFunction");
	public static final ASName k_Frequency = create("Frequency");
	public static final ASName k_Angle = create("Angle");
	public static final ASName k_AccurateScreens = create("AccurateScreens");
	public static final ASName k_XSquare = create("XSquare");
	public static final ASName k_YSquare = create("YSquare");
	public static final ASName k_Width2 = create("Width2");
	public static final ASName k_Height2 = create("Height2");
	public static final ASName k_Matte = create("Matte");
	public static final ASName k_Luminosity = create("Luminosity");
	public static final ASName k_LL = create("LL");
	public static final ASName k_LLE = create("LLE");
	public static final ASName k_LLO = create("LLO");
	public static final ASName k_MN = create("MN");
	public static final ASName k_SeparationColorNames = create("SeparationColorNames");
	public static final ASName k_TrapRegions = create("TrapRegions");
	public static final ASName k_TrapStyles = create("TrapStyles");
	public static final ASName k_Bl = create("Bl");
	public static final ASName k_WP = create("WP");
	public static final ASName k_WS = create("WS");
	public static final ASName k_DC = create("DC");
	public static final ASName k_Base = create("Base");
	public static final ASName k_Volume = create("Volume");
	public static final ASName k_Mix = create("Mix");
	public static final ASName k_Repeat = create("Repeat");
	public static final ASName k_Synchronous = create("Synchronous");
	public static final ASName k_Operation = create("Operation");
	public static final ASName k_MR = create("MR");
	public static final ASName k_SR = create("SR");
	public static final ASName k_Z = create("Z");
	public static final ASName k_MH = create("MH");
	public static final ASName k_SP = create("SP");
	public static final ASName k_UC = create("UC");
	public static final ASName k_TT = create("TT");
	public static final ASName k_PL = create("PL");
	public static final ASName k_MCD = create("MCD");
	public static final ASName k_MCS = create("MCS");
	public static final ASName k_CT = create("CT");
	public static final ASName k_BU = create("BU");
	public static final ASName k_TF = create("TF");
	public static final ASName k_MU = create("MU");
	public static final ASName k_NU = create("NU");
	public static final ASName k_PID = create("PID");
	public static final ASName k_LI = create("LI");
	public static final ASName k_HI = create("HI");
	public static final ASName k_PreserveRB = create("PreserveRB");
	public static final ASName k_Field = create("Field");
	public static final ASName k_IF = create("IF");
	public static final ASName k_TP = create("TP");
	public static final ASName k_AC = create("AC");
	public static final ASName k_IX = create("IX");
	public static final ASName k_Reasons = create("Reasons");
	public static final ASName k_MDP = create("MDP");
	public static final ASName k_TimeStamp = create("TimeStamp");
	public static final ASName k_LegalAttestation = create("LegalAttestation");
	public static final ASName k_DigestMethod = create("DigestMethod");
	public static final ASName k_AddRevInfo = create("AddRevInfo");
	public static final ASName k_Issuer = create("Issuer");
	public static final ASName k_OID = create("OID");
	public static final ASName k_SubjectDN = create("SubjectDN");
	public static final ASName k_KeyUsage = create("KeyUsage");
	public static final ASName k_URLType = create("URLType");
	public static final ASName k_Changes = create("Changes");
	public static final ASName k_Prop_AuthTime = create("Prop_AuthTime");
	public static final ASName k_Prop_AuthType = create("Prop_AuthType");
	public static final ASName k_ByteRangeRef = create("ByteRangeRef");
	public static final ASName k_SW = create("SW");
	public static final ASName k_FB = create("FB");
	public static final ASName k_Start = create("Start");
	public static final ASName k_Duration = create("Duration");
	public static final ASName k_FWPosition = create("FWPosition");
	public static final ASName k_FWScale = create("FWScale");
	public static final ASName k_Mode = create("Mode");
	public static final ASName k_Rate = create("Rate");
	public static final ASName k_ShowControls = create("ShowControls");
	public static final ASName k_SI = create("SI");
	public static final ASName k_TS = create("TS");
	public static final ASName k_SPS = create("SPS");
	public static final ASName k_SIS = create("SIS");
	public static final ASName k_TID = create("TID");
	public static final ASName k_AU = create("AU");
	public static final ASName k_1_3 = create("1_3");
	public static final ASName k_2_0 = create("2_0");
	public static final ASName k_Color = create("Color");
	public static final ASName k_CropRect = create("CropRect");
	public static final ASName k_Overprint = create("Overprint");
	public static final ASName k_Tags = create("Tags");
	public static final ASName k_CropFixed = create("CropFixed");
	public static final ASName k_Position = create("Position");
	public static final ASName k_Resolution = create("Resolution");
	public static final ASName k_ColorType = create("ColorType");
	public static final ASName k_Tint = create("Tint");
	public static final ASName k_ImageType = create("ImageType");
	public static final ASName k_GrayMap = create("GrayMap");
	public static final ASName k_Inks = create("Inks");
	public static final ASName k_MainImage = create("MainImage");
	public static final ASName k_IncludedImageDimensions = create("IncludedImageDimensions");
	public static final ASName k_IncludedImageQuality = create("IncludedImageQuality");
	public static final ASName k_ColSpan = create("ColSpan");
	public static final ASName k_RowSpan = create("RowSpan");
	public static final ASName k_Headers = create("Headers");
	public static final ASName k_Scope = create("Scope");
	public static final ASName k_ListNumbering = create("ListNumbering");
	public static final ASName k_BaselineShift = create("BaselineShift");
	public static final ASName k_Placement = create("Placement");
	public static final ASName k_TextDecorationType = create("TextDecorationType");
	public static final ASName k_TextDecorationColor = create("TextDecorationColor");
	public static final ASName k_TextDecorationThickness = create("TextDecorationThickness");
	public static final ASName k_RubyAlign = create("RubyAlign");
	public static final ASName k_RubyPosition = create("RubyPosition");
	public static final ASName k_GlyphOrientationVertical = create("GlyphOrientationVertical");
	public static final ASName k_EndIndent = create("EndIndent");
	public static final ASName k_InlineAlign = create("InlineAlign");
	public static final ASName k_TBorderStyle = create("TBorderStyle");
	public static final ASName k_TPadding = create("TPadding");
	public static final ASName k_LineHeight = create("LineHeight");
	public static final ASName k_SpaceAfter = create("SpaceAfter");
	public static final ASName k_SpaceBefore = create("SpaceBefore");
	public static final ASName k_StartIndent = create("StartIndent");
	public static final ASName k_TextAlign = create("TextAlign");
	public static final ASName k_TextIndent = create("TextIndent");
	public static final ASName k_WritingMode = create("WritingMode");
	public static final ASName k_BlockAlign = create("BlockAlign");
	public static final ASName k_BackgroundColor = create("BackgroundColor");
	public static final ASName k_BorderColor = create("BorderColor");
	public static final ASName k_BorderStyle = create("BorderStyle");
	public static final ASName k_BorderThickness = create("BorderThickness");
	public static final ASName k_Padding = create("Padding");
	public static final ASName k_Attached = create("Attached");
	public static final ASName k_JavaScriptActions = create("JavaScriptActions");
	public static final ASName k_LaunchActions = create("LaunchActions");
	public static final ASName k_URIActions = create("URIActions");
	public static final ASName k_MovieActions = create("MovieActions");
	public static final ASName k_SoundActions = create("SoundActions");
	public static final ASName k_HideAnnotationActions = create("HideAnnotationActions");
	public static final ASName k_GotToRemote = create("GotToRemote");
	public static final ASName k_AlternateImages = create("AlternateImages");
	public static final ASName k_ExternalStreams = create("ExternalStreams");
	public static final ASName k_TrueTypeFonts = create("TrueTypeFonts");
	public static final ASName k_ExternalRefXobjects = create("ExternalRefXobjects");
	public static final ASName k_ExternalOPIdicts = create("ExternalOPIdicts");
	public static final ASName k_NonEmbeddedFonts = create("NonEmbeddedFonts");
	public static final ASName k_DevDepGS_OP = create("DevDepGS_OP");
	public static final ASName k_DevDepGS_HT = create("DevDepGS_HT");
	public static final ASName k_DevDepGS_TR = create("DevDepGS_TR");
	public static final ASName k_DevDepGS_UCR = create("DevDepGS_UCR");
	public static final ASName k_DevDepGS_BG = create("DevDepGS_BG");
	public static final ASName k_DevDepGS_FL = create("DevDepGS_FL");
	public static final ASName k_Annotations = create("Annotations");
	public static final ASName k_OptionalContent = create("OptionalContent");
	public static final ASName k_Attestation = create("Attestation");
	public static final ASName k_TransformMethod = create("TransformMethod");
	public static final ASName k_TransformParams = create("TransformParams");
	public static final ASName k_DigestLocation = create("DigestLocation");
	public static final ASName k_Document = create("Document");
	public static final ASName k_Signature = create("Signature");
	public static final ASName k_Msg = create("Msg");
	public static final ASName k_FormEx = create("FormEx");
	public static final ASName k_AnnotsEx = create("AnnotsEx");
	public static final ASName k_NA = create("NA");
	public static final ASName k_TA = create("TA");
	public static final ASName k_3DD = create("3DD");
	public static final ASName k_3DV = create("3DV");
	public static final ASName k_3DA = create("3DA");
	public static final ASName k_3DAnimationStyle = create("3DAnimationStyle");
	public static final ASName k_3DI = create("3DI");
	public static final ASName k_3DB = create("3DB");
	public static final ASName k_3DCrossSection = create("3DCrossSection");
	public static final ASName k_3DLightingScheme = create("3DLightingScheme");
	public static final ASName k_3DU = create("3DU");
	public static final ASName k_M3N = create("M3N");
	public static final ASName k_TSm = create("TSm");
	public static final ASName k_TSn = create("TSn");
	public static final ASName k_USm = create("USm");
	public static final ASName k_USn = create("USn");
	public static final ASName k_UU = create("UU");
	public static final ASName k_DSm = create("DSm");
	public static final ASName k_DSn = create("DSn");
	public static final ASName k_DU = create("DU");
	public static final ASName k_Window = create("Window");
	public static final ASName k_Transparent = create("Transparent");
	public static final ASName k_Embedded = create("Embedded");
	public static final ASName k_Windowed = create("Windowed");
	public static final ASName k_RichMediaWindow = create("RichMediaWindow");
	public static final ASName k_RichMediaPosition = create("RichMediaPosition");
	public static final ASName k_Max = create("Max");
	public static final ASName k_Min = create("Min");
	public static final ASName k_HAlign = create("HAlign");	
	public static final ASName k_VAlign = create("VAlign");	
	public static final ASName k_HOffset = create("HOffset");	
	public static final ASName k_VOffset = create("VOffset");	
	public static final ASName k_Instance = create("Instance");	
	public static final ASName k_RichMediaInstance = create("RichMediaInstance");	
	public static final ASName k_Asset = create("Asset");	
	public static final ASName k_RichMediaParams = create("RichMediaParams");	
	public static final ASName k_FlashVars = create("FlashVars");	
	public static final ASName k_Binding = create("Binding");	
	public static final ASName k_BindingMaterialName = create("BindingMaterialName");	
	public static final ASName k_CuePoints = create("CuePoints");	
	public static final ASName k_3DMeasure = create("3DMeasure");	
	public static final ASName k_3DRef = create("3DRef");
	public static final ASName k_Settings = create("Settings");	
	public static final ASName k_TRL = create("TRL");	
	public static final ASName k_MA = create("MA");	
	public static final ASName k_VA = create("VA");
	public static final ASName k_OnInstantiate = create("OnInstantiate");
	public static final ASName k_XN = create("XN");
	public static final ASName k_IN = create("IN");
	public static final ASName k_MS = create("MS");
	public static final ASName k_C2W = create("C2W");
	public static final ASName k_U3DPath = create("U3DPath");
	public static final ASName k_RM = create("RM");
	public static final ASName k_LS = create("LS");
	public static final ASName k_NR = create("NR");
	public static final ASName k_FOV = create("FOV");
	public static final ASName k_OB = create("OB");
	public static final ASName k_EA = create("EA");
	public static final ASName k_DIS = create("DIS");
	public static final ASName k_TB = create("TB");
	public static final ASName k_NP = create("NP");
	public static final ASName k_MD5 = create("MD5");
	public static final ASName k_FC = create("FC");
	public static final ASName k_CV = create("CV");
	public static final ASName k_IV = create("IV");
	public static final ASName k_Schema = create("Schema");
	public static final ASName k_Sort = create("Sort");
	public static final ASName k_Suspects = create("Suspects");
	public static final ASName k_SlideShow = create("SlideShow");
	public static final ASName k_CardBackground = create("CardBackground");
	public static final ASName k_CardBorder = create("CardBorder");
	public static final ASName k_PrimaryText = create("PrimaryText");
	public static final ASName k_SecondaryText = create("SecondaryText");
	public static final ASName k_ST = create("ST");
	public final static ASName k_Create	= create("Create");
	public final static ASName k_Delete	= create("Delete");
	public final static ASName k_Modify	= create("Modify");
	public final static ASName k_Copy = create("Copy");
	public final static ASName k_Import	= create("Import");
	public final static ASName k_Online	= create("Online");
	public final static ASName k_SummaryView = create("SummaryView");
	public final static ASName k_FullSave = create("FullSave");
	public final static ASName k_FillIn = create("FillIn");
	public final static ASName k_SubmitStandalone	= create("SubmitStandalone");
	public final static ASName k_SpawnTemplate	= create("SpawnTemplate");
	public final static ASName k_BarcodePlaintext	= create("BarcodePlaintext");
	public final static ASName k_Add = create("Add");
	public final static ASName k_LD3 = create("LD3");
	public final static ASName k_PD3 = create("PD3");
	public final static ASName k_AD3 = create("AD3");
	public final static ASName k_RD3 = create("RD3");
	public final static ASName k_3DC = create("3DC");
	public final static ASName k_3DNode = create("3DNode");
	public final static ASName k_Solid = create("Solid");
	public final static ASName k_SolidWireframe = create("SolidWireframe");
	public final static ASName k_TransparentWireframe = create("TransparentWireframe");
	public final static ASName k_BoundingBox = create("BoundingBox");
	public final static ASName k_TransparentBoundingBox = create("TransparentBoundingBox");
	public final static ASName k_TransparentBoundingBoxOutline = create("TransparentBoundingBoxOutline");
	public final static ASName k_Wireframe = 	create("Wireframe");
	public final static ASName k_ShadedWireframe = create("ShadedWireframe");
	public final static ASName k_HiddenWireframe = create("HiddenWireframe");
	public final static ASName k_ShadedVertices = create("ShadedVertices");
	public final static ASName k_Illustration = create("Illustration");
	public final static ASName k_SolidOutline = create("SolidOutline");
	public final static ASName k_ShadedIllustration = create("ShadedIllustration"); 	
	public final static ASName k_3DRenderMode =	create("3DRenderMode");
	public final static ASName k_3DBG =	create("3DBG");
	public final static ASName k_3DView = create("3DView");
	public final static ASName k_Absolute = create("Absolute");
	public final static ASName k_XNF = create("XNF");
	public final static ASName k_ANF = create("ANF");
	public final static ASName k_FWParams = create("FWParams");
	public final static ASName k_MediaClip = create("MediaClip");
	public final static ASName k_MediaCriteria = create("MediaCriteria");
	public final static ASName k_MediaDuration = create("MediaDuration");
	public final static ASName k_MediaOffset = create("MediaOffset");
	public final static ASName k_MediaPermissions = create("MediaPermissions");
	public final static ASName k_MediaPlayerInfo = create("MediaPlayerInfo");
	public final static ASName k_MediaPlayers = create("MediaPlayers");
	public final static ASName k_MediaPlayParams = create("MediaPlayParams");
	public final static ASName k_MediaScreenParams = create("MediaScreenParams");
	public final static ASName k_MinBitDepth = create("MinBitDepth");
	public final static ASName k_MinScreenSize = create("MinScreenSize");
	public final static ASName k_Once = create("Once");
	public final static ASName k_Palindrome = create("Palindrome");
	public final static ASName k_Flash = create("Flash");
	public final static ASName k_Video = create("Video");
	public final static ASName k_Near = create("Near");
	public final static ASName k_Center = create("Center");
	public final static ASName k_Far = create("Far");
	public final static ASName k_Raw = create("Raw");
	public final static ASName k_Signed = create("Signed");
	public final static ASName k_muLaw = create("muLaw");
	public final static ASName k_ALaw = create("ALaw"); 
	public final static ASName k_Timespan = create("Timespan");
	public final static ASName k_UseOutputIntent = create("UseOutputIntent");
	public final static ASName k_NoICCAlternates = create("NoICCAlternates");
	public final static ASName k_StrictSeparationColors = create("StrictSeparationColors");
	public final static ASName k_IgnoreDeviceGS = create("IgnoreDeviceGS");
	public final static ASName k_RequireOverprintPreview = create("RequireOverprintPreview");
	public final static ASName k_StrictJP2KColor = create("StrictJP2KColor");
	public final static ASName k_StrictAppearances = create("StrictAppearances");
	public final static ASName k_DisplayAnnotationContents = create("DisplayAnnotationContents");
	public final static ASName k_DisableNonGoToActions = create("DisableNonGoToActions");
	public final static ASName k_DisplayActionInfo = create("DisplayActionInfo");
	public final static ASName k_IgnoreDocInfo = create("IgnoreDocInfo");
	public final static ASName k_IgnoreThumbs = create("IgnoreThumbs");
	public final static ASName k_IgnoreTransitions = create("IgnoreTransitions");
	public final static ASName k_StrictOptionalContent = create("StrictOptionalContent");
	public final static ASName k_DisplayEmbeddedFiles = create("DisplayEmbeddedFiles");	
	public final static ASName k_OutputIntentChoice = create("OutputIntentChoice");
	public final static ASName k_GTS_PDFX = create("GTS_PDFX");
	public final static ASName k_GTS_PDFA1 = create("GTS_PDFA1");
	public final static ASName k_ISO_PDFE1 = create("ISO_PDFE1");
	public final static ASName k_PtData = create("PtData");
	public final static ASName k_Cloud = create("Cloud");
	public final static ASName k_XPTS = create("XPTS");
	public final static ASName k_PROJCS = create("PROJCS");
	public final static ASName k_EPSG = create("EPSG");
	public final static ASName k_WKT = create("WKT");
	public final static ASName k_GEOGCS = create("GEOGCS");
	public final static ASName k_GCS = create("GCS");
	public final static ASName k_DCS = create("DCS");
	public final static ASName k_PDU = create("PDU");
	public final static ASName k_GPTS = create("GPTS");
	public final static ASName k_LPTS = create("LPTS");
	public final static ASName k_GEO = create("GEO");
	public final static ASName k_LAT = create("LAT");
	public final static ASName k_LON = create("LON");
	public final static ASName k_ALT = create("ALT");
	public final static ASName k_DPartRoot = create("DPartRoot");
	public final static ASName k_DPartRootNode = create("DPartRootNode");
	public final static ASName k_RecordLevel = create("RecordLevel");
	public final static ASName k_NodeNameList = create("NodeNameList");
	public final static ASName k_DParts = create("DParts");
	public final static ASName k_DPM = create("DPM");
	public final static ASName k_AF = create("AF");
	public final static ASName k_DPart = create("DPart");
	public final static ASName k_Span = create("Span");
	public final static ASName k_T1_0 = create("/T1_0");
	public final static ASName k_T1_1 = create("/T1_1");
	public final static ASName k_T1_2 = create("/T1_2");
	public final static ASName k_T1_3 = create("/T1_3");
	public final static ASName k_TT0 = create("/TT0");
	public final static ASName k_TT1 = create("/TT1");
	public final static ASName k_TT2 = create("/TT2");;
	public final static ASName k_F4 = create("/F4");
	public final static ASName k_F5 = create("/F5");
	public final static ASName k_F6 = create("/F6");
	public final static ASName k_F7 = create("/F7");
	public final static ASName k_End = create("End");
	public final static ASName k_AFRelationship = create("AFRelationship");
	public final static ASName k_Source = create("Source");
	public final static ASName k_Conformance = create("Conformance");
	public final static ASName k_ModificationDate = create("ModificationDate");
	public final static ASName k_JobID = create("JobID");
	public final static ASName k_CIP4Root = create("CIP4:Root");
	public final static ASName k_CuePoint = create("CuePoint");
	public final static ASName k_Navigation = create("Navigation");
	public final static ASName k_Time = create("Time");
	public final static ASName k_Unspecified = create("Unspecified");
	public final static ASName k_DestOutputProfileRef = create("DestOutputProfileRef");
	public final static ASName k_ICCVersion = create("ICCVersion");
	public final static ASName k_ProfileCS = create("ProfileCS");
	public final static ASName k_ProfileName = create("ProfileName");
	public final static ASName k_ColorantTable = create("ColorantTable");
	public final static ASName k_GoToDp = create("GoToDp");
	public final static ASName k_Dp = create("Dp");
	public final static ASName k_SignaturePolicyOID = create("SignaturePolicyOID");
	public final static ASName k_SignaturePolicyHashValue = create("SignaturePolicyHashValue");
	public final static ASName k_SignaturePolicyHashAlgorithm = create("SignaturePolicyHashAlgorithm");
	public final static ASName k_SignaturePolicyCommitmentType = create("SignaturePolicyCommitmentType");
	public final static ASName k_Penalty = create("Penalty");
	public final static ASName k_OCInteract = create("OCInteract");
	public final static ASName k_OCAutoStates = create("OCAutoStates");		
	public final static ASName k_AcroFormInteract = create("AcroFormInteract");		
	public final static ASName k_XFARender = create("XFARender");		
	public final static ASName k_XFAInteract = create("XFAInteract");		
	public final static ASName k_Markup = create("Markup");		
	public final static ASName k_3DMarkup = create("3DMarkup");		
	public final static ASName k_Multimedia = create("Multimedia");		
	public final static ASName k_Attachment = create("Attachment");		
	public final static ASName k_U3D = create("U3D");		
	public final static ASName k_PRC = create("PRC");		
	public final static ASName k_AttachmentEditing = create("AttachmentEditing");		
	public final static ASName k_CollectionEditing = create("CollectionEditing");		
	public final static ASName k_DigSigValidation = create("DigSigValidation");		
	public final static ASName k_DigSig = create("DigSig");		
	public final static ASName k_DigSigMDP = create("DigSigMDP");		
	public final static ASName k_Barcode = create("Barcode");		
	public final static ASName k_Geospatial2D = create("Geospatial2D");		
	public final static ASName k_Geospatial3D = create("Geospatial3D");		
	public final static ASName k_DPartInteract = create("DPartInteract");		
	public final static ASName k_SeparationSimulation = create("SeparationSimulation");		
	public final static ASName k_StrictAnnotations = create("StrictAnnotations");		
	public final static ASName k_StrictActions = create("StrictActions");		
	public final static ASName k_Transitions = create("Transitions");		
	public final static ASName k_Encryption = create("Encryption");	
	
	static 
	{
		TRIE.add(ASName.k_BDC);
		TRIE.add(ASName.k_EMC);
		TRIE.add(ASName.k_Tw);
		TRIE.add(ASName.k_Tm);
		TRIE.add(ASName.k_Tf);
		TRIE.add(ASName.k_Tc);
		TRIE.add(ASName.k_ET);
		TRIE.add(ASName.k_TD);
		TRIE.add(ASName.k_TJ);
		TRIE.add(ASName.k_BT);
		TRIE.add(ASName.k_Tj);
		TRIE.add(ASName.k_Td);
		TRIE.add(ASName.k_c);
		TRIE.add(ASName.k_l);
		TRIE.add(ASName.k_m);
		TRIE.add(ASName.k_SCN);
		TRIE.add(ASName.k_MCID);
		TRIE.add(ASName.k_M);
		TRIE.add(ASName.k_Q);
		TRIE.add(ASName.k_S);
		TRIE.add(ASName.k_f);
		TRIE.add(ASName.k_g);
		TRIE.add(ASName.k_h);
		TRIE.add(ASName.k_w);
		TRIE.add(ASName.k_q);
		TRIE.add(ASName.create("Span"));
		TRIE.add(ASName.k_Type);
		TRIE.add(ASName.k_T1_0);
		TRIE.add(ASName.k_T1_1);
		TRIE.add(ASName.k_T1_2);
		TRIE.add(ASName.k_T1_3);
		TRIE.add(ASName.k_TT1);
		TRIE.add(ASName.k_TT2);
		TRIE.add(ASName.k_TT0);
		TRIE.add(ASName.k_scn);
		TRIE.add(ASName.k_F4);
		TRIE.add(ASName.k_F5);
		TRIE.add(ASName.k_F6);
		TRIE.add(ASName.k_F7);
		TRIE.add(ASName.k_BStar);
		TRIE.add(ASName.k_TStar);
		TRIE.add(ASName.k_gs);
		TRIE.add(ASName.k_cs);
		TRIE.add(ASName.k_cm);
		TRIE.add(ASName.k_re);
	}
	
}
