/***************************************************************************/
/*                                                                         */
/*                      ADOBE CONFIDENTIAL                                 */
/*                      _ _ _ _ _ _ _ _ _ _                                */
/*                                                                         */
/*  Copyright 2001-2002, Adobe Systems Incorporated                        */
/*  All Rights Reserved.                                                   */
/*                                                                         */
/*  NOTICE: All information contained herein is, and remains the property  */
/*  of Adobe Systems Incorporated and its suppliers, if any.  The          */
/*  intellectual and technical concepts contained herein are proprietary   */
/*  to Adobe Systems Incorporated and its suppliers and may be covered by  */
/*  U.S. and Foreign Patents, patents in process, and are protected by     */
/*  trade secret or copyright law.  Dissemination of this information or   */
/*  reproduction of this material is strictly forbidden unless prior       */
/*  written permission is obtained from Adobe Systems Incorporated.        */
/*                                                                         */
/***************************************************************************/
package com.adobe.versioncue.nativecomm.msg;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;

/**
 * List of NCType objects.
 * 
 * Note that this class is **NOT** Thread-safe in its current form.
 * 
 * @author <a href="mailto:tnaroska@adobe.com">Timo Naroska</a>
 * @version $Revision: #1 $
 */
public final class NCList extends NCType implements Iterable<NCType>
{
	// --------------------------------------------------------------------------- Private Variables

	/** underlying list object */
	private final ArrayList<NCType> list;

	// -------------------------------------------------------------------------- Public Constructor

	/** Constructs a new <code>NCList</code> object. */
	public NCList()
	{
		this(10);
	}

	/** Constructs a new <code>NCList</code> object.
	 * @param initialCapacity - initial capacity of the created Array object
	 */
	public NCList(int initialCapacity)
	{
		list = new ArrayList<NCType>(initialCapacity);
	}

	// ------------------------------------------------------------------------------ Public Methods

	/**
	 * Returns the size of this <code>NCList</code> object.
	 * 
	 * @return the size of this <code>NCList</code> object
	 */
	public int size()
	{
		return list.size();
	}

	/**
	 * Returns true if this <code>NCList</code> object contains no items.
	 * 
	 * @return true if this <code>NCList</code> object contains no items
	 */
	public boolean isEmtpy()
	{
		return list.isEmpty();
	}

	/**
	 * Returns the <code>NCType</code> located at the specified index inside
	 * this NCList.
	 * 
	 * @param index the index of the NCType to return
	 * @return the <code>NCType</code> located at the specified index inside
	 *         this NCList
	 */
	public NCType get(int index)
	{
		return list.get(index);
	}

	/**
	 * Returns the elements of this NCList instance as <code>NCType[]</code>.
	 * 
	 * @return elements of this NCList instance as <code>NCType[]</code>.
	 */
	public NCType[] get()
	{
		return list.toArray(new NCType[list.size()]);
	}

	/**
	 * Returns true if this Array contains the specified NCType.
	 * 
	 * @param val - NCType to look for
	 * @return true if this Array contains the specified NCType
	 */
	public boolean contains(NCType val)
	{
		return list.contains(val);
	}

	/**
	 * Appends the specified NCType to the end of this NCList.
	 * 
	 * @param value - NCType to be added
	 * @return the NCList
	 */
	public NCList add(NCType value)
	{
		// == PRE-CONDITION ========================================================================
		if (value == null)
			throw new NullPointerException("value can't be null");
		// == PRE-CONDITION ========================================================================

		list.add(value);
		return this;
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(INCExternalizable value)
	{
		return add(value.externalize());
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(String value)
	{
		return add(new NCString(value));
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(boolean value)
	{
		return add(value ? NCBool.TRUE : NCBool.FALSE);
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(int value)
	{
		return add(new NCInt(value));
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(long value)
	{
		return add(new NCLong(value));
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(double value)
	{
		return add(new NCDouble(value));
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(Date value)
	{
		return add(new NCDate(value));
    }

	/** Appends a value to the NCList.
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(byte[] value)
	{
		return add(new NCData(value));
	}

	/**
	 * Appends a value to the NCList.
	 * 
	 * @param value the value
	 * @return the NCList
	 */
	public NCList add(ByteBuffer value)
	{
		return add(new NCData(value));
	}

	/**
	 * Returns the String value at the specified index.
	 * 
	 * @param index the index
	 * @return String value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public String getString(int index) throws BadMessageException
	{
		return ((NCString) checkedGet(index, NCString.class)).string();
	}

	/**
	 * Returns the boolean value at the specified index.
	 * 
	 * @param index the index
	 * @return boolean value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public boolean getBool(int index) throws BadMessageException
	{
		return ((NCBool) checkedGet(index, NCBool.class)).booleanValue();
	}

	/**
	 * Returns the double value at the specified index.
	 * 
	 * @param index the index
	 * @return double value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public double getDouble(int index) throws BadMessageException
	{
		return ((NCNumber) checkedGet(index, NCNumber.class)).doubleValue();
	}

	/**
	 * Returns the int value at the specified index.
	 * 
	 * @param index the index
	 * @return int value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public int getInt(int index) throws BadMessageException
	{
		return ((NCNumber) checkedGet(index, NCNumber.class)).intValue();
	}

	/**
	 * Returns the long value at the specified index.
	 * 
	 * @param index the index
	 * @return long value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public long getLong(int index) throws BadMessageException
	{
		return ((NCNumber) checkedGet(index, NCNumber.class)).longValue();
	}

 	/** Returns the date value at the specified index.
	 * @param index the index
	 * @return date value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public Date getDate(int index) throws BadMessageException
	{
		return ((NCDate) checkedGet(index, NCDate.class)).date();
	}

 	/** Returns the byte[] value at the specified index.
	 * @param index the index
	 * @return byte[] value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public byte[] getByteArray(int index) throws BadMessageException
	{
		return ((NCData) checkedGet(index, NCData.class)).byteArray();
	}

	/**
	 * Returns the ByteBuffer value at the specified index.
	 * 
	 * @param index the index
	 * @return ByteBuffer value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public ByteBuffer getBytes(int index) throws BadMessageException
	{
		return ((NCData) checkedGet(index, NCData.class)).bytes();
	}

	/**
	 * Returns the NCList value at the specified index.
	 * 
	 * @param index the index
	 * @return NCList value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public NCList getList(int index) throws BadMessageException
	{
		return (NCList) checkedGet(index, NCList.class);
	}

	/**
	 * Returns the NCMap value at the specified index.
	 * 
	 * @param index the index
	 * @return NCMap value
	 * @throws BadMessageException if the requested value is of a bad type
	 */
	public NCMap getMap(int index) throws BadMessageException
	{
		return (NCMap) checkedGet(index, NCMap.class);
	}

	/**
	 * Replaces the <code>NCType</code> object at the specified index inside the
	 * NCList.
	 * 
	 * @param index of element to replace.
	 * @param value - NCType to be stored at the specified position.
	 * @return the NCList
	 */
	public NCList set(NCType value, int index)
	{
		// == PRE-CONDITION ========================================================================
		if (value == null)
			throw new NullPointerException("value can't be null");
		// == PRE-CONDITION ========================================================================

		list.set(index, value);
		return this;
	}

	/**
	 * Removes all Values from this NCList.
	 * 
	 * @return the NCList
	 */
	public NCList clear()
	{
		list.clear();
		return this;
	}

	/**
	 * Removes the NCType at the specified position in this NCList and returns
	 * it. Shifts any subsequent values to the left (subtracts one from their
	 * indices).
	 * 
	 * @param index the index of the element to removed
	 * @return the removed NCType object
	 */
	public NCType remove(int index)
	{
		return list.remove(index);
	}

	/** @see java.lang.Iterable#iterator() */
	public Iterator<NCType> iterator()
	{
		return list.iterator();
	}

	// ---------------------------------------------------------------------------- NCType Overrides

	@Override
	public int getType()
	{
		return TYPE_LIST;
	}

	// ------------------------------------------------------------------ java.lang.Object Overrides

	@Override
	public int hashCode()
	{
		return list.hashCode();
	}

	@Override
	public boolean equals(Object other)
	{
		if (other != null && other instanceof NCList)
		{
			NCList otherArray = (NCList) other;
			return list.equals(otherArray.list);
		}
		return false;
	}

	@Override
	public String toString()
	{
		final String nl = System.getProperty("line.separator");

		final StringBuilder buf = new StringBuilder(64);

		buf.append(this.getClass().getSimpleName()).append(nl).append('{').append(nl);
		for (Iterator<NCType> it = list.iterator(); it.hasNext();)
		{
			buf.append("    ");
			buf.append(it.next().toString());
			buf.append(nl);
		}
		buf.append('}');
		return buf.toString();
	}

	// ----------------------------------------------------------------------------- Private Methods

	/**
	 * Checks the type of and returns the value at index.
	 * 
	 * @param index
	 *            the index
	 * @param expectedType
	 *            expected value type
	 * @return checked value
	 * @throws BadMessageException
	 *             if a value is of a bad type
	 */
	private NCType checkedGet(int index, Class<?> expectedType) throws BadMessageException
	{
		NCType value = get(index);
		if (!expectedType.isInstance(value))
		{
			throw new BadMessageException("Expected <" + expectedType + "> at [" + index +
					"]. Actual value: " + value);
		}
		return value;
	}
}