/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2008 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;


/**
 * This class is used to correlate schema strings to their eTag (int) values.
 * The lookup mechanism is unique in that we want to avoid string comparisons at all costs.
 * Lookups are based on the fact that all Strings are interned and we match by string identity.
 * <br />
 * The lookup algorithm is a simple fixed-size hash table. The table size is always a
 * power of 2, which is typically not ideal for linear probing, but using a prime number
 * (and the associated modulus operation) would make the lookup significantly more expensive.
 *
 * @exclude from published api
 */
public class SchemaStrings {
	
	private final String[] keys;
	private final int[] values;
	private int size;	// # elements in the map
	private final int hashMask;

	
	/**
	 * Constructor that allocates our internal data structures to 
	 * the specified size.
	 * @param size - the size of our internal arrays. (They can't grow!)
	 */
	public SchemaStrings(int size) {
		final int capacity = calculateTableSize(size);
		keys = new String[capacity];
		values = new int[capacity];
		hashMask = capacity - 1;
	}
	
	private static int calculateTableSize(int capacity) {
		assert capacity > 0;
		
		// Linear probing goes bad when the load factor is > 80%
		capacity = capacity * 10 / 8;
		
		int n = 16;
		while (capacity > n)
			n *= 2;
		
		return n;
	}
	
	/**
	 * This method populates the data structures.
	 * It assumes that the array does not need to be resized.
	 * @param sStr the String to add.
	 * @param eTag the corresponding eTag
	 */
	public void put(String sStr, int eTag) {
		final int capacity = keys.length;
		// There must always be a free entry to terminate probing
		assert size < capacity - 1;
		
		int position = sStr.hashCode() & hashMask;		
		
		while (true) {
			// It is an error to add a key twice
			assert !sStr.equals(keys[position]);
			
			if (keys[position] == null) {
				keys[position] = sStr;
				values[position] = eTag;
				
				size++;
				
				return;
			}
			
			position++;
			if (position == capacity)
				position = 0;
		}
	}
	
	/**
	 * Find the tag integer matching this schema string
	 * @param sStr the schema string to look up.
	 * @return the corresponding tag. -1 if not found.
	 */
	public int getInt(String sStr) {
		final int capacity = keys.length;
		int position = sStr.hashCode() & hashMask;
		
		while (true) {
			final String key = keys[position];			
			
			if (key == null)
				return -1;	// not found
			
			if (key.equals(sStr))
				return values[position];
			
			position++;
			if (position == capacity)
				position = 0;
		}
	}
}