/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2007 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.formcalc;


import com.adobe.xfa.Obj;
import com.adobe.xfa.ut.StringUtils;


/**
 * This class defines static methods to implement
 * the FormCalc miscellaneous calculations.
 *
 *  M I S C E L L A N E O U S   F U N C T I O N S
 *      eval, messagebox, null, ref.
 *
 * @author Mike P. Tardif
 *
 * @exclude from published api.
 */
final class BuiltinMisc {


	/*
	 *  Disallow instances of this class.
	 */
	private BuiltinMisc() {
	}

	/*
	 *  Eval
	 *      This function parses and evaluates a given FormCalc expression.
	 *  
	 */
	static void Eval(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null; 
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 1);
			Builtins.maxArgs(nArgs, 1);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			//
			// retrieve the given FormCalc expression.
			//
			String sVal = oParser.getString(oArgSym[0]);
			if (StringUtils.isEmpty(sVal))
				throw new CalcException();
			//
			// create new parser instance and
			// initialize scripthost and FormCalc interpreter.
			//
			CalcParser evalParser = (CalcParser) oParser.clone();    
			if (evalParser != null) {
				//
				// evaluate given expression.
				//
				evalParser.putScript(sVal);
				boolean bRet = evalParser.yyParse();
				if (bRet)
					evalParser.execute();
				CalcSymbol oResult = evalParser.getCalcResult();
				oRetSym = new CalcSymbol(oResult);
				CalcSymbol.delete(oResult, oParser);
			}
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}


	/*
	 *  Mbox
	 *      This function display a message box
	 *      and returns the user-selected command button.
	 */
	static void MessageBox(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null; 
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 2);
			Builtins.maxArgs(nArgs, 3);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			Builtins.limitNullArgs(oParser, nArgs, oArgSym);
			//
			// retrieve the given arguments.
			//
			int n = (int) oParser.getNumeric(oArgSym[0]);
			if (n < 0)
				n = 0;
			if (n > 5)
				n = 0;
			String message = oParser.getString(oArgSym[1]);
			String title = (nArgs > 2) ? oParser.getString(oArgSym[2]) : "";
			//
			// If display host is available Then display message box
			// and get selected command button number.
			//
			int nRetVal = 0;
			if (oParser.moDisplayHost != null) {
				nRetVal = oParser.moDisplayHost.messageBox(n, message, title);
			}
			oRetSym = new CalcSymbol(nRetVal);
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}


	/*
	 *  Null  
	 *      This function returns the null value.
	 */
	static void Null(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null; 
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.maxArgs(nArgs, 0);
			//
			// generate the null value.
			//
			oRetSym = new CalcSymbol();
			oRetSym.setType(CalcSymbol.TypeNull);
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}


	/*
	 *  Ref  
	 *      This function returns the null value.
	 */
	static void Ref(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null; 
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 1);
			Builtins.maxArgs(nArgs, 1);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			//
			// generate the null value.
			//
			int fType = oArgSym[0].getType();
			Obj oObj = null;
			switch (fType)  {
			case CalcSymbol.TypeAccessor:
				oObj = oParser.moScriptHost.getItem(
							oArgSym[0].getName(), oArgSym[0].getObjValues());
				oRetSym = new CalcSymbol(oObj, oArgSym[0].getName());
				break;
			case CalcSymbol.TypeNull:
				oRetSym = new CalcSymbol();
				oRetSym.setObjValue(oObj);
				oRetSym.setType(CalcSymbol.TypeReference);
				break;
			case CalcSymbol.TypeReference:
			case CalcSymbol.TypeError:
			case CalcSymbol.TypeReturn:
				oRetSym = new CalcSymbol(oArgSym[0]);
				break;
			case CalcSymbol.TypeString:
			case CalcSymbol.TypeDouble:
			case CalcSymbol.TypeVariable:
				oRetSym = new CalcSymbol(oParser.getString(oArgSym[0]));
				break;
			default:
				assert(fType != 0);
				break;
			}
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}
}