package com.adobe.xfa.text;

import com.adobe.xfa.gfx.GFXColour;

/**
 * Class TextSelection represents a range of selected text.  As such,
 * it extends class TextRange withy selection colour information.  One
 * can pass a text selection object to the text drawing method to render
 * text with selection highlighting.  Note that a multi-view application
 * would maintain a separate selection for each view.
 * @exclude from published api.
 */

public class TextSelection extends TextRange {
	private GFXColour moColour;
	private GFXColour moColourBg;

/**
 * Default constructor.
 * <p>
 * Construct a text selection object with no initial stream association.
 * The default foreground colour is white and the default background
 * colour is black.
 */
	public TextSelection () {
		moColour = GFXColour.WHITE;
		moColourBg = GFXColour.BLACK;
	}

/**
 * Copy constructor.
 * <p>
 * Copy the source selection object including range and colours.
 * @param oSource - Source selection to copy.
 */
	public TextSelection (TextSelection oSource) {
		super (oSource);
		moColour = oSource.moColour;
		moColourBg = oSource.moColourBg;
	}

/**
 * Constructor with colours and stream.
 * <p>
 * Construct a text selection object associated with the given stream
 * and using the given colours.  The selection range initially covers
 * the entire text stream.
 * @param oColour - Foreground colour.
 * @param oColourBg - Background colour.
 * @param poStream - (optional) Text stream for association.  NULL
 * (default) creates the text selection with no initial association.
 */
	public TextSelection (GFXColour oColour, GFXColour oColourBg, TextStream poStream) {
		super (poStream);
		moColour = oColour;
		moColourBg = oColourBg;
	}

/**
 * Obtain the current foreground colour of the selection.
 * @return Foreground colour.
 */
	public GFXColour colour () {
		return moColour;
	}

/**
 * Change the foreground colour of the selection.
 * <p>
 * Note: changing the colour does not automatically cause an
 * invalidation of the selected text in any view.
 * @param oNewColour - New foreground colour.
 */
	public void colour (GFXColour oNewColour) {
		moColour = oNewColour;
	}

/**
 * Obtain the current background colour of the selection.
 * @return Background colour.
 */
	public GFXColour colourBg () {
		return moColourBg;
	}

/**
 * Change the background colour of the selection.
 * <p>
 * Note: changing the colour does not automatically cause an
 * invalidation of the selected text in any view.
 * @param oNewColourBg - New background colour.
 */
	public void colourBg (GFXColour oNewColourBg) {
		moColourBg = oNewColourBg;
	}

/**
 * Assignment operator.
 * <p>
 * Copies all attrbutes of the source selection, including underlying
 * text range and colours.
 * @param oSource - Source selection to copy.
 */
	public void copyFrom (TextSelection oSource) {
		super.copyFrom (oSource);
		moColour = oSource.moColour;
		moColourBg = oSource.moColourBg;
	}

/**
 * Assignment operator, from base text range class.
 * <p>
 * Copies the given range (stream association and start/end).  Does not
 * change this object's colours.
 * @param oSource - Source text range to copy.
 */
// Javaport: not needed if method merely calls super.
//	public void copyFrom (TextRange oSource) {
//		super.copyFrom (oSource);
//	}

/**
 * Equality comparison.
 * <p>
 * Two selections are considered equal of the represent the same range
 * in the same text stream, and have the same foreground and background
 * colours.
 * @param object the object to compare against.
 * @return TRUE if the text selections are equal; FALSE if not.
 */
	public boolean equals (Object object) {
		
		if (this == object)
			return true;
		
		if (!super.equals(object))
			return false;
		
		if (! (object instanceof TextSelection))
			return false;
		
		TextSelection other = (TextSelection)object;
		return moColour.equals(other.moColour) && 
			   moColourBg.equals(other.moColourBg);
	}

	public int hashCode() {
		int hash = 83;
		hash = (hash * 31) ^ super.hashCode();
		hash = (hash * 31) ^ moColour.hashCode();
		hash = (hash * 31) ^ moColourBg.hashCode();
		return hash;
	}

}
