/*
 *	ADOBE CONFIDENTIAL
 *
 *	Copyright 1994 Adobe Systems Incorporated.  All Rights Reserved.
 *
 *	NOTICE:  All information contained herein is, and remains
 *	the property of Adobe Systems Incorporated and its
 *	suppliers, if any.	The intellectual and technical
 *	concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S.
 *	and Foreign Patents, patents in process, and are protected
 *	by trade secret or copyright law.  Dissemination of this
 *	information or reproduction of this material is strictly
 *	forbidden unless prior written permission is obtained from
 *	Adobe Systems Incorporated.
 *
 *	@author Richard Devitt
 */

package com.adobe.xfa.ut;

/**
 * A class to describe a coordinate (in Euclidean space).
 * It consists of a pair of (x, y) values.
 * <p>
 * Instances of this class are immutable.  All change operations
 * return a new instance of this <code>CoordPair</code> class.
 */
public final class CoordPair {

	/**
	 * The <code>CoordPair</code> corresponding to the (0, 0) origin.
	 * 
	 * @exclude from published api.
	 */
	public static final CoordPair ZERO_ZERO = new CoordPair();

	private final UnitSpan mX;
	private final UnitSpan mY;

	/**
	 * Instantiates a <code>CoordPair</code> with the coordinate (0, 0).
	 * 
	 * @see #zeroZero() 
	 */
	public CoordPair() {
		mX = UnitSpan.ZERO;
		mY = UnitSpan.ZERO;
	}

	/**
	 * Instantiates a <code>CoordPair</code> from the given <code>CoordPair</code>.
	 * 
	 * @param source 
	 *            the <code>CoordPair</code> to copy to this object.
	 * @deprecated CoordPair is immutable, so there is no need to copy an instance.
	 */
	public CoordPair(CoordPair source) {
		mX = source.x();
		mY = source.y();
	}

	/**
	 * Instantiates a <code>CoordPair</code> from the given <code>UnitSpan</code>(s).
	 * 
	 * @param x 
	 *            the x coordinate.
	 * @param y 
	 *            the y coordinate.
	 */
	public CoordPair(UnitSpan x, UnitSpan y) {
		mX = x;
		mY = y;
	}

	/**
	 * Returns a <code>CoordPair</code> representing this
	 * <code>CoordPair</code> snaped to the nearest
	 * grid coordinates of the given <code>CoordPair</code>.
	 * 
	 * @param grid 
	 *            the grid coordinate
	 * @return
	 *            a coordinate aligned to the grid.
	 * @see UnitSpan#grid(UnitSpan) 
	 */
	public CoordPair grid(CoordPair grid) {
		return new CoordPair(mX.grid(grid.x()), mY.grid(grid.y()));
	}

	/**
	 * Returns a <code>CoordPair</code> representing the
	 * rounding of this object's coordinates to the given <code>CoordPair</code>.
	 * 
	 * @param round 
	 *            the rounding coordinate.
	 * @return
	 *            a coordinate of the rounding.
	 * @see UnitSpan#round(UnitSpan) 
	 */
	public CoordPair round(CoordPair round) {
		return new CoordPair(mX.round(round.x()), mY.round(round.y()));
	}

	/**
	 * Gets this object's x coordinate.
	 *
	 * @return
	 *            the x coordinate.
	 */
	public UnitSpan x() {
		return mX;
	}

// Javaport: commented out for immutability.
//	public void x(UnitSpan oNewX) {
//		moX = oNewX;
//	}

	/**
	 * Gets this object's y coordinate.
	 *
	 * @return
	 *            the y coordinate.
	 */
	public UnitSpan y() {
		return mY;
	}

// Javaport: commented out for immutability.
//	public void y(UnitSpan oNewY) {
//		moY = oNewY;
//	}

	/**
	 * Returns a <code>CoordPair</code> representing the
	 * rotation of this object through the given <code>Angle</code>
	 * and about the given <code>CoordPair</code>.
	 *
	 * @param oRPoint
	 *            the point of rotation.
	 * @param oAngle
	 *            the angle of rotation.
	 * @return
	 *            a coordinate of the rotation.
	 *
	 * @exclude from published api.
	 */
	public CoordPair rotatePoint(CoordPair oRPoint, Angle oAngle) {
		int degrees = oAngle.degrees();
		if (degrees == 0) {
			return this;
		}

		UnitSpan oTX = new UnitSpan(UnitSpan.INCHES_72K, mX.units(), mX.value());
		UnitSpan oTY = new UnitSpan(UnitSpan.INCHES_72K, mY.units(), mY.value());
		UnitSpan oRPX = new UnitSpan(UnitSpan.INCHES_72K, oRPoint.x().units(), oRPoint.x().value());
		UnitSpan oRPY = new UnitSpan(UnitSpan.INCHES_72K, oRPoint.y().units(), oRPoint.y().value());

		int lX = oTX.value();
		int lY = - (oTY.value());
		int lXR = oRPX.value();
		int lYR = - (oRPY.value());

		switch (degrees) {
		case 90:
			oTX = new UnitSpan(UnitSpan.INCHES_72K, lXR - (lY - lYR));
			oTY = new UnitSpan(UnitSpan.INCHES_72K, -(lYR + (lX - lXR)));
			break;

		case 180:
			oTX = new UnitSpan(UnitSpan.INCHES_72K, lXR - (lX - lXR));
			oTY = new UnitSpan(UnitSpan.INCHES_72K, -(lYR - (lY - lYR)));
			break;

		case 270:
			oTX = new UnitSpan(UnitSpan.INCHES_72K, lXR + (lY - lYR));
			oTY = new UnitSpan(UnitSpan.INCHES_72K, -(lYR - (lX - lXR)));
			break;

		default:
			double dRadians = oAngle.degrees() * (Math.PI / 180.0);
			double dCosAngle = Math.cos(dRadians);
			double dSinAngle = Math.sin(dRadians);
			oTX = new UnitSpan(UnitSpan.INCHES_72K, (int) Math.round (lXR + (lX - lXR) * dCosAngle - (lY - lYR) * dSinAngle));
			oTY = new UnitSpan(UnitSpan.INCHES_72K, (int) -Math.round (lYR + (lY - lYR) * dCosAngle + (lX - lXR) * dSinAngle));
			break;
		}
		return new CoordPair(oTX, oTY);
	}

	/**
	 * Determines if this object is equal to the given <code>Object</code>.
	 * Comparisons with instances of non-<code>CoordPair</code> objects are never equal. 
	 * When comparing instances of <code>CoordPair</code> objects,
	 * the object's <code>UnitSpan</code> coordinates
	 * are each compared for equality.
	 * 
	 * @param object 
	 *            the <code>Object</code> to compare.
	 * @return
	 *            true if equal, false otherwise.
	 * @see UnitSpan#equals(Object)
	 */
	public boolean equals(Object object) {
		
		if (this == object)
			return true;
		
		// This overrides Object.equals(boolean) directly, so...
		if (object == null)
			return false;
		
		if (object.getClass() != getClass())
			return false;
		
		CoordPair cmp = (CoordPair) object;
		return mX.equals(cmp.mX) && mY.equals(cmp.mY);
	}

	/** 
	 * @exclude from published api.
	 */
	public int hashCode() {
		int hash = 11;
		hash = (hash * 31) ^ mX.hashCode();
		hash = (hash * 31) ^ mY.hashCode();
		return hash;
	}

	/**
	 * Determines if this object is not equal to the given <code>Object</code>.
	 * Comparisons with instances of non-<code>CoordPair</code> objects are always not equal. 
	 * When comparing instances of <code>CoordPair</code> objects,
	 * the object's <code>UnitSpan</code> coordinates
	 * are each compared for inequality.
	 * 
	 * @param compare 
	 *            the <code>Object</code> to compare.
	 * @return
	 *            true if not equal, false othewise.
	 * @see #equals(Object)
	 */
	public boolean notEquals(Object compare) {
		return ! equals(compare);
	}

	/**
	 * Returns a <code>CoordPair</code> representing the
	 * addition of this object and the given <code>CoordPair</code>.
	 * 
	 * @param add 
	 *            the <code>CoordPair</code> to add.
	 * @return
	 *            a coordinate of the addition.
	 */
	public CoordPair add(CoordPair add) {
		return new CoordPair(mX.add(add.mX), mY.add(add.mY));
	}

	/**
	 * Returns a <code>CoordPair</code> representing the
	 * subtraction of this object and the given <code>CoordPair</code>.
	 * 
	 * @param subtract 
	 *            the <code>CoordPair</code> to subtract.
	 * @return
	 *            a coordinate of the subtraction.
	 */
	public CoordPair subtract(CoordPair subtract) {
		return new CoordPair(mX.subtract(subtract.mX), mY.subtract(subtract.mY));
	}

	/**
	 * Returns a <code>CoordPair</code> representing the
	 * scaling of this object to the given scaling factors.
	 * 
	 * @param dXScale 
	 *            the X coordinate scaling factor.
	 * @param dYScale 
	 *            the Y coordinate scaling factor.
	 * @return
	 *            a coordinate of the scaling.
	 */
	public CoordPair scale(double dXScale, double dYScale) {
		return new CoordPair(mX.multiply(dXScale), mY.multiply(dYScale));
	}

	/**
	 * Determines if this object is greater than the given <code>CoordPair</code>.
	 * Coordinate comparisons rules are as follows: order by the y coordnate first,
	 * and if equal, then by the x coordinate.
	 * 
	 * @param compare 
	 *            the <code>CoordPair</code> to compare.
	 * @return
	 *            true if greater than, false otherwise.
	 */
	public boolean gt(CoordPair compare) {
		if (mY.gt(compare.mY)) {
			return true;
		} else if (mY.lt(compare.mY)) {
			return false;
		}
		return mX.gt(compare.mX);
	}

	/**
	 * Determines if this object is less than the given <code>CoordPair</code>.
	 * Coordinate comparisons rules as as follows: order by the y coordnate first,
	 * and if equal, then by the x coordinate.
	 * 
	 * @param compare 
	 *            the <code>CoordPair</code> to compare.
	 * @return
	 *            true if greater than, false otherwise.
	 */
	public boolean lt(CoordPair compare) {
		if (mY.lt(compare.mY)) {
			return true;
		} else if (mY.gt(compare.mY)) {
			return false;
		}
		return mX.lt(compare.mX);
	}

	/**
	 * The zero (0, 0) coordinate.
	 * 
	 * @return
	 *            the coordinate equal to (0, 0).
	 */
	public static CoordPair zeroZero() {
		return ZERO_ZERO;
	}

}
