/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2008 Adobe Systems Incorporated
 * All Rights Reserved
 *
 * Notice: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */

package com.adobe.xfa.wsdl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.adobe.xfa.Element;
import com.adobe.xfa.Node;
import com.adobe.xfa.ut.StringUtils;

/**
 * The <code>WSDLNode</code> interface is the primary datatype for the entire
 * Web Services Description Language Model.
 * @exclude from published api.
 */

public class WSDLNode extends Element {

/**
 * Enumeration of supported node type definitions.
 * <pre>
 *		WSDL_DOCUMENT			 : this node is a <code>WSDLDocument</code>
 *		WSDL_DEFINITIONS		 : this node is a <code>WSDLNode</code> of type WSDL_DEFINITIONS
 *		WSDL_TYPES				 : this node is a <code>WSDLNode</code> of type WSDL_TYPES
 *		WSDL_MESSAGE			 : this node is a <code>WSDLMessage</code>
 *		WSDL_PART				 : this node is a <code>WSDLPart</code>
 *		WSDL_OPERATION			 : this node is a <code>WSDLOperation</code>
 *		WSDL_PORTTYPE			 : this node is a <code>WSDLNode</code> of type WSDL_PORTYPE
 *		WSDL_BINDING			 : this node is a <code>WSDLNode</code> of type WSDL_BINDING
 *		WSDL_BINDING_OPERATION	 : this node is a <code>WSDLBindingOperation</code>
 *		WSDL_SERVICE			 : this node is a <code>WSDLNode</code> of type WSDL_SERVICE
 *		WSDL_INPUT				 : this node is a <code>WSDLNode</code> of type WSDL_INPUT
 *		WSDL_OUTPUT 			 : this node is a <code>WSDLNode</code> of type WSDL_OUTPUT
 *		WSDL_FAULT				 : this node is a <code>WSDLNode</code> of type WSDL_FAULT
 *		WSDL_PORT				 : this node is a <code>WSDLNode</code> of type WSDL_PORT
 *		WSDL_EXTEN				 : this node is a <code>WSDLExten</code>
 * </pre>
 */
	public static final int WSDL_UNKNOWN = 0;
	public static final int WSDL_DOCUMENT = 1;
	public static final int WSDL_DEFINITIONS = 2;
	public static final int WSDL_TYPES = 3;
	public static final int WSDL_MESSAGE = 4;
	public static final int WSDL_PART = 5;
	public static final int WSDL_OPERATION = 6;
	public static final int WSDL_PORTTYPE = 7;
	public static final int WSDL_BINDING = 8;
	public static final int WSDL_BINDING_OPERATION = 9;
	public static final int WSDL_SERVICE = 10;
	public static final int WSDL_INPUT = 11;
	public static final int WSDL_OUTPUT = 12;
	public static final int WSDL_FAULT = 13;
	public static final int WSDL_PORT = 14;
	public static final int WSDL_EXTEN = 15;
/**
 * Enumeration of supported attributes.
 * <pre>
 *		WSDLA_NAME		  : the "name" attribute
 *		WSDLA_ELEMENT	  : the "element" attribute in the WSDL_PART node
 *		WSDLA_TYPE		  : the "type" attribute in the WSDL_PART and WSDL_BINDING nodes
 *		WSDLA_MESSAGE	  : the "message" attribute in the WSDL_INPUT and WSDL_OUTPUT nodes
 *		WSDLA_BINDING	  : the "binding" attribute in the WSDL_PORT node
 * </pre>
 */
	public static final int WSDLA_BINDING = 0;
	public static final int WSDLA_ELEMENT = 1;
	public static final int WSDLA_MESSAGE = 2;
	public static final int WSDLA_NAME = 3;
	public static final int WSDLA_NAMESPACE = 4;
	public static final int WSDLA_PARAMETERORDER = 5;
	public static final int WSDLA_TYPE = 6;
	public static final int WSDLA_MAX = 7;

	private static final String[] mAttrList = {
		WSDL.BINDING,
		WSDL.ELEMENT,
		WSDL.MESSAGE,
		WSDL.NAME,
		WSDL.NAMESPACE,
		WSDL.PARAMETER_ORDER,
		WSDL.TYPE
	};

	private Element moDomNode;
	private WSDLDocument mpoOwner;
	private WSDLNode mpParentNode;
	private int meNodeType;
	private String msWSDLName = "";
	private String msPrefix;
//	private String msLocalName;
	private String msNamespaceURI;
	private String msTargetNamespace;
	private Map<String,String> mPrefixToNS;
	private Map<String,String> mNSToPrefix;

/**
 * Constructor.
 * @param poDocument Owning WSDL document for the new node.
 * @param oSrc Source DOM element.
 * @param eType WSDL node type.
 */
	public WSDLNode (WSDLDocument poDocument, Element oSrc, int eType) {
		mpoOwner = poDocument;
		mpParentNode = null;
		moDomNode = oSrc;
		meNodeType = eType;
		loadNamespaces();
	}

/*
 * Clone a WSDLNode.
 * @param bDeep - if true then clone recursively.
 */
// Note: cloneNode() commented out.  In C++, it is not used by XFA,
// Designer or Acrobat.  There is currently no requirement for it in
// Java.  The ported code below is untested.  This code could be restored
// if a requirement arose.
//	public WSDLNode cloneNode (boolean bDeep) {
//		WSDLNode poNewNode = new WSDLNode (mpoOwner, moDomNode, meNodeType);
//		cloneNodeHelper (poNewNode, bDeep);
//		return poNewNode;
//	}

/**
 * The code representing the type of object represented by this
 *	 <code>WSDLNode</code>
 * @return A code representing the type of object represented by this
 *	 <code>WSDLNode</code>
 */
	public int getNodeType () {
		return meNodeType;
	}

/**
 * Gets the <code>Document</code> object associated with this node.
 *
 * This is also
 * the <code>Document</code> object used to create new nodes. When this
 * node is a <code>Document</code> this is <code>NULL</code>.
 */
	public WSDLDocument getWSDLOwnerDocument () {
		return mpoOwner;
	}

	public WSDLNode getWSDLParentNode () {
		return mpParentNode;
	}

	public WSDLNode getWSDLChildNode (int inType, String inName) {
// first check the prefix to make sure it corresponds to the targetnamespace of the Document
		String localName = inName;

// check if the name has a prefix.
		if (! StringUtils.isEmpty (inName)) {
			WSDLDocument.PrefixParseInfo prefixParseInfo = getWSDLOwnerDocument().checkAndParsePrefix (inName, this);
			if (prefixParseInfo == null) {
				return null;
			}
			localName = prefixParseInfo.mLocalName;
		}

		WSDLNode node = getFirstWSDLNode();
		while (node != null) {
			int nodeType = node.getNodeType();
			if (nodeType == inType) {
				if (StringUtils.isEmpty (localName)) {
// check if the local name matches if it's specified
					return node;
				} else if (StringUtils.equalsWithNull (node.getWSDLName(), localName)) {
					return node;
				}
			}
			node = node.getNextWSDLNode();
		}
		return null;
	}

	public WSDLNode getWSDLChildNode (int inType, String inName, String inTargetNS) {
// first check the prefix to make sure it corresponds to the targetnamespace of the Document
		String localName = inName;

		WSDLNode node = getFirstWSDLNode();
		while (node != null) {
			int nodeType = node.getNodeType();
			if (nodeType == inType) {
				if (StringUtils.equalsWithNull (node.getWSDLName(), localName)
				 && StringUtils.equalsWithNull (node.getTargetNamespace(), inTargetNS)) {
// check if the local name matches if it's specified
					return node;
				}
			}
			node = node.getNextWSDLNode();
		}
		return null;
	}

	public List<WSDLNode> getWSDLChildNodesOfType (int inType) {
		List<WSDLNode> outNodesArray = new ArrayList<WSDLNode>();
		WSDLNode node = getFirstWSDLNode();
		while (node != null) {
			if (node.getNodeType() == inType) {
				outNodesArray.add (node);
			}
			node = node.getNextWSDLNode();
		}
		return outNodesArray;
	}

/**
 * The namespace URI of this node
 * @return The namespace URI of this node or <code>String::EmptyString()</code>
 *	 if it is unspecified
 */
	public String getNamespaceURI () {
		return msNamespaceURI;
	}

	public void setNamespaceURI (String sNSURI) {
		msNamespaceURI = sNSURI;
	}

/**
 * The target namespace of this node
 * @return The target namespace of this node or <code>String::EmptyString()</code>
 *	 if it is unspecified.	This is equal to the targetnameSpace on the owner document,
 *	 unless this node origniated from an import elememt with a different namespace
 */
	public String getTargetNamespace () {
		return msTargetNamespace;
	}

	public void setTargetNamespace (String sTargetNS) {
		msTargetNamespace = sTargetNS;
	}

	public String getNSURI (String sPrefix) {
		if (mPrefixToNS != null) {
			String value = mPrefixToNS.get (sPrefix);
			if (value != null) {
				return value;
			}
		}

// try ancestors to find any namespace in scope
		Element parent = getXMLParent();
		if (parent instanceof WSDLNode) {
			WSDLNode wsdlParent = (WSDLNode) parent;
			return wsdlParent.getNSURI (sPrefix);
		}

		return "";
	}

	public String getNSPrefix (String sURI) {
		String value = mNSToPrefix.get (sURI);
		return (value == null) ? "" : value;
	}

/**
 * The qualified name (W3C::QName) of this node
 *	 depending on its type (specified by <code>getNodeType()</code>).
 * @return The QName of this node depending on its type.
 *	 <pre>
 *	   WSDL_TYPES				: the literal string "#types"
 *	   WSDL_MESSAGE,			: the name of the message
 *	   WSDL_PART,				: the name of the part
 *	   WSDL_OPERATION,			: the name of the operation
 *	   WSDL_PORTTYPE,			: the name of the portType
 *	   WSDL_BINDING,			: the name of the binding
 *	   WSDL_SERVICE,			: the name of the service
 *	   WSDL_INPUT,				: the name of the input
 *	   WSDL_OUTPUT, 			: the name of the output
 *	   WSDL_FAULT,				: the name of the fault
 *	   WSDL_PORT				: the name of the port
 *	 </pre>
 * <dl><b>See Also: </b>
 *	 <a href="http://www.w3.org/TR/xmlschema-2/#QName">W3C::QName</a>
 * </dl>
 */
	public String getWSDLName () {
		return msWSDLName;
	}

	public void setWSDLName (String sNodeName) {
		msWSDLName = sNodeName;
	}

	public String getWSDLPrefix () {
		return msPrefix;
	}

	public void setWSDLPrefix (String sPrefix) {
		msPrefix = sPrefix;
	}

	public Element getDomNode () {
		return moDomNode;
	}

/**
 * Gets a <code>String</code> containing the named attribute of this node (if it
 * is has the named attribute or Sting::EmptyString() otherwise.
 */
	public String getWSDLAttribute (int eNodeAttribute) {
		return getWSDLAttribute (mAttrList, eNodeAttribute);
	}

	final String getWSDLAttribute (String[] attrList, int index) {
		String result = null;
		if ((index >= 0) && (index < attrList.length)) {
			int attrIndex = moDomNode.findAttr (null, attrList[index]);
			if (attrIndex >= 0) {
				result = moDomNode.getAttrVal (attrIndex);
			}
		}
		return (result == null) ? "" : result;
	}

// See comments with cloneNode() as to why this is commented out.
//	void cloneNodeHelper (WSDLNode poNewNode, boolean bDeep) {
//		poNewNode.setLocalName (getLocalName());
//		poNewNode.setNamespaceURI (msNamespaceURI);
//		poNewNode.setWSDLName (msWSDLName);
//		poNewNode.setWSDLPrefix (msPrefix);
//
//		if (bDeep) {
//			Node node = poNewNode.getFirstXMLChild();
//			Node copy = null;
//			while (node != null) {
//				if (node instanceof WSDLNode) {
//					WSDLNode wsdlNode = (WSDLNode) node;
//					copy = wsdlNode.cloneNode (true);
//					poNewNode.appendWSDLChild ((WSDLNode) copy);
//				} else {
//					copy = node.clone (poNewNode);
//				}
//				node = node.getNextXMLSibling();
//			}
//		}
//	}

	final void appendWSDLChild (WSDLNode child) {
		appendChild (child);
		child.mpParentNode = this;
	}

	final WSDLNode getFirstWSDLNode () {
		return getFirstWSDLNode (WSDL_UNKNOWN);
	}

	final WSDLNode getFirstWSDLNode (int nodeType) {
		return getNextWSDLNode (getFirstXMLChild(), nodeType, false);
	}

	final WSDLNode getNextWSDLNode () {
		return getNextWSDLNode (WSDL_UNKNOWN);
	}

	final WSDLNode getNextWSDLNode (int nodeType) {
		return getNextWSDLNode (this, nodeType, true);
	}

	private static final WSDLNode getNextWSDLNode (Node node, int nodeType, boolean skipFirst) {
		while (node != null) {
			if (skipFirst) {
				skipFirst = false;
			} else if (node instanceof WSDLNode) {
				WSDLNode wsdlNode = (WSDLNode) node;
				if ((nodeType == WSDL_UNKNOWN) || (wsdlNode.getNodeType() == nodeType)) {
					return wsdlNode;
				}
			}
			node = node.getNextXMLSibling();
		}
		return null;
	}

	private void loadNamespaces () {
		if (moDomNode == null) {
			return;
		}

		int attrCount = moDomNode.getNumAttrs();
		for (int i = 0; i < attrCount; i++) {
// if an attribute's nodeName contains a ":", it is probably
// a "xmlns:prefix" attribute where the localName is the
// prefix and the nodeValue is the namespace URI
			String attrName = moDomNode.getAttrQName (i);
			int pos = attrName.indexOf (':');
			if ((pos >= 0) && attrName.substring(0,pos).equals (WSDL.XMLNS)) {
				String prefix = moDomNode.getAttrName(i);
				String namespace = moDomNode.getAttrVal(i);
				if (mPrefixToNS == null) {
					mPrefixToNS = new HashMap<String,String>();
				}
				mPrefixToNS.put (prefix, namespace);
				if (mNSToPrefix == null) {
					mNSToPrefix = new HashMap<String,String>();
				}
				mNSToPrefix.put (namespace, prefix);
			}
		}
	}
}
