/**************************************************************************
 *                                                                        *
 * ADOBE CONFIDENTIAL                                                     *
 * ___________________                                                    *
 *                                                                        *
 *  Copyright 2011 Adobe Systems Incorporated                             *
 *  All Rights Reserved.                                                  *
 *                                                                        *
 * NOTICE:  All information contained herein is, and remains              *
 * the property of Adobe Systems Incorporated and its suppliers,          *
 * if any.  The intellectual and technical concepts contained             *
 * herein are proprietary to Adobe Systems Incorporated and its           *
 * suppliers and are protected by trade secret or copyright law.          *
 * Dissemination of this information or reproduction of this material     *
 * is strictly forbidden unless prior written permission is obtained      *
 * from Adobe Systems Incorporated.                                       *
 **************************************************************************/
package com.adobe.xmp.schema.service;

import java.io.InputStream;

import com.adobe.xmp.path.XMPPath;
import com.adobe.xmp.schema.model.PropertyDescription;
import com.adobe.xmp.schema.model.PropertyType;
import com.adobe.xmp.schema.model.SchemaDescription;


/**
 * Interface for a service to provide schema information for XMP schemas and properties.
 * A simple property path of QNames, '/' separators and array indices '[xxx]' can be used
 * to determine a properties type. 
 * 
 * @author Stefan Makswit
 */
public interface SchemaService
{
	
	/**
	 * Request the schema information for a namespace.
	 * 
	 * @param ns a namespace URI, e.g. "http://purl.org/dc/elements/1.1/" for Dublin Core
	 * @return Returns a <code>SchemaDescription</code> or <code>null</code> if it is not available.
	 * @throws SchemaServiceException Forwards exceptions that occur during schema file loading and parsing.
	 */
	SchemaDescription getSchemaForURI(String ns) throws SchemaServiceException;
	
	
	/**
	 * Provides the type information for a single property (simple, array (and item type), 
	 * struct or qualifier) for a given XMP path.
	 * Note: Qualifiers are currently not supported.
	 * 
	 * The prefixes used in the property paths are resolved using the <code>NamespaceUtil</code>.
	 * <p>Examples:
	 * <ul>
	 * 		<li>dc:Rating -- a simple property type</li>
	 * 		<li>exif:Flash/exif:Fired -- a "struct" containing a simple "boolean" property</li>
	 * 		<li>dc:subject -- an "Alt-array" with item type "Text"</li>
	 * 		<li>dc:subject[3] -- the item type of the array, which is "Text".
	 * 			<em>Note</em> The array index is ignored and can also be left out "[]"</li>
	 * </ul>
	 * 
	 * @param propertyPath {@link XMPPath} which identifies the property
	 * 
	 * @return Returns the property type for the requested property, 
	 * 		or <code>null</null> if the property is not part of the schema and no type information is available.
	 * 		An exception is thrown, if the path is syntactically not correct.
	 * 		<code>null</code> is returned if path is not accessible
	 * 		(e.g. child property of a simple property).
	 * 
	 * @throws SchemaServiceException Reports errors during the retrieval of schema or type information.
	 */
	PropertyType getType(XMPPath propertyPath) throws SchemaServiceException;
	
	
	/**
	 * Retrieves the property description for all kinds of properties for a given XMP path.
	 * Note: Qualifiers are currently not supported.
	 * @param propertyPath {@link XMPPath} which identifies the property
	 * 
	 * @return Returns the property description for the requested property, 
	 * 		or <code>null</null> if the property is not part of the schema and no type information is available.
	 * 		An exception is thrown, if the path is syntactically not correct.
	 * 		<code>null</code> is returned if path is not accessible
	 * 		(e.g. child property of a simple property).
	 * 
	 * @throws SchemaServiceException Reports errors during the retrieval of schema or type information.
	 */
	PropertyDescription getProperty(XMPPath propertyPath) throws SchemaServiceException;
	
	
	/**
	 * Parses a RelaxNG file from the inputstream and reports namespaces with corresponding prefixes
	 * to the callback interface.
	 * 
	 * @param input a RelaxNG file
	 * @param namespaceCallback a callback function to report the namespaces found during parsing
	 * @return Returns a {@link com.adobe.xmp.schema.model.SchemaDescription}
	 * 
	 * @throws SchemaServiceException Reports errors during the retrieval of schema or type information.
	 */
	SchemaDescription parseRelaxNG(InputStream input, INamespaceCallback namespaceCallback) throws SchemaServiceException;
	
	
//	TODO uncomment if it can be built with Java 1.6 
//	/**
//	 * Serializes a {@link SchemaDescription} into RelaxNG format.
//	 * The result is written to the output writer.
//	 * 
//	 * @param schemaDescription an SchemaDescription
//	 * @param nsPrefixMap a namespace-to-prefix binding that should be used preferably during serialization.
//	 * 		  			  If no prefix is available for a namespace, an prefix is generated (ns1, ns2...)
//	 * @param output an OutputStream to write the Relax NG to.
//	 * @throws SchemaServiceException Thrown if the serialization produces errors.
//	 */
//	void serializeRelaxNG(SchemaDescription schemaDescription, Map<String, String> nsPrefixMap, OutputStream output) throws SchemaServiceException;
	
	
	/**
	 * An interface to report prefix/namespace bindings during Relax NG parsing.
	 * 
	 * @author Stefan Makswit
	 */
	public interface INamespaceCallback
	{
		/**
		 * Reports a new prefix-namespace mapping found during parsing.
		 * 
		 * @param namespace a namespace
		 * @param prefix a prefix
		 */
		void notify(String namespace, String prefix);
	}
	
	
// TODO discuss if this is needed
//	/**
//	 * Access to all available namespaces.
//	 * 
//	 * @return Returns a list of all available schema namespaces.
//	 */
//	Set<String> getAvailableNamespaces();
}
