/*******************************************************************************
 * ADOBE CONFIDENTIAL
 *  __________________
 *  Copyright 2018 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 *  NOTICE:  All information contained herein is, and remains
 *  the property of Adobe Systems Incorporated and its suppliers,
 *  if any.  The intellectual and technical concepts contained
 *  herein are proprietary to Adobe Systems Incorporated and its
 *  suppliers and are protected by trade secret or copyright law.
 *  Dissemination of this information or reproduction of this material
 *  is strictly forbidden unless prior written permission is obtained
 *  from Adobe Systems Incorporated.
 ******************************************************************************/

package com.day.cq.analytics.testandtarget;

import org.apache.commons.lang.StringUtils;

import com.day.cq.analytics.testandtarget.mac.MacMetadata;

/**
 * Parent object for the request objects involved in Adobe Target offer synchronization
 */
public class OfferBase {
    private long id;
    private String name;
    private String type = "html";
    private transient String contentPath;
    private String content;
    private String workspace;
    
    // mark as "transient" so it won't be serialized automatically
    private transient MacMetadata metadata;

    public OfferBase(long id, String name, String content) {
        this.id = id;
        this.name = name;
        this.content = content;
    }

    /**
     * Returns the name of the offer
     * @return a {@link String} object containing the offer name
     */
    public String getName() {
        return name;
    }

    /**
     * Returns the offer id, as retrieved from Adobe Target
     * @return the id of the offer. If the offer hasn't been synchronized with Target this value is 0
     */
    public long getId() {
        return id;
    }

    /**
     * Returns the path of this offer from the AEM repository
     * @return the offer path
     */
    public String getContentPath() {
        return contentPath;
    }

    /**
     * Returns the offer content, as set in Adobe Target
     * @return the offer content. This is usually a callback to an AEM function which pulls the content from the
     * repository
     */
    public String getContent() {
        return content;
    }

    /**
     * Sets the metadata for this offer.
     *
     * @param metadata a {@link MacMetadata} object. Note that the object must contain the mandatory metadata.
     */
    public void setMacMetadata(MacMetadata metadata) {
        this.metadata = metadata;
    }

    /**
     * Retrieves the metadata object attached to this offer reuquest.
     * @return a {@link MacMetadata} object
     */
    public MacMetadata getMacMetadata() {
        return metadata;
    }

    /**
     * Returns the offer type.
     * @return the offer type.
     */
    public String getType() {
        return type;
    }

    /**
     *
     * @param type offer type
     */
    public void setType(String type) {
        this.type = type;
    }

    /**
     * @return the offer workspace id
     */
    public String getWorkspace() {
        return workspace;
    }

    /**
     * Optional id of workspace to which the offer belongs.
     * Max length is 250 characters. By default, Default workspace is assumed.
     * @param workspace id of workspace
     */
    public void setWorkspace(String workspace) {
        this.workspace = workspace;
        // blank workspace accepted by Target API, but offer with blank workspace shows not up in target.
        if (StringUtils.isBlank(workspace)) {
            this.workspace = null;
        }
    }
}
