package com.day.cq.dam.api;

import java.io.InputStream;

import javax.jcr.Binary;

import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ValueMap;
import org.osgi.annotation.versioning.ProviderType;

/**
 * The <code>Rendition</code> interface specifies the handling of an {@link Asset}'s rendition. Renditions are based on
 * the renditions found in the asset's rendition folder (<i>./jcr:content/renditions</i>). Such renditions are based on
 * nodes of type <code>nt:file</code>.
 */
@ProviderType
public interface Rendition extends Resource {

    /**
     * Returns the mime type of the rendition's binary, as denoted by its <code>jcr:mimeType</code> property.
     *
     * @return The mime type of the rendition's binary.
     */
    String getMimeType();

    /**
     * Returns the name of the rendition. The name corresponds to the node name underlying the rendition, e.g.
     * <i>./jcr:content/renditions/myrendition</i> gives a name of <i>myrendition</i>
     *
     * @return The name of the rendition.
     */
    String getName();

    /**
     * Returns the path of the rendition, e.g. <i>/content/dam/myasset/jcr:content/renditions/myrendition</i>.
     *
     * @return The path of the rendition.
     */
    String getPath();

    /**
     * Returns the {@link ValueMap} of the rendition's underlying <i>./jcr:content</i> node.
     *
     * @return The content properties.
     */
    ValueMap getProperties();

    /**
     * Returns the size in bytes of the rendition's binary.
     *
     * @return The size.
     */
    long getSize();

    /**
     * Returns the <code>InputStream</code> representing the binary of this rendition. Alternatively the stream can be
     * obtained by adapting the rendition:
     * <blockquote><pre>
     *    ...
     *    final Rendition rendition = asset.getRendition("myrendition");
     *    final InputStream stream = rendition.adaptTo(InputStream.class);
     *    ...
     * </pre></blockquote>
     *
     * @return The input stream of the rendition's binary, or <code>null</code> if the binary is not found.
     */
    InputStream getStream();

    /**
     * Returns the <code>Binary</code> of this rendition.
     *
     * @return The rendition's binary, or <code>null</code> if the binary is not found.
     */
    Binary getBinary();

    /**
     * Returns this rendition's parent Asset.
     *
     * @return The {@link Asset} this rendition belongs to.
     */
    Asset getAsset();
}
