/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */

package com.day.cq.reporting.helpers;

import java.text.DateFormat;
import java.util.Calendar;
import java.util.Locale;
import java.util.TimeZone;

import com.day.cq.reporting.impl.snapshots.AggregationInterval;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.request.RequestPathInfo;
import org.apache.sling.commons.json.JSONException;
import org.apache.sling.commons.json.io.JSONWriter;

/**
 * Class that provides some helper functionality.
 */
public class Util {

    /**
     * Prevent instantiation.
     */
    private Util() {
        // prevent
    }

    /**
     * Creates a timestamp (format: yyyymmddhhmmss) from the specified {@link Calendar},
     *
     * @param cal The calendar
     * @return The timestamp
     */
    public static long createTimeStamp(Calendar cal) {
        long ds = cal.get(Calendar.YEAR);
        ds *= 100;
        ds += cal.get(Calendar.MONTH) + 1;
        ds *= 100;
        ds += cal.get(Calendar.DAY_OF_MONTH);
        ds *= 100;
        ds += cal.get(Calendar.HOUR_OF_DAY);
        ds *= 100;
        ds += cal.get(Calendar.MINUTE);
        ds *= 100;
        ds += cal.get(Calendar.SECOND);
        return ds;
    }

    /**
     * Writes the specified {@link Calendar} to the specified {@link JSONWriter}.
     *
     * @param writer The writer
     * @param interval The aggregation interval to create display values for
     * @param cal The calendar
     * @param timeZone The time zone
     * @param locale The locale
     * @throws JSONException if writing data failed
     */
    public static void writeToJSON(JSONWriter writer, AggregationInterval interval,
                                   Calendar cal, TimeZone timeZone, Locale locale)
            throws JSONException {
        writer.object();
        writer.key("tc").value(cal.getTimeInMillis());
        DateFormat df;
        if (interval == AggregationInterval.HOUR) {
            df = DateFormat.getTimeInstance(DateFormat.SHORT, locale);
        } else if (interval != null) {
            df = DateFormat.getDateInstance(DateFormat.SHORT, locale);
        } else {
            df = DateFormat.getDateTimeInstance(DateFormat.SHORT, DateFormat.SHORT, locale);
        }
        df.setTimeZone(timeZone);
        writer.key("display").value(df.format(cal.getTime()));
        writer.endObject();
    }

    /**
     * Clears the time information by setting it back to midnight.
     *
     * @param calendar The calendar to clear time information from
     */
    public static void clearTime(Calendar calendar) {
        calendar.set(Calendar.HOUR_OF_DAY, 0);
        calendar.set(Calendar.MINUTE, 0);
        calendar.set(Calendar.SECOND, 0);
        calendar.set(Calendar.MILLISECOND, 0);
    }

    /**
     * Determines if the specified <code>String</code> array contains the specified string.
     *
     * @param array The array
     * @param str The string
     * @return <code>true</code> if the array contains the given string
     */
    public static boolean arrayContainsString(String[] array, String str) {
        if (array == null) {
            return false;
        }
        for (String item : array) {
            if (item != null) {
                if (item.equals(str)) {
                    return true;
                }
            } else {
                if (str == null) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Gets the render type from the reuest.
     *
     * @param req The request to determine the render type from
     * @return The render type ({@link Const#RENDER_TYPE_TABULAR} or
     *         {@link Const#RENDER_TYPE_CHART}; <code>null</code> for rendering the
     *         chart in default mode
     */
    public static String getRenderTypeFromRequest(SlingHttpServletRequest req) {
        String renderType = null;
        RequestPathInfo pathInfo = req.getRequestPathInfo();
        String[] selectors = (pathInfo != null ? pathInfo.getSelectors() : null);
        if ((selectors != null) && (selectors.length >= 1)) {
            String typeSelector = selectors[0];
            if (typeSelector.equals(Const.RENDER_TYPE_TABULAR)
                    || typeSelector.equals(Const.RENDER_TYPE_CHART)) {
                renderType = typeSelector;
            }
        }
        return renderType;
    }

    /**
     * Gets a chart id for rendering from the specified request, if applicable.
     *
     * @param req The request
     * @return The chart id; <code>null</code> if no chart id is available/required
     */
    public static String getChartIdFromRequest(SlingHttpServletRequest req) {
        String chartId = null;
        RequestPathInfo pathInfo = req.getRequestPathInfo();
        String[] selectors = (pathInfo != null ? pathInfo.getSelectors() : null);
        if ((selectors != null) && (selectors.length >= 2)) {
            String typeSelector = selectors[0];
            if (typeSelector.equals(Const.RENDER_TYPE_CHART)) {
                chartId = selectors[1];
            }
        }
        return chartId;
    }

    /**
     * Checks if the specified request triggers a single view rendering.
     *
     * @param req The request to be analyzed
     * @return <code>true</code> if a single view rendering is triggered by the request
     */
    public static boolean isSingleViewRendering(SlingHttpServletRequest req) {
        return (Util.getRenderTypeFromRequest(req) != null);
    }

    /**
     * Determines a predefines size from the specified request, if applicable.
     *
     * @param req The request
     * @return A two element array specifiying a predefined width (first element) and
     *         height (second element); each of the element may be <code>null</code>,
     *         indicating that the respective dimension has to be autocalculated
     */
    public static Integer[] getRenderSizeFromRequest(SlingHttpServletRequest req) {
        Integer[] size = new Integer[2];
        RequestPathInfo pathInfo = req.getRequestPathInfo();
        String[] selectors = (pathInfo != null ? pathInfo.getSelectors() : null);
        if (selectors != null) {
            // todo too simple once more selectors are supported
            int selectorCnt = selectors.length;
            if (selectorCnt >= 3) {
                Integer width = null;
                try {
                    width = Integer.parseInt(selectors[selectorCnt - 2]);
                } catch (NumberFormatException nfe) {
                    // handle by null value
                }
                Integer height = null;
                try {
                    height = Integer.parseInt(selectors[selectorCnt - 1]);
                } catch (NumberFormatException nfe) {
                    // handle by null value
                }
                size[0] = width;
                size[1] = height;
            }
        }
        return size;
    }

    /**
     * Determines a provided chart layout from the specified request, if applicable.
     *
     * @param req The request
     * @return The chart layout ("horizontal", "vertical" or <code>null</code> (if no
     *         chart layout was provided)
     */
    public static String getChartLayoutFromRequest(SlingHttpServletRequest req) {
        String chartLayout = null;
        RequestPathInfo pathInfo = req.getRequestPathInfo();
        String[] selectors = (pathInfo != null ? pathInfo.getSelectors() : null);
        if ((selectors != null) && (selectors.length >= 3)) {
            String typeSelector = selectors[0];
            if (typeSelector.equals(Const.RENDER_TYPE_CHART)) {
                String chartLayoutCandidate = selectors[2];
                if (chartLayoutCandidate.equals("h")) {
                    chartLayout = "horizontal";
                } else if (chartLayoutCandidate.equals("v")) {
                    chartLayout = "vertical";
                }
            }
        }
        return chartLayout;
    }

    /**
     * <p>Compares two objects by their respective {@link #equals} methods. Both objects
     * may be <code>null</code>.</p>
     *
     * @param obj1 The first object
     * @param obj2 The second object
     * @return <code>true</code> if both objects are equal
     */
    public static boolean equalsNullAware(Object obj1, Object obj2) {
        if ((obj1 == null) && (obj2 == null)) {
            return true;
        }
        return (obj1 != null) && obj1.equals(obj2);
    }

}
