/*
 * Copyright 1997-2010 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.search.suggest;

import java.util.List;

import javax.jcr.RepositoryException;

import aQute.bnd.annotation.ProviderType;

/**
 * Represents a term suggestion index, with methods for reading from and writing
 * to the index.
 * 
 * <p>
 * To persist single changes made to the index ({@link #add(String, String[])
 * add()}, {@link #remove(String) remove()}), call {@link #save()}. The possibly
 * large-scale operations {@link #index(List, int, boolean) index()} and
 * {@link #delete()} will save automatically.
 * 
 * <p>
 * <b>Note:</b> make sure to {@link #close()} the index after usage to properly
 * release affected resources, e.g. an underlying JCR session. This should be
 * done in a try/finally block.
 */
@ProviderType
public interface SuggestionIndex {

    /**
     * Looks up the suggestions for a given term.
     * 
     * @param term
     *            partial word to look up in the index; one or more characters
     * @return a list of suggestions or an empty array if no suggestions were
     *         found
     * @throws RepositoryException
     *             if an unexpected repository error occurred
     */
    String[] read(String term) throws RepositoryException;

    /**
     * Adds suggestions for a given term to the index. If the term already
     * exists in the index, the previous suggestions will be overwritten.
     * 
     * <p>
     * To persist the change, {@link #save()} must be called.
     * 
     * @param term
     *            partial word for which the suggestions apply
     * @param suggestions
     *            an array of suggestions
     * @throws RepositoryException
     *             if an unexpected repository error occurred
     */
    void add(String term, String[] suggestions) throws RepositoryException;

    /**
     * Removes the term and its suggestions from the index.
     * 
     * <p>
     * To persist the change, {@link #save()} must be called.
     * 
     * @param term
     *            partial word to remove from the index
     * @throws RepositoryException
     *             if an unexpected repository error occurred
     */
    void remove(String term) throws RepositoryException;

    /**
     * Deletes the entire index.
     * 
     * <p>
     * Changes will be persisted automatically.
     * 
     * @throws RepositoryException
     *             if an unexpected repository error occurred
     */
    void delete() throws RepositoryException;
    
    /**
     * Saves any single changes made to the index.
     * 
     * @throws RepositoryException
     *             if an unexpected repository error occurred
     */
    void save() throws RepositoryException;

    /**
     * Closes the index and releases all allocated resources with it, e.g. the
     * underlying JCR session. After this has been called, all methods will fail
     * if called.
     * 
     * @throws RepositoryException
     *             if an unexpected repository error occurred
     */
    void close() throws RepositoryException;

    /**
     * Extracts a suggestion index based on the given terms, weighted by their
     * frequencies. Will create all partial words starting with one letter, two
     * letters and so on, for which full words exist, and will add up to the
     * given maximum number of suggested terms for each partial word, weighted
     * by the frequency.
     * 
     * <p>
     * Changes will be persisted automatically.
     * 
     * @param terms
     *            a list of {@link Term}s, with the string term and the
     *            frequency
     * @param maxSuggestions
     *            the maximum number of suggestion per term
     * @param minimize
     *            Whether the index should be minimized. That means (longer)
     *            term prefixes that result in only one suggestion should not be
     *            indexed; only the first prefix that will result in this single
     *            suggestion would be stored, but not for the remaining letters
     *            of the suggested word.
     * @throws RepositoryException
     *             if an unexpected repository error occurred
     */
    void index(List<Term> terms, int maxSuggestions, boolean minimize) throws RepositoryException;
}