/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2016 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 *
 **************************************************************************/
package com.day.cq.wcm.api;

import com.adobe.granite.ui.components.ExpressionCustomizer;
import com.adobe.granite.ui.components.ExpressionResolver;
import com.adobe.granite.ui.components.FilteringResourceWrapper;
import com.day.cq.wcm.api.designer.Designer;
import com.day.cq.wcm.api.policies.ContentPolicy;
import com.day.cq.wcm.api.policies.ContentPolicyManager;
import org.apache.sling.api.SlingHttpServletRequest;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ValueMap;

/**
 * Filters child resources of the given <code>resource</code> based on the given <code>contentResource</code> design properties
 *
 * The current wrapper is filtering child resources using Granite <code>ExpressionResolver</code>
 * It does that by adding custom variables to the <code>ExpressionCustomizer</code> which will use the request as a data vessel.
 *
 * Later the <code>ExpressionResolver</code> via its <code>CustomVariableELResolver</code> will read the registered custom variables to resolve the <code>granite:hide</code> expression.
 * That expression can then refer to the custom variables
 *
 * @see com.adobe.granite.ui.components.impl.ExpressionResolverImpl
 * @see com.adobe.granite.ui.components.impl.CustomVariableELResolver
 * @see ExpressionCustomizer
 */
public class WCMFilteringResourceWrapper extends FilteringResourceWrapper {

    private static ResourceResolver resourceResolver;

    /**
     * Filters child resources of the given <code>resource</code> based on the given <code>contentResource</code> design properties
     *
     * e.g.
     *
     * - resource: the content of a dialog that contains fields that are conditionally hidden base on the design properties associated with the <code>contentResource</code>
     * - contentResource: a resource of type image that has design properties associated
     *
     * @param filteredResource Resource for which to filter child resources
     * @param contentResource Content Resource from which to extract design properties
     * @param expressionResolver Expression Resolver Service
     * @param slingRequest <code>SlingHttpServletRequest</code>
     */
    public WCMFilteringResourceWrapper(Resource filteredResource, Resource contentResource, ExpressionResolver expressionResolver, SlingHttpServletRequest slingRequest) {
        super(filteredResource, expressionResolver, slingRequest);

        registerCustomVariables(contentResource, slingRequest);
    }

    /**
     * Returns the design properties whether it is stored as a design or content policy node
     *
     * @param resource Resource for which to get design properties
     * @return
     */
    private ValueMap getDesignProperties(Resource resource) {
        ValueMap design = ValueMap.EMPTY;
        PageManager pageManager = resourceResolver.adaptTo(PageManager.class);
        Page page = pageManager.getContainingPage(resource);

        if (page != null) {
            Template template = page.getTemplate();

            if (template != null && template.hasStructureSupport()) {
                ContentPolicyManager policyManager = resourceResolver.adaptTo(ContentPolicyManager.class);
                ContentPolicy policy = policyManager.getPolicy(resource);

                if (policy != null) {
                    Resource policyResource = policy.adaptTo(Resource.class);
                    design = policyResource.adaptTo(ValueMap.class);
                }
            } else {
                Designer designer = resourceResolver.adaptTo(Designer.class);
                design = designer.getStyle(resource);
            }
        }

        return design;
    }

    /**
     * Loads necessary custom variables/scopes for a <i>Granite</i> resolution into the given request. It will later be read by the <code>CustomVariableELResolver</code>
     *
     * @param contentResource Resource representing the content from which to extract scopes
     * @param request
     *
     * @see com.adobe.granite.ui.components.impl.ExpressionResolverImpl
     * @see com.adobe.granite.ui.components.impl.CustomVariableELResolver
     * @see ExpressionCustomizer
     */
    private void registerCustomVariables(Resource contentResource, SlingHttpServletRequest request) {
        if (contentResource == null || request == null) {
            return;
        }

        resourceResolver = contentResource.getResourceResolver();
        ValueMap designVM = getDesignProperties(contentResource);

        ExpressionCustomizer customizer = ExpressionCustomizer.from(request);
        customizer.setVariable("cqDesign", designVM);
    }
}
