/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.api;

import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import com.day.cq.wcm.designimporter.DesignImporterContext;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.parser.HTMLContentHandler;

/*
AdobePatentID="2613US01"
*/

/**
 * Interface for handling HTML tags in the input HTML document. The tag handlers are registered with the
 * {@link TagHandlerProvider} according to tag/attribute matching rules and receive callbacks from
 * the {@link HTMLContentHandler} while it is handling the SAX events.
 *
 * <p>
 * Tag Handlers work in a chain of responsibility with each tag handler creating a new instance of a
 * nested tag handler and delegating to it. Majority of the tag handlers don't have a standalone existence
 * and work nested within other tag handlers. The tag handlers which are capable of working stand alone need
 * to implement this interface.
 * </p>
 */
public interface TagHandler {

    /**
     * Callback to signal the start of handling giving the tag handler a chance for running intialization routines.
     * <p>
     * {@link TagHandler}s are registered against tag+attribute combination and are invoked by the {@link HTMLContentHandler} as and when it
     * encounters a tag matching the tag attribute combination the tag handler is registered with.
     * 
     * @param uri
     * @param localName
     * @param qName
     * @param atts
     */
    void beginHandling(String uri, String localName, String qName, Attributes atts) throws DesignImportException;

    /**
     * Callback to signal the encounter of character content within tags.
     * 
     * @param ch
     * @param start
     * @param length
     * @throws SAXException
     */
    void characters(char[] ch, int start, int length) throws DesignImportException;

    /**
     * Callback to signal the encounter of a tag end.
     * 
     * @param uri
     * @param localName
     * @param qName
     * @throws SAXException
     */
    void endElement(String uri, String localName, String qName) throws DesignImportException;

    /**
     * Callback to signal the end of handling giving the tag handler a chance for finalizing things.
     * <p>
     * This is arguably the last callback a tag handler receives. After this, the current tag handler is popped out from the chain and the
     * subsequent Tika events are directed to preceding tag handlers.
     * 
     * @param uri
     * @param localName
     * @param qName
     */
    void endHandling(String uri, String localName, String qName) throws DesignImportException;

    /**
     * Sets the design importer context
     * 
     * @param designImporterContext
     */
    void setDesignImporterContext(DesignImporterContext designImporterContext);

    void setTagHandlerProvider(TagHandlerProvider tagHandlerProvider);

    /**
     * Callback to signal the encounter of a tag start.
     * <p>
     * Once a tag handler begins handling, it receives the start tag events for all the nested tags within the purview of the tag this tag
     * handler is registered against.
     * 
     * @param uri
     * @param localName
     * @param qName
     * @param atts
     * @throws SAXException
     */
    void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException;
}
