/*
 *
 *	File: AutoColor.java
 *
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2008 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 */

package com.adobe.fontengine.font.cff;

import java.util.ArrayList;

import com.adobe.fontengine.font.HintedOutlineConsumer;
import com.adobe.fontengine.font.Matrix;
import com.adobe.fontengine.font.OutlineConsumer;

public class AutoColor implements OutlineConsumer
{
	/* Public definitions */
	public static final int AC_ALLOWEDIT		 = 1<<0;	/* allow modifying paths */
	public static final int AC_HINTSUB		 = 1<<1;	/* generate hint substitution operators */
	public static final int AC_FIXWINDING		 = 1<<2;	/* fix each sub-path direction based on the non-zero winding rule */
	public static final int AC_GENERATEVSTEMS	 = 1<<3;	/* generate vertical stem hints */

	/* Zone hint types */
	public static final int ZN_VSTEM = 0;
	public static final int ZN_HSTEM = 1;
	public static final int ZN_CHAREXTREME = 2;
	public static final int ZN_CHARZONE = 3;
	public static final int ZN_ZONE = 4;

	/* Declarations */
	private static final int FIXED_POS_INF = 0x7FFFFFFF;
	private static final int FIXED_NEG_INF = 0x80000000;
	private static final int FIX_SHIFT = 8;
	private static final int FIX_16 = 0x1000;
	private static final int FIX_ONE = 0x100;
	private static final int FIX_HALF = 0x80;
	private static final int FIX_QUARTER = 0x40;
	private static final int FIX_SIXTEENTH = 0x10;

	/* widely used definitions */

	/* values for ClrSeg.sType */
	private static final int S_LINE = 0;
	private static final int S_BEND = 1;
	private static final int S_CURVE = 2;
	private static final int S_GHOST = 3;

	/* values for PathElt.type */
	private static final int MOVETO = 0;
	private static final int LINETO = 1;
	private static final int CURVETO = 2;
	private static final int CLOSEPATH = 3;

	/* values for pathelt control points */
	private static final int CP_START = 0;
	private static final int CP_CURVE1 = 1;
	private static final int CP_CURVE2 = 2;
	private static final int CP_END = 3;

	/* widths of ghost bands */
	private static final int BOTGHST = -21;
	private static final int TOPGHST = -20;

	/* various parameters */
	private static final double THETA = 0.38;				/* must be <= .38 for Ryumin-Light-32 c49*/
	private static final int BEND_TAN = 577;				/* 30 sin 30 cos div abs == .57735 */;
	private static final int S_CURVE_TAN = 25;
	private static final boolean Y_GOES_UP = true;
	private static final int MAXBLUES = 20;
	private static final int MAXSERIFS = 5;
	private static final int MAXSTEMS = 20;
	private static final int MAXFIXES = 100;
	private static final int X0 = 0;
	private static final int Y0 = 0;
	private static final int SFACTOR = 20;
	private static final int SPCBONUS = 1000;
	private static final int MAXSTEMDIST = 150;
	private static final int MAXF = (1 << 15);
	private static final int PRNFCTR = 3;
	private static final int MUCHFCTR = 50;
	private static final int VERYMUCHFCTR = 100;
	private static final int STARTING = 0;
	private static final int GOINGUP = 1;
	private static final int GOINGDOWN = 2;
	private static final double LENGTHRATIOCUTOFF = 0.11;
	private static final int MAXCNT = 100;
	private static final int WD_CCW = 0;					/* winding direction counter-clockwise */
	private static final int WD_CW = 1;					/* winding direction clockwise */
	private static final int MINIFLTNMAXDEPTH = 6;
	private static final int MINIBLKSZ = 10;
	private static final int FRP_NZWIND = 0;
	private static final int FRP_YEXTREME = 1;
	private static final int FRP_CHKDT = 2;
	private static final int FRP_CHKBBDT = 3;
	private static final int FRP_FPBBOXPT = 4;
	private static final int MAX_NUM_SUBPATHS = 200;
	private static final int MAX_NUM_PATH_ELEMENTS = 1000;

	/* stem flag values */
	private static final int AC_GC_VERT_STEM	= 1<<0;
//	private static final int AC_GC_CNTR_STEM	= 1<<1;
	private static final int AC_GC_STEM3_STEM	= 1<<2;
	private static final int AC_GC_NEW_HINTS	= 1<<3;
//	private static final int AC_GC_NEW_GROUP	= 1<<4;

	private static final class ACException extends Exception
	{
		static final long serialVersionUID = 1;

		public ACException()
		{
			super();
		}

		public ACException(String message)
		{
			super(message);
		}

		public ACException(String message, Throwable cause)
		{
			super(message, cause);
		}

		public ACException(Throwable cause)
		{
			super(cause);
		}
	}

	private static final class Cd implements Cloneable
	{
		private int x, y;

		private void copyFrom(Cd src)
		{
			x = src.x;
			y = src.y;
		}

		public Object clone()
		{
			try {
				return super.clone();
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}
	}

	private static final class FltnRec
	{
		private short limit;
		private int/*ACFixed*/ feps;
		private int report;
		private Object param;
		private Cd ll = new Cd(), ur = new Cd();
		private int/*ACFixed*/ llx, lly;
	}

	private static final class ClrSeg
	{
		private int csgSN;
		private ClrSeg sNxt;
		/* points to next ClrSeg in list */
		/* separate lists for top, bottom, left, and right segments */
		private int/*ACFixed*/ sLoc, sMax, sMin;
		/* sLoc is X loc for vertical seg, Y loc for horizontal seg */
		/* sMax and sMin give Y extent for vertical seg, X extent for horizontal */
		/* i.e., sTop = sMax, sBot = sMin, sLft = sMin, sRght = sMax. */
		private int/*ACFixed*/ sBonus;
		/* nonzero for segments in sol-eol subpaths */
		/* (probably a leftover that is no longer needed) */
		private ClrVal sLnk;
		/* points to the best ClrVal that uses this ClrSeg */
		/* set by FindBestValForSegs in pick.c */
		private PathElt sElt;
		/* points to the path element that generated this ClrSeg */
		/* set by AddSegment in gen.c */
		private short sType;
		/* tells what type of segment this is: S_LINE S_BEND S_CURVE or S_GHOST */
	}

	private static final class SegLnk
	{
		private ClrSeg seg;
	}

	private static final class SegLnkLst {
		private SegLnkLst next;
		private SegLnk lnk;
	}

	private static final class ClrVal implements Cloneable
	{
		private int cvlSN;
		private ClrVal vNxt;
		/* points to next ClrVal in list */
		private int/*ACFixed*/ vVal, vSpc, initVal;
		/* vVal is value given in eval.c */
		/* vSpc is nonzero for "special" ClrVals */
		/* such as those with a segment in a blue zone */
		/* initVal is the initially assigned value */
		/* used by FndBstVal in pick.c */
		private int/*ACFixed*/ vLoc1, vLoc2;
		/* vLoc1 is location corresponding to vSeg1 */
		/* vLoc2 is location corresponding to vSeg2 */
		/* for horizontal ClrVal, vBot = vLoc1 and vTop = vLoc2 */
		/* for vertical ClrVal, vLft = vLoc1 and vRght = vLoc2 */
		private boolean vGhst;	/* true iff one of the ClrSegs is a S_GHOST seg */
		private boolean pruned;
		/* flag used by FindBestHVals and FindBestVVals */
		/* and by PruneVVals and PruneHVals */
		private boolean merge;
		/* flag used by ReplaceVals in merge.c */
		private ClrSeg vSeg1, vSeg2;
		/* vSeg1 points to the left ClrSeg in a vertical, bottom in a horizontal */
		/* vSeg2 points to the right ClrSeg in a vertical, top in a horizontal */
		private ClrVal vBst;
		/* points to another ClrVal if this one has been merged or replaced */

		public Object clone()
		{
			try {
				return super.clone();
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}
	}

	private static final class PathElt implements Cloneable
	{
		private int eltSN;
		private PathElt prev, next;
		private short type;
		private SegLnkLst Hs, Vs;
		private boolean Hcopy, Vcopy, isFlex;
		private short count, newcolors;
		private int/*ACFixed*/ x, y, x1, y1, x2, y2, x3, y3;

		public Object clone()
		{
			try {
				return super.clone();
			} catch (CloneNotSupportedException e) {
				return null;
			}
		}
	}

	private static final class ClrPoint
	{
		private int cptSN;
		private ClrPoint next;
		private int/*ACFixed*/ x0, y0, x1, y1;
		/* for vstem, only interested in x0 and x1 */
		/* for hstem, only interested in y0 and y1 */
		private PathElt p0, p1;
		/* p0 is source of x0, y0; p1 is source of x1, y1 */
		private char c;
		/* tells what kind of coloring: 'b' 'y' 'm' or 'v' */
		/* MA -
		 * 'b' - horizontal stem (blue)
		 * 'y' - vertical stem (yellow)
		 * 'm' - vertical counter hint
		 * 'v' - horizontal counter hint
		 */
		private boolean done;
	}

	private static final class ACBBox
	{
		private int/*ACFixed*/ xmin, ymin, xmax, ymax, vMn, vMx, hMn, hMx;
		private PathElt pxmn, pxmx, pymn, pymx, pe, pvMn, pvMx, phMn, phMx;

		void copyFrom(ACBBox src)
		{
			xmin = src.xmin;
			ymin = src.ymin;
			xmax = src.xmax;
			ymax = src.ymax;
			vMn = src.vMn;
			vMx = src.vMx;
			hMn = src.hMn;
			hMx = src.hMx;
			pxmn = src.pxmn;
			pxmx = src.pxmx;
			pymn = src.pymn;
			pymx = src.pymx;
			pe = src.pe;
			pvMn = src.pvMn;
			pvMx = src.pvMx;
			phMn = src.phMn;
			phMx = src.phMx;
		}
	}

	/* sub-path information such as bbox, the current winding direction,
	 * the desired winding direction
	 */
	private static final class SubPathInfo
	{
		private PathElt head;
		private PathElt tail;
		private Cd startPt = new Cd();
		private Cd endPt = new Cd();
		private ACBBox bbox = new ACBBox();
		private int currentWind;
		private boolean filled;
	}

	private static final class FindExParam
	{
		private int/*ACFixed*/ yMax;			/* the current max y value */
		private int windDir;				/* WD_CCW or WD_CW */
		private Cd prerisePt = new Cd();		/* the last point with y less than risePt */
		private Cd risePt = new Cd();			/* the first point with the current max y */
		private Cd lastPt = new Cd();			/* last point */
		private Cd firstPt = new Cd();			/* first point */
		private Cd beforeFirstSinkPt = new Cd();	/* the the last point at the y extreme before sinking (used when firstWasExtreme) */
		private Cd firstSinkPt = new Cd();		/* sink point after the first point */
		private boolean	hasPrerisePt;			/* prerisePt is valid */
		private boolean	hasRisePt;			/* risePt it valid */
		private boolean	hasLastPt;			/* lastPt is valid */
		private boolean	lastWasExtreme;			/* lastPt was at extreme */
		private boolean	firstWasExtreme;		/* there were not enough points when extreme was found (at head) */
		private boolean	hasRisePtForFirst;		/* risePt is for the first point at extreme */
		private boolean plaeauEnded;			/* a sequence of the current extreme points has been ended */
	}

	private static final class NZWindParam
	{
		private int windCount;
		private double testx, testy;
		private Cd testPt = new Cd();
		private Cd lastPt = new Cd();
		private Cd firstPt = new Cd();
		private boolean onLine;
		private boolean fromAbove;
		private boolean hasLastPt;
		private boolean testToRight;
	}

	private static final class CheckData
	{
		private boolean xflat, yflat, xdone, ydone, bbquit;
		private int xstate, ystate, xstart, ystart;
		private int/*ACFixed*/ x0, cy0, x1, cy1, xloc, yloc;
		private int/*ACFixed*/ x, y, xnxt, ynxt;
		private int/*ACFixed*/ yflatstartx, yflatstarty, yflatendx, yflatendy;
		private int/*ACFixed*/ xflatstarty, xflatstartx, xflatendx, xflatendy;
		private boolean vert, started, reCheckSmooth;
		private int/*ACFixed*/ loc, frst, lst;
		private PathElt e;
		private boolean forMultiMaster;
	}

	/* a set of stem hint data */
	private static final class HintData
	{
		private int flags;
		private int/*ACFixed*/ v1, v2;
	}

	/* storage used by WriteGlyph */
	private static final class WriteData
	{
		private boolean firstFlex;
		private boolean wrtColorInfo;
		private ClrPoint bst;
		private char bch;
		private int/*ACFixed*/ bx, by;
		private boolean bstB;
		private Cd fc1 = new Cd(), fc2 = new Cd(), fc3 = new Cd();
		/*To avoid pointless hint subs*/
		private ArrayList hintArray;
		private ArrayList prevHintArray;
	}

	/* Class variables */
	private PathElt mPathStart;
	private PathElt mPathEnd;
	private double mOrigEmSquare;
	private boolean mGenerateVStems;			/* true if vertical stem hints should be generated */
	private boolean mUseV;
	private boolean mUseH;
	private boolean mAutoVFix;
	private boolean mAutoHFix;
	private boolean mEditChar;				/* whether character can be modified when adding hints */
	private boolean mScalinghints;
	private boolean mExtracolor;				/* if true, hint substitution is enabled */
	private boolean mFixWinding;				/* if true, fix the path winding direction */
	private boolean mFlexStrict;
	private boolean mFlexOK;
	private boolean mDoSmoothing;
	private boolean mDoCounters;
	private int/*ACFixed*/ mHBigDist;
	private int/*ACFixed*/ mVBigDist;
	private int/*ACFixed*/ mInitBigDist;
	private int/*ACFixed*/ mMinDist;
	private int/*ACFixed*/ mGhostWidth;
	private int/*ACFixed*/ mGhostLength;
	private int/*ACFixed*/ mBendLength;
	private int/*ACFixed*/ mBandMargin;
	private int/*ACFixed*/ mMaxFlare;
	private int/*ACFixed*/ mMaxBendMerge;
	private int/*ACFixed*/ mMaxMerge;
	private int/*ACFixed*/ mMinColorElementLength;
	private int/*ACFixed*/ mFlexCand;
	private int/*ACFixed*/ mBluefuzz;
	private int mUnicode;					/* if non-zero, Unicode of the glyph */
	private int/*ACFixed*/ mPruneA;
	private int/*ACFixed*/ mPruneB;
	private int/*ACFixed*/ mPruneC;
	private int/*ACFixed*/ mPruneD;
	private int/*ACFixed*/ mPruneValue;
	private int/*ACFixed*/ mBonus;
	private double mHBigDistR;
	private double mVBigDistR;
	private double mMaxVal;
	private double mMinVal;
	private int mDMIN;					/* flex parameters (constants) */
	private int mCPpercent;					/* constants */
	private ClrVal mHcoloring;
	private ClrVal mHprimary;
	private ClrVal mValList;
	private ClrVal mVcoloring;
	private ClrVal mVprimary;
	private ClrSeg[] mSegLists = new ClrSeg[4];		/* left, right, top, bot */
	private ClrPoint mPointList;
	private ClrPoint[] mPtLstArray;
	private int mPtLstIndex;
	private int mNumPtLsts;
	private int mMaxPtLsts;
	private int/*ACFixed*/[] mTopBands = new int[MAXBLUES];
	private int/*ACFixed*/[] mBotBands = new int[MAXBLUES];
	private int/*ACFixed*/[] mSerifs = new int[MAXSERIFS];
	private int mLenTopBands;
	private int mLenBotBands;
	private int mNumSerifs;
	private int/*ACFixed*/[] mVStems = new int[MAXSTEMS];
	private int/*ACFixed*/[] mHStems = new int[MAXSTEMS];
	private int mNumVStems;
	private int mNumHStems;
	private boolean mDoAligns;				/* for calling-back with zones and/or stems */
	private boolean mDoWriteGlyph;				/* if true a hinted glyph is returned to the client */
	private boolean mCounterFailed;
	private SegLnkLst mHlnks;
	private SegLnkLst mVlnks;
	private int mCpFrom;
	private int mCpTo;
	private int/*ACFixed*/[] mHFixYs = new int[MAXFIXES];
	private int/*ACFixed*/[] mHFixDYs = new int[MAXFIXES];
	private int mHFixCount;
	private int/*ACFixed*/[] mVFixXs = new int[MAXFIXES];
	private int/*ACFixed*/[] mVFixDXs = new int[MAXFIXES];
	private int mVFixCount;
	private boolean mClrBBox;
	private boolean mClrHBounds;
	private boolean mClrVBounds;
	private boolean mHaveHBnds;
	private boolean mMergeMain;
	private boolean mHaveVBnds;
	private ACBBox mBBox = new ACBBox();
	private byte[] mLinks;
	private int mRowCnt;
	private boolean mSubPathOpen;
	private PathElt mInputPathStart;
	private PathElt mInputPathEnd;
	private double mGlyphWidth;
	private int mNumSubPaths;
	private SubPathInfo[] mSubPaths;
	private ACBBox mPathBBox;
	private boolean	mAllStems;			/* if false, then stems defined by curves are excluded from the reporting */
	private int[] mTopZones;			/* an array of top zone value pairs */
	private int[] mBottomZones;			/* an array of values of bottom zone value pairs */
	private HintedOutlineConsumer mOutlineConsumer;
	private int mEltSN;
	private int mCptSN;
	private int mCvlSN;
	private int mCsgSN;
	private double mCurX;
	private double mCurY;
	private ArrayList mZoneData = new ArrayList();

	/* Module ACinterface.c */

	public AutoColor(HintedOutlineConsumer outlineConsumer, double unitsPerEm,
			 int flags, boolean doWriteGlyph, boolean doAligns,
			 int[] topZones, int[] bottomZones)
	{
		mOutlineConsumer = outlineConsumer;
		mOrigEmSquare = unitsPerEm;
		mDoWriteGlyph = doWriteGlyph;						/* True in AutoColorString mode, false in ReportZones mode */
		mDoAligns = doAligns;							/* False in AutoColorString mode, true in ReportZones mode */
		mTopZones = (topZones != null) ? topZones : new int[0];			/* Non-null in AutoColorString mode */
		mBottomZones = (bottomZones != null) ? bottomZones : new int[0];	/* Non-null in AutoColorString mode */
		mScalinghints = true;							/* Always true in AC.c */
		mAllStems = true;							/* Always set true by CFFHinter.cpp */
		mAutoVFix = mAutoHFix = false;						/* Always false in AC.c */
		mSubPathOpen = false;
		mExtracolor = (flags & AC_HINTSUB) != 0;
		mEditChar = (flags & AC_ALLOWEDIT) != 0;
		mFixWinding = (flags & AC_FIXWINDING) != 0;
		mGenerateVStems = (flags & AC_GENERATEVSTEMS) != 0;
		copyBands();
		initAll();
	}

	public void newGlyph(int glyphID, double width, int unicode)
	{
		mGlyphWidth = width;
		mUnicode = unicode;
		mInputPathStart = null;
		mInputPathEnd = null;
		mZoneData = new ArrayList();
	}

	public int[] reportZones()
	{
		int[] intArray = new int[mZoneData.size()];
		for (int i = 0; i < intArray.length; i++) {
			intArray[i] = ((Integer)mZoneData.get(i)).intValue();
		}
		return intArray;
	}

	/* Module AC.c */

	private void initData()
	{
		double tmp;
		mDMIN = 50;
		mInitBigDist = psDist(MAXSTEMDIST);
		/* must be <= 168 for ITC Garamond Book Italic p, q, thorn */
		mMinDist = psDist(7);
		mGhostWidth = psDist(20);
		mGhostLength = psDist(4);
		mBendLength = psDist(2);
		mPruneA = fixInt(50);
		mPruneC = 100;
		mPruneD = FIX_ONE;
		tmp = 10.24;			/* set to 1024 times the threshold value */
		mPruneValue = mPruneB = fltToFix(tmp);
		/* mPruneB must be <= .01 for Yakout/Light/heM */
		/* mPruneValue must be <= .01 for Yakout/Light/heM */
		mCPpercent = 40;
		/* must be < 46 for Americana-Bold d bowl vs stem coloring */
		mBandMargin = psDist(30);
		mMaxFlare = psDist(10);
		mMaxBendMerge = psDist(6);
		mMaxMerge = psDist(2);		/* must be < 3 for Cushing-BookItalic z */
		mMinColorElementLength = psDist(12);
		mFlexCand = psDist(4);
		mMaxVal = 8000000.0;
		mMinVal = 1.0 / FIX_ONE;
		mAutoHFix = mAutoVFix = false;
		/* Default is to change a curve with collinear points into a line. */
		mFlexOK = false;
		mFlexStrict = false;
		mDoSmoothing = false;
		mDoCounters = false;
		mBluefuzz = (int/*ACFixed*/)(mOrigEmSquare / 2000.0);	/* .5 pixel */
		mPointList = null;
		mMaxPtLsts = 5;
		mPtLstArray = new ClrPoint[mMaxPtLsts];
		mPtLstIndex = 0;
		mPtLstArray[0] = null;
		mNumPtLsts = 1;
		mNumSubPaths = 0;
		mSubPaths = null;
		mPathBBox = null;
		mEltSN = 0;
		mCptSN = 0;
		mCvlSN = 0;
		mCsgSN = 0;
	}

	private int/*ACFixed*/ scaleAbs(int/*ACFixed*/ unscaled)
	{
		if (!mScalinghints) {
			return unscaled;
		}
		return (int/*ACFixed*/)(1000.0 / mOrigEmSquare * unscaled);
	}

	private int/*ACFixed*/ unScaleAbs(int/*ACFixed*/ scaled)
	{
		int/*ACFixed*/ temp1;
		if (!mScalinghints) {
			return scaled;
		}
		temp1 = (int/*ACFixed*/)(mOrigEmSquare / 1000.0 * scaled);
		temp1 = fRnd(temp1);
		return temp1;
	}

	/* Module ACauto.c */

	private void initAuto()
	{
		mClrBBox = mClrHBounds = mHaveHBnds = false;
		mClrVBounds = mHaveVBnds = false;
	}

	private PathElt getSubPathNxt(PathElt e)
	{
		if (e.type == CLOSEPATH) {
			return getDest(e);
		}
		return e.next;
	}

	private PathElt getSubPathPrv(PathElt e)
	{
		if (e.type == MOVETO) {
			e = getClosedBy(e);
		}
		return e.prev;
	}

	private ClrVal findClosestVal(ClrVal sLst, int/*ACFixed*/ loc)
	{
		int/*ACFixed*/ dist = fixInt(10000), bot, top, d;
		ClrVal best = null;
		while (sLst != null) {
			bot = sLst.vLoc1;
			top = sLst.vLoc2;
			if (bot > top) {
				int/*ACFixed*/ tmp = bot;
				bot = top;
				top = tmp;
			}
			if (loc >= bot && loc <= top) {
				best = sLst;
				break;
			}
			if (loc < bot) {
				d = bot - loc;
			} else {
				d = loc - top;
			}
			if (d < dist) {
				dist = d;
				best = sLst;
			}
			sLst = sLst.vNxt;
		}
		return best;
	}

	private void cpyHClr(PathElt e)
	{
		ClrVal best;
		Cd endPoint = new Cd();
		getEndPoint(e, endPoint);
		best = findClosestVal(mHprimary, endPoint.y);
		if (best != null) {
			addHPair(best, 'b');
		}
	}

	private void cpyVClr(PathElt e)
	{
		ClrVal best;
		Cd endPoint = new Cd();
		getEndPoint(e, endPoint);
		best = findClosestVal(mVprimary, endPoint.x);
		if (best != null) {
			addVPair(best, 'y');
		}
	}

	private void pruneColorSegs(PathElt e, boolean hFlg)
	{
		SegLnkLst lst, nxt, prv;
		SegLnk lnk;
		ClrSeg seg;
		ClrVal val;
		lst = hFlg ? e.Hs : e.Vs;
		prv = null;
		while (lst != null) {
			val = null;
			lnk = lst.lnk;
			if (lnk != null) {
				seg = lnk.seg;
				if (seg != null) {
					val = seg.sLnk;
				}
			}
			nxt = lst.next;
			if (val == null) {
				/* prune this one */
				if (prv == null) {
					if (hFlg) {
						e.Hs = nxt;
					} else {
						e.Vs = nxt;
					}
				} else {
					prv.next = nxt;
				}
				lst = nxt;
			} else {
				prv = lst;
				lst = nxt;
			}
		}
	}

	private void pruneElementColorSegs()
	{
		PathElt e;
		e = mPathStart;
		while (e != null) {
			pruneColorSegs(e, true);
			pruneColorSegs(e, false);
			e = e.next;
		}
	}

	private SegLnkLst elmntClrSegLst(PathElt e, boolean hFlg)
	{
		return (hFlg) ? e.Hs : e.Vs;
	}

	private void remLnk(PathElt e, boolean hFlg, SegLnkLst rm)
	{
		SegLnkLst lst, prv, nxt;
		lst = hFlg ? e.Hs : e.Vs;
		prv = null;
		while (lst != null) {
			nxt = lst.next;
			if (lst == rm) {
				if (prv == null) {
					if (hFlg) {
						e.Hs = nxt;
					} else {
						e.Vs = nxt;
					}
				} else {
					prv.next = nxt;
				}
				return;
			}
			prv = lst;
			lst = nxt;
		}
	}

	private boolean alreadyOnList(ClrVal v, ClrVal lst)
	{
		while (lst != null) {
			if (v == lst) {
				return true;
			}
			lst = lst.vNxt;
		}
		return false;
	}

	private void autoVSeg(ClrVal sLst)
	{
		addVPair(sLst, 'y');
	}

	private void autoHSeg(ClrVal sLst)
	{
		addHPair(sLst, 'b');
	}

	private void addHColoring(ClrVal h)
	{
		if (mUseH || alreadyOnList(h, mHcoloring)) {
			return;
		}
		h.vNxt = mHcoloring;
		mHcoloring = h;
		autoHSeg(h);
	}

	private void addVColoring(ClrVal v)
	{
		if (mUseV || alreadyOnList(v, mVcoloring)) {
			return;
		}
		v.vNxt = mVcoloring;
		mVcoloring = v;
		autoVSeg(v);
	}

	private int testColor(ClrSeg s, ClrVal colorList, boolean flg, boolean doLst)
	{
		/* -1 means already in colorList; 0 means conflicts; 1 means ok to add */
		ClrVal v, clst;
		int/*ACFixed*/ top, bot, cTop, cBot, vT, vB, loc;
		boolean loc1;
		if (s == null) {
			return -1;
		}
		v = s.sLnk;
		loc = s.sLoc;
		if (v == null) {
			return -1;
		}
		vT = top = v.vLoc2;
		vB = bot = v.vLoc1;
		if (v.vGhst) {
			/* collapse width for conflict test */
			if (v.vSeg1.sType == S_GHOST) {
				bot = top;
			} else {
				top = bot;
			}
		}
		if (v.vGhst) {
			/* if best value for segment uses a ghost, and
			   segment loc is already in the colorList, then return -1 */
			clst = colorList;
			if (Math.abs(loc - vT) < Math.abs(loc - vB)) {
				loc1 = false;
				loc = vT;
			} else {
				loc1 = true;
				loc = vB;
			}
			while (clst != null) {
				if ((loc1 ? clst.vLoc1 : clst.vLoc2) == loc) {
					return -1;
				}
				clst = clst.vNxt;
				if (!doLst) {
					break;
				}
			}
		}
		if (flg) {
			top += mBandMargin;
			bot -= mBandMargin;
		} else {
			top -= mBandMargin;
			bot += mBandMargin;
		}
		while (colorList != null) {
			/* check for conflict */
			cTop = colorList.vLoc2;
			cBot = colorList.vLoc1;
			if (vB == cBot && vT == cTop) {
				return -1;
			}
			if (colorList.vGhst) {
				/* collapse width for conflict test */
				if (colorList.vSeg1.sType == S_GHOST) {
					cBot = cTop;
				} else {
					cTop = cBot;
				}
			}
			if ((flg && (cBot <= top) && (cTop >= bot)) || (!flg && (cBot >= top) && (cTop <= bot))) {
				return 0;
			}
			colorList = colorList.vNxt;
			if (!doLst) {
				break;
			}
		}
		return 1;
	}

	private int testColorLst(SegLnkLst lst, ClrVal colorList, boolean flg, boolean doLst)
	{
		/* -1 means already in colorList; 0 means conflicts; 1 means ok to add */
		int result, i, cnt;
		result = -1;
		cnt = 0;
		while (lst != null) {
			i = testColor(lst.lnk.seg, colorList, flg, doLst);
			if (i == 0) {
				result = 0;
				break;
			}
			if (i == 1) {
				result = 1;
			}
			lst = lst.next;
			if (++cnt > 100) {
				return 0;
			}
		}
		return result;
	}

	private boolean resolveConflictBySplit(PathElt e, boolean Hflg, SegLnkLst lnk1, SegLnkLst lnk2)
	{
		/* insert new pathelt immediately following e */
		/* e gets first half of split; new gets second */
		/* e gets lnk1 in Hs or Vs; new gets lnk2 */
		PathElt newPE;
		Cd d0 = new Cd(), d1 = new Cd(), d2 = new Cd(), d3 = new Cd(), d4 = new Cd(), d5 = new Cd(), d6 = new Cd(), d7 = new Cd();
		if (e.type != CURVETO || e.isFlex) {
			return false;
		}
		newPE = new PathElt();
		newPE.next = e.next;
		e.next = newPE;
		newPE.prev = e;
		if (newPE.next == null) {
			mPathEnd = newPE;
		} else {
			newPE.next.prev = newPE;
		}
		if (Hflg) {
			e.Hs = lnk1;
			newPE.Hs = lnk2;
		} else {
			e.Vs = lnk1;
			newPE.Vs = lnk2;
		}
		if (lnk1 != null) {
			lnk1.next = null;
		}
		if (lnk2 != null) {
			lnk2.next = null;
		}
		newPE.type = CURVETO;
		getEndPoint(e.prev, d0);
		d1.x = e.x1;
		d1.y = e.y1;
		d2.x = e.x2;
		d2.y = e.y2;
		d3.x = e.x3;
		d3.y = e.y3;
		d4.copyFrom(d0);
		d5.copyFrom(d1);
		d6.copyFrom(d2);
		d7.copyFrom(d3);
		newPE.x3 = d3.x;
		newPE.y3 = d3.y;
		d2.x = (d6.x + d7.x) >> 1;
		d2.y = (d6.y + d7.y) >> 1;
		d7.x = (d5.x + d6.x) >> 1;
		d7.y = (d5.y + d6.y) >> 1;
		d5.x = (d4.x + d5.x) >> 1;
		d5.y = (d4.y + d5.y) >> 1;
		d6.x = (d5.x + d7.x) >> 1;
		d6.y = (d5.y + d7.y) >> 1;
		d1.x = (d7.x + d2.x) >> 1;
		d1.y = (d7.y + d2.y) >> 1;
		d7.x = (d6.x + d1.x) >> 1;
		d7.y = (d6.y + d1.y) >> 1;
		e.x1 = d5.x;
		e.y1 = d5.y;
		e.x2 = d6.x;
		e.y2 = d6.y;
		e.x3 = d7.x;
		e.y3 = d7.y;
		newPE.x1 = d1.x;
		newPE.y1 = d1.y;
		newPE.x2 = d2.x;
		newPE.y2 = d2.y;
		return true;
	}

	private void remDupLnks(PathElt e, boolean Hflg)
	{
		SegLnkLst l1, l2, l2nxt;
		l1 = Hflg ? e.Hs : e.Vs;
		while (l1 != null) {
			l2 = l1.next;
			while (l2 != null) {
				l2nxt = l2.next;
				if (l1.lnk.seg == l2.lnk.seg) {
					remLnk(e, Hflg, l2);
				}
				l2 = l2nxt;
			}
			l1 = l1.next;
		}
	}

	private boolean okToRemLnk(int/*ACFixed*/ loc, boolean Hflg, int/*ACFixed*/ spc)
	{
		if (!Hflg || spc == 0) {
			return true;
		}
		if (inBlueBand(loc, mLenTopBands, mTopBands)) {
			return false;
		}
		if (inBlueBand(loc, mLenBotBands, mBotBands)) {
			return false;
		}
		return true;
	}

	/* The changes made here were to fix a problem in MinisterLight/E.
	   The top left point was not getting colored. */
	private boolean tryResolveConflict(PathElt e, boolean Hflg)
	{
		int typ;
		SegLnkLst lst, lnk1, lnk2;
		ClrSeg seg, seg1, seg2;
		ClrVal val1, val2;
		int/*ACFixed*/ lc1, lc2, loc0, loc1, loc2, loc3;
		Cd p0 = new Cd();
		Cd p1 = new Cd();
		remDupLnks(e, Hflg);
		typ = e.type;
		if (typ == MOVETO) {
			getEndPoints(getClosedBy(e), p0, p1);
		} else if (typ == CURVETO) {
			p0.x = e.x1;
			p0.y = e.y1;
			p1.x = e.x3;
			p1.y = e.y3;
		} else {
			getEndPoints(e, p0, p1);
		}
		loc1 = Hflg ? p0.y : p0.x;
		loc2 = Hflg ? p1.y : p1.x;
		lst = Hflg ? e.Hs : e.Vs;
		seg1 = lst.lnk.seg;
		lc1 = seg1.sLoc;
		lnk1 = lst;
		lst = lst.next;
		seg2 = lst.lnk.seg;
		lc2 = seg2.sLoc;
		lnk2 = lst;
		if (lc1!= loc1 && lc2!= loc2) {
			if (Math.abs(lc1 - loc1) > Math.abs(lc1 - loc2) ||
			    Math.abs(lc2 - loc2) > Math.abs(lc2 - loc1)) {
				seg = seg1;
				seg1 = seg2;
				seg2 = seg;
				lst = lnk1;
				lnk1 = lnk2;
				lnk2 = lst;
			}
		}
		val1 = seg1.sLnk;
		val2 = seg2.sLnk;
		if (val1.vVal < fixInt(50) && okToRemLnk(loc1, Hflg, val1.vSpc)) {
			remLnk(e, Hflg, lnk1);
			return true;
		}
		if (val2.vVal < fixInt(50) && val1.vVal > val2.vVal * 20 && okToRemLnk(loc2, Hflg, val2.vSpc)) {
			remLnk(e, Hflg, lnk2);
			return true;
		}
		if (typ != CURVETO ||
		    ((((Hflg && isHorizontal(p0.x, p0.y, p1.x, p1.y)) ||
		       (!Hflg && isVertical(p0.x, p0.y, p1.x, p1.y)))) &&
		     okToRemLnk(loc1, Hflg, val1.vSpc))) {
			remLnk(e, Hflg, lnk1);
			return true;
		}
		getEndPoints(getSubPathPrv(e), p0, p1);
		loc0 = Hflg ? p0.y : p0.x;
		if (prodLt0(loc2-loc1, loc0-loc1)) {
			remLnk(e, Hflg, lnk1);
			return true;
		}
		getEndPoint(getSubPathNxt(e), p1);
		loc3 = Hflg ? p1.y : p1.x;
		if (prodLt0(loc3-loc2, loc1-loc2)) {
			remLnk(e, Hflg, lnk2);
			return true;
		}
		if ((loc2 == val2.vLoc1 || loc2 == val2.vLoc2) &&
		    loc1 != val1.vLoc1 && loc1 != val1.vLoc2) {
			remLnk(e, Hflg, lnk1);
			return true;
		}
		if ((loc1 == val1.vLoc1 || loc1 == val1.vLoc2) &&
		    loc2 != val2.vLoc1 && loc2 != val2.vLoc2) {
			remLnk(e, Hflg, lnk2);
			return true;
		}
		if (mEditChar && resolveConflictBySplit(e, Hflg, lnk1, lnk2)) {
			return true;
		}
		return false;
	}

	private boolean checkColorSegs(PathElt e, boolean flg, boolean Hflg)
	{
		SegLnkLst lst;
		SegLnkLst lst2;
		ClrSeg seg;
		ClrVal val;
		lst = Hflg ? e.Hs : e.Vs;
		while (lst != null) {
			lst2 = lst.next;
			if (lst2 != null) {
				seg = lst.lnk.seg;
				val = seg.sLnk;
				if (val != null && testColorLst(lst2, val, flg, false) == 0) {
					if (tryResolveConflict(e, Hflg)) {
						return checkColorSegs(e, flg, Hflg);
					}
					if (Hflg) {
						e.Hs = null;
					} else {
						e.Vs = null;
					}
					return true;
				}
			}
			lst = lst2;
		}
		return false;
	}

	private void checkElmntClrSegs()
	{
		PathElt e;
		e = mPathStart;
		while (e != null) {
			if (!checkColorSegs(e, Y_GOES_UP, true)) {
				checkColorSegs(e, true, false);
			}
			e = e.next;
		}
	}

	private boolean clrLstsClash(SegLnkLst lst1, SegLnkLst lst2, boolean flg)
	{
		ClrSeg seg;
		ClrVal val;
		SegLnkLst lst;
		while (lst1 != null) {
			seg = lst1.lnk.seg;
			val = seg.sLnk;
			if (val != null) {
				lst = lst2;
				while (lst != null) {
					if (testColorLst(lst, val, flg, false) == 0) {
						return true;
					}
					lst = lst.next;
				}
			}
			lst1 = lst1.next;
		}
		return false;
	}

	private SegLnkLst bestFromLsts(SegLnkLst lst1, SegLnkLst lst2)
	{
		SegLnkLst lst, bst;
		ClrSeg seg;
		ClrVal val;
		int/*ACFixed*/ bstval;
		int i;
		bst = null;
		bstval = 0;
		for (i = 0; i < 2; i++) {
			lst = (i != 0) ? lst1 : lst2;
			while (lst != null) {
				seg = lst.lnk.seg;
				val = seg.sLnk;
				if (val != null && val.vVal > bstval) {
					bst = lst;
					bstval = val.vVal;
				}
				lst = lst.next;
			}
		}
		return bst;
	}

	private boolean clrsClash(PathElt e, PathElt p, SegLnkLst[] hLst, SegLnkLst[] vLst, SegLnkLst[] phLst, SegLnkLst[] pvLst)
	{
		boolean clash = false;
		SegLnkLst bst, newLst;
		if (clrLstsClash(hLst[0], phLst[0], Y_GOES_UP)) {
			clash = true;
			bst = bestFromLsts(hLst[0], phLst[0]);
			if (bst != null) {
				newLst = new SegLnkLst();
				newLst.next = null;
				newLst.lnk = bst.lnk;
			} else {
				newLst = null;
			}
			e.Hs = p.Hs = hLst[0] = phLst[0] = newLst;
		}
		if (clrLstsClash(vLst[0], pvLst[0], true)) {
			clash = true;
			bst = bestFromLsts(vLst[0], pvLst[0]);
			if (bst != null) {
				newLst = new SegLnkLst();
				newLst.next = null;
				newLst.lnk = bst.lnk;
			} else {
				newLst = null;
			}
			e.Vs = p.Vs = vLst[0] = pvLst[0] = newLst;
		}
		return clash;
	}

	private void getColorLsts(PathElt e, SegLnkLst[] phLst, SegLnkLst[] pvLst, int[] ph, int[] pv)
	{
		SegLnkLst hLst, vLst;
		int h, v;
		if (mUseH) {
			hLst = null;
			h = -1;
		} else {
			hLst = e.Hs;
			if (hLst == null) {
				h = -1;
			} else {
				h = testColorLst(hLst, mHcoloring, Y_GOES_UP, true);
			}
		}
		if (mUseV) {
			vLst = null;
			v = -1;
		} else {
			vLst = e.Vs;
			if (vLst == null) {
				v = -1;
			} else {
				v = testColorLst(vLst, mVcoloring, true, true);
			}
		}
		pvLst[0] = vLst;
		phLst[0] = hLst;
		ph[0] = h;
		pv[0] = v;
	}

	private void reClrBounds(PathElt e)
	{
		if (!mUseH) {
			if (mClrHBounds && mHcoloring == null && !mHaveHBnds) {
				reClrHBnds(mBBox);
			} else if (!mClrBBox) {
				if (mHcoloring == null) {
					cpyHClr(e);
				}
				if (mMergeMain) {
					mergeFromMainColors('b');
				}
			}
		}
		if (!mUseV) {
			if (mClrVBounds && mVcoloring == null && !mHaveVBnds) {
				reClrVBnds(mBBox);
			} else if (!mClrBBox) {
				if (mVcoloring == null) {
					cpyVClr(e);
				}
				if (mMergeMain) {
					mergeFromMainColors('y');
				}
			}
		}
	}

	private void addColorLst(SegLnkLst lst, boolean vert)
	{
		ClrVal val;
		ClrSeg seg;
		while (lst != null) {
			seg = lst.lnk.seg;
			val = seg.sLnk;
			if (vert) {
				addVColoring(val);
			} else {
				addHColoring(val);
			}
			lst = lst.next;
		}
	}

	private void startNewColoring(PathElt e, SegLnkLst hLst, SegLnkLst vLst)
	{
		reClrBounds(e);
		xtraClrs(e);
		mClrBBox = false;
		if (mDoCounters && mUseV) {
			copyMainV();
		}
		mVcoloring = null;
		if (mDoCounters && mUseH) {
			copyMainH();
		}
		mHcoloring = null;
		if (!mUseH) {
			addColorLst(hLst, false);
		}
		if (!mUseV) {
			addColorLst(vLst, true);
		}
	}

	private boolean isIn(int h, int v)
	{
		return (h == -1 && v == -1);
	}

	private boolean isOk(int h, int v)
	{
		return (h != 0 && v != 0);
	}

	private void addIfNeedV(int v, SegLnkLst vLst)
	{
		if (!mUseV && v == 1) {
			addColorLst(vLst, true);
		}
	}

	private void addIfNeedH(int h, SegLnkLst hLst)
	{
		if (!mUseH && h == 1) {
			addColorLst(hLst, false);
		}
	}

	private void setHColors(ClrVal lst)
	{
		if (mUseH) {
			return;
		}
		mHcoloring = lst;
		while (lst != null) {
			autoHSeg(lst);
			lst = lst.vNxt;
		}
	}

	private void setVColors(ClrVal lst)
	{
		if (mUseV) {
			return;
		}
		mVcoloring = lst;
		while (lst != null) {
			autoVSeg(lst);
			lst = lst.vNxt;
		}
	}

	private ClrVal copyClrs(ClrVal lst)
	{
		ClrVal v, vlst;
		int cnt;
		vlst = null;
		cnt = 0;
		while (lst != null) {
			v = (ClrVal)lst.clone();
			v.vNxt = vlst;
			vlst = v;
			if (++cnt > 100) {
				return vlst;
			}
			lst = lst.vNxt;
		}
		return vlst;
	}

	private boolean isFlare(int/*ACFixed*/ loc, PathElt e, PathElt n, boolean Hflg)
	{
		Cd p = new Cd();
		while (e != n) {
			getEndPoint(e, p);
			if ((Hflg && Math.abs(p.y - loc) > mMaxFlare) || (!Hflg && Math.abs(p.x - loc) > mMaxFlare)) {
				return false;
			}
			e = getSubPathNxt(e);
		}
		return true;
	}

	private boolean isTopSegOfVal(int/*ACFixed*/ loc, int/*ACFixed*/ top, int/*ACFixed*/ bot)
	{
		int/*ACFixed*/ d1, d2;
		d1 = top-loc;
		d2 = bot-loc;
		return (Math.abs(d1) <= Math.abs(d2)) ? true : false;
	}

	private void remFlareLnk(PathElt e, boolean hFlg, SegLnkLst rm, PathElt e2, int i)
	{
		remLnk(e, hFlg, rm);
	}

	private boolean compareValues(ClrVal val1, ClrVal val2, int factor, int ghstshift)
	{
		int/*ACFixed*/ v1 = val1.vVal, v2 = val2.vVal, mx;
		mx = v1 > v2 ? v1 : v2;
		mx <<= 1;
		while (mx > 0) {
			mx <<= 1;
			v1 <<= 1;
			v2 <<= 1;
		}
		if (ghstshift > 0 && val1.vGhst != val2.vGhst) {
			if (val1.vGhst) {
				v1 >>= ghstshift;
			}
			if (val2.vGhst) {
				v2 >>= ghstshift;
			}
		}
		if ((val1.vSpc > 0 && val2.vSpc > 0) || (val1.vSpc == 0 && val2.vSpc == 0)) {
			return v1 > v2;
		}
		if (val1.vSpc > 0) {
			return (v1 < FIXED_POS_INF / factor) ? (v1 * factor > v2) : (v1 > v2 / factor);
		}
		return (v2 < FIXED_POS_INF/factor) ? (v1 > v2 * factor) : (v1 / factor > v2);
	}

	private void remFlares(boolean Hflg)
	{
		SegLnkLst lst1, lst2, nxt1, nxt2;
		PathElt e, n;
		ClrSeg seg1, seg2;
		ClrVal val1, val2;
		int/*ACFixed*/ diff;
		boolean nxtE;
		boolean Nm1, Nm2;
		if (Hflg) {
			Nm1 = true;
			Nm2 = false;
		} else {
			Nm1 = false;
			Nm2 = true;
		}
		e = mPathStart;
		while (e != null) {
			if (Nm1 ? e.Hs == null : e.Vs == null) {
				e = e.next;
				continue;
			}
			/* e now is an element with Nm1 prop */
			n = getSubPathNxt(e);
			nxtE = false;
			while (n != e && !nxtE) {
				if (Nm1 ? n.Hs != null : n.Vs != null) {
					lst1 = elmntClrSegLst(e, Nm1);
					while (lst1 != null) {
						seg1 = lst1.lnk.seg;
						nxt1 = lst1.next;
						lst2 = elmntClrSegLst(n, Nm1);
						while (lst2 != null) {
							seg2 = lst2.lnk.seg;
							nxt2 = lst2.next;
							if (seg1 != null && seg2 != null) {
								diff = seg1.sLoc - seg2.sLoc;
								if (Math.abs(diff) > mMaxFlare ||
								    !isFlare(seg1.sLoc, e, n, Hflg)) {
									nxtE = true;
									lst2 = nxt2;
									continue;
								}
								val1 = seg1.sLnk;
								val2 = seg2.sLnk;
								if (diff != 0 &&
								    isTopSegOfVal(seg1.sLoc, val1.vLoc2, val1.vLoc1) ==
								    isTopSegOfVal(seg2.sLoc, val2.vLoc2, val2.vLoc1)) {
									if (compareValues(val1, val2, SPCBONUS, 0)) {
										/* This change was made to fix flares in Bodoni2. */
										if (val2.vSpc == 0 && val2.vVal < fixInt(1000)) {
											remFlareLnk(n, Nm1, lst2, e, 1);
										}
									} else if (val1.vSpc == 0 && val1.vVal < fixInt(1000)) {
										remFlareLnk(e, Nm1, lst1, n, 2);
										break;
									}
								}
							}
							lst2 = nxt2;
						}
						lst1 = nxt1;
					}
				}
				if (Nm2 ? n.Hs != null : n.Vs != null) {
					break;
				}
				n = getSubPathNxt(n);
			}
			e = e.next;
		}
	}

	private void carryIfNeed(int/*ACFixed*/ loc, boolean vert, ClrVal clrs)
	{
		ClrSeg seg;
		ClrVal seglnk;
		int/*ACFixed*/ l0, l1, tmp, halfMargin;
		if ((vert && mUseV) || (!vert && mUseH)) {
			return;
		}
		halfMargin = fixHalfMul(mBandMargin);
		if (halfMargin > fixInt(10)) {
			halfMargin = fixInt(10);
		}
		while (clrs != null) {
			seg = clrs.vSeg1;
			if (clrs.vGhst && seg.sType == S_GHOST) {
				seg = clrs.vSeg2;
			}
			if (seg != null) {
				l0 = clrs.vLoc2;
				l1 = clrs.vLoc1;
				if (l0 > l1) {
					tmp = l1;
					l1 = l0;
					l0 = tmp;
				}
				l0 -= halfMargin;
				l1 += halfMargin;
				if (loc > l0 && loc < l1) {
					seglnk = seg.sLnk;
					seg.sLnk = clrs;
					if (vert) {
						if (testColor(seg, mVcoloring, true, true) == 1) {
							addVColoring(clrs);
							seg.sLnk = seglnk;
							break;
						}
					} else {
						if (testColor(seg, mHcoloring, Y_GOES_UP, true) == 1) {
							addHColoring(clrs);
							seg.sLnk = seglnk;
							break;
						}
					}
					seg.sLnk = seglnk;
				}
			}
			clrs = clrs.vNxt;
		}
	}

	private void proClrs(PathElt e, boolean hFlg, int/*ACFixed*/ loc)
	{
		SegLnkLst lst, plst;
		PathElt prv;
		Cd c = new Cd();
		int/*ACFixed*/ dst;
		lst = elmntClrSegLst(e, hFlg);
		if (lst == null) {
			return;
		}
		if (hFlg ? e.Hcopy : e.Vcopy) {
			return;
		}
		prv = e;
		while (true) {
			prv = getSubPathPrv(prv);
			plst = elmntClrSegLst(prv, hFlg);
			if (plst != null) {
				return;
			}
			getEndPoint(prv, c);
			dst = (hFlg ? c.y : c.x) - loc;
			if (Math.abs(dst) > fixInt(50)) {
				return;
			}
			if (hFlg) {
				prv.Hs = lst;
				prv.Hcopy = true;
			} else {
				prv.Vs = lst;
				prv.Vcopy = true;
			}
		}
	}

	private void promoteColors()
	{
		PathElt e;
		Cd c = new Cd();
		e = mPathStart;
		while (e != null) {
			getEndPoint(e, c);
			proClrs(e, true, c.y);
			proClrs(e, false, c.x);
			e = e.next;
		}
	}

	private void remPromotedClrs()
	{
		PathElt e;
		e = mPathStart;
		while (e != null) {
			if (e.Hcopy) {
				e.Hs = null;
				e.Hcopy = false;
			}
			if (e.Vcopy) {
				e.Vs = null;
				e.Vcopy = false;
			}
			e = e.next;
		}
	}

	private void remShortColors()
	{
		/* Must not change colors at a short element. */
		PathElt e;
		Cd cdc = new Cd();
		Cd cde = new Cd();
		e = mPathStart;
		while (e != null) {
			getEndPoint(e, cde);
			if (Math.abs(cdc.x - cde.x) < mMinColorElementLength &&
			    Math.abs(cdc.y - cde.y) < mMinColorElementLength) {
				e.Hs = null;
				e.Vs = null;
			}
			e = e.next;
			cdc.x = cde.x;
			cdc.y = cde.y;
		}
	}

	private void autoExtraColors(boolean movetoNewClrs)
	{
		int[] h = new int[1];
		int[] v = new int[1];
		int[] ph = new int[1];
		int[] pv = new int[1];
		PathElt e, cp, p;
		int etype;
		SegLnkLst[] hLst = new SegLnkLst[1];
		SegLnkLst[] vLst = new SegLnkLst[1];
		SegLnkLst[] phLst = new SegLnkLst[1];
		SegLnkLst[] pvLst = new SegLnkLst[1];
		ClrVal mtHclrs, prvHclrs;
		ClrVal mtVclrs, prvVclrs;
		boolean tst, newClrs = true;
		Cd cd = new Cd();
		mClrBBox = mClrVBounds = mClrHBounds = false;
		mMergeMain = (countSubPaths(null) <= 5);
		e = mPathStart;
		remFlares(true);
		remFlares(false);
		checkElmntClrSegs();
		promoteColors();
		remShortColors();
		mHaveVBnds = mClrVBounds;
		mHaveHBnds = mClrHBounds;
		p = null;
		tst = false;					/* it is ok to add to primary coloring */
		mtVclrs = null;
		mtHclrs = null;
		while (e != null) {
			etype = e.type;
			if (movetoNewClrs && etype == MOVETO) {
				startNewColoring(e, null, null);
				tst = false;
			}
			if (newClrs && e == p) {
				startNewColoring(e, null, null);
				setHColors(mtHclrs);
				if (mGenerateVStems) {
					setVColors(mtVclrs);
				}
				tst = true;
			}
			getColorLsts(e, hLst, vLst, h, v);
			if (etype == MOVETO && isShort(cp = getClosedBy(e))) {
				getColorLsts(p = cp.prev, phLst, pvLst, ph, pv);
				if (clrsClash(e, p, hLst, vLst, phLst, pvLst)) {
					getColorLsts(e, hLst, vLst, h, v);
					getColorLsts(p, phLst, pvLst, ph, pv);
				}
				boolean tstB = tst ? !isIn(ph[0], pv[0]) || !isIn(h[0], v[0]) : !isOk(ph[0], pv[0]) || !isOk(h[0], v[0]);
				if (tstB) {
					startNewColoring(e, hLst[0], vLst[0]);
					tst = false;
					ph[0] = pv[0] = 1;	/* force add of colors for p also */
				} else {
					addIfNeedH(h[0], hLst[0]);
					addIfNeedV(v[0], vLst[0]);
				}
				addIfNeedH(ph[0], phLst[0]);
				addIfNeedV(pv[0], pvLst[0]);
				newClrs = false;		/* so can tell if added new colors in subpath */
			} else {
				boolean tstB = tst ? !isIn(h[0], v[0]) : !isOk(h[0], v[0]);
				if (tstB) {
					/* e needs new coloring */
					if (etype == CLOSEPATH) {
						/* do not attach extra colors to closepath */
						e = e.prev;
						getColorLsts(e, hLst, vLst, h, v);
					}
					prvVclrs = null;
					prvHclrs = copyClrs(mHcoloring);
					if (mGenerateVStems) {
						prvVclrs = copyClrs(mVcoloring);
					}
					if (!newClrs) {
						/* this is the first extra since mt */
						newClrs = true;
						if (mGenerateVStems) {
							mtVclrs = copyClrs(prvVclrs);
						}
						mtHclrs = copyClrs(prvHclrs);
					}
					startNewColoring(e, hLst[0], vLst[0]);
					tst = false;
					if (etype == CURVETO) {
						cd.x = e.x1;
						cd.y = e.y1;
					} else {
						getEndPoint(e, cd);
					}
					carryIfNeed(cd.y, false, prvHclrs);
					if (mGenerateVStems) {
						carryIfNeed(cd.x, true, prvVclrs);
					}
				} else {
					/* do not need to start new coloring */
					addIfNeedH(h[0], hLst[0]);
					addIfNeedV(v[0], vLst[0]);
				}
			}
			e = e.next;
		}
		reClrBounds(mPathEnd);
		remPromotedClrs();
	}

	/* Module ACgen.c */

	private void initGen()
	{
		int i;
		for (i = 0; i < 4; i++) {
			mSegLists[i] = null;
		}
		mHlnks = null;
		mVlnks = null;
	}

	private void linkSegment(PathElt e, boolean Hflg, ClrSeg seg)
	{
		SegLnk newlnk;
		SegLnkLst newlst, globlst;
		newlnk = new SegLnk();
		newlnk.seg = seg;
		newlst = new SegLnkLst();
		globlst = new SegLnkLst();
		globlst.lnk = newlnk;
		newlst.lnk = newlnk;
		if (Hflg) {
			newlst.next = e.Hs;
			e.Hs = newlst;
			globlst.next = mHlnks;
			mHlnks = globlst;
		} else {
			newlst.next = e.Vs;
			e.Vs = newlst;
			globlst.next = mVlnks;
			mVlnks = globlst;
		}
	}

	private void copySegmentLink(PathElt e1, PathElt e2, boolean Hflg)
	{
		/* copy reference to first link from e1 to e2 */
		SegLnkLst newlst;
		newlst = new SegLnkLst();
		if (Hflg) {
			newlst.lnk = e1.Hs.lnk;
			newlst.next = e2.Hs;
			e2.Hs = newlst;
		} else {
			newlst.lnk = e1.Vs.lnk;
			newlst.next = e2.Vs;
			e2.Vs = newlst;
		}
	}

	private void addSegment(int/*ACFixed*/ from, int/*ACFixed*/ to, int/*ACFixed*/ loc,
				int lftLstNm, int rghtLstNm, PathElt e1, PathElt e2, boolean Hflg, int typ)
	{
		ClrSeg seg, segList, prevSeg;
		int segNm;
		seg = new ClrSeg();
		seg.csgSN = ++mCsgSN;
		if (seg.csgSN == 0) mCsgSN++;
		seg.sLoc = loc;
		if (from > to) {
			seg.sMax = from;
			seg.sMin = to;
		} else {
			seg.sMax = to;
			seg.sMin = from;
		}
		seg.sBonus = mBonus;
		seg.sType = (short)typ;
		if (e1 != null) {
			if (e1.type == CLOSEPATH) {
				e1 = getDest(e1);
			}
			linkSegment(e1, Hflg, seg);
			seg.sElt = e1;
		}
		if (e2 != null) {
			if (e2.type == CLOSEPATH) {
				e2 = getDest(e2);
			}
			copySegmentLink(e1, e2, Hflg);
			if (e1 == null || e2 == e1.prev) {
				seg.sElt = e2;
			}
		}
		segNm = (from > to) ? lftLstNm: rghtLstNm;
		segList = mSegLists[segNm];
		prevSeg = null;
		while (true) {
			/* keep list in increasing order by sLoc */
			if (segList == null) {
				/* at end of list */
				if (prevSeg == null) {
					mSegLists[segNm] = seg;
					break;
				}
				prevSeg.sNxt = seg;
				break;
			}
			if (segList.sLoc >= loc) {
				/* insert before this one */
				if (prevSeg == null) {
					mSegLists[segNm] = seg;
				} else {
					prevSeg.sNxt = seg;
				}
				seg.sNxt = segList;
				break;
			}
			prevSeg = segList;
			segList = segList.sNxt;
		}
	}

	private void addVSegment(int/*ACFixed*/ from, int/*ACFixed*/ to, int/*ACFixed*/ loc, PathElt p1, PathElt p2, int typ, int i)
	{
		if (Y_GOES_UP) {
			addSegment(from, to, loc, 0, 1, p1, p2, false, typ);
		} else {
			addSegment(from, to, loc, 1, 0, p1, p2, false, typ);
		}
	}

	private void addHSegment(int/*ACFixed*/ from, int/*ACFixed*/ to, int/*ACFixed*/ loc, PathElt p1, PathElt p2, int typ, int i)
	{
		addSegment(from, to, loc, 2, 3, p1, p2, true, typ);
	}

	private int/*ACFixed*/ cpFrom(int/*ACFixed*/ cp2, int/*ACFixed*/ cp3)
	{
		return ((cp3-cp2)*mCpFrom)/100 + cp2;
	}

	private int/*ACFixed*/ cpTo(int/*ACFixed*/ cp0, int/*ACFixed*/ cp1)
	{
		return ((cp1-cp0)*mCpTo)/100 + cp0;
	}

	private boolean testBend(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, int/*ACFixed*/ x2, int/*ACFixed*/ y2)
	{
		/* return true if bend angle is sharp enough (135 degrees or less) */
		double dx1, dy1, dx2, dy2, dotprod, lensqprod;
		dx1 = fixToFlt(x1 - x0);
		dy1 = fixToFlt(y1 - y0);
		dx2 = fixToFlt(x2 - x1);
		dy2 = fixToFlt(y2 - y1);
		dotprod = dx1*dx2 + dy1*dy2;
		lensqprod = (dx1*dx1 + dy1*dy1) * (dx2*dx2 + dy2*dy2);
		return (dotprod*dotprod / lensqprod) <= .5;
	}

	private boolean testTan(int/*ACFixed*/ d1, int/*ACFixed*/ d2)
	{
		return (Math.abs(d1) > (Math.abs(d2) * BEND_TAN) / 1000);
	}

	private int fRound(int/*ACFixed*/ x)
	{
		return fTrunc(fRnd(x));
	}

	private boolean isCCW(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, int/*ACFixed*/ x2, int/*ACFixed*/ y2)
	{
		/* returns true if (x0, y0) . (x1, y1) . (x2, y2) is counter clockwise in character space */
		int dx0, dy0, dx1, dy1;
		boolean ccw;
		dx0 = fRound(x1-x0);
		dy0 = fRound(y1-y0);
		dx1 = fRound(x2-x1);
		dy1 = fRound(y2-y1);
		if (!Y_GOES_UP) {
			dy0 = -dy0;
			dy1 = -dy1;
		}
		ccw = (dx0*dy1) >= (dx1*dy0);
		return ccw;
	}

	private void doHBendsNxt(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, PathElt p)
	{
		Cd cd2 = new Cd();
		Cd cd3 = new Cd();
		int/*ACFixed*/ delta, strt, end;
		boolean ysame, ccw, above, doboth;
		if (y0 == y1) {
			return;
		}
		nxtForBend(p, cd2, cd3);
		ysame = prodLt0(cd2.y - y1, y1 - y0);			/* y0 and y2 on same side of y1 */
		if (ysame ||
		    (testTan(x1 - cd2.x, y1 - cd2.y) &&
		     (prodLt0(cd2.x - x1, x1 - x0) ||
		      (isVertical(x0, y0, x1, y1) && testBend(x0, y0, x1, y1, cd2.x, cd2.y))))) {
			delta = fixHalfMul(mBendLength);
			doboth = false;
			if (!((x0 <= x1 && x1 < cd2.x) || (x0 < x1 && x1 <= cd2.x))) {
				if ((cd2.x < x1 && x1 <= x0) || (cd2.x <= x1 && x1 < x0)) {
					delta = -delta;
				} else if (ysame) {
					above = y0 > y1;
					if (!Y_GOES_UP) {
						above = !above;
					}
					ccw = isCCW(x0, y0, x1, y1, cd2.x, cd2.y);
					if (above != ccw) {
						delta = -delta;
					}
				} else {
					doboth = true;
				}
			}
			strt = x1 - delta;
			end = x1 + delta;
			addHSegment(strt, end, y1, p, null, S_BEND, 0);
			if (doboth) {
				addHSegment(end, strt, y1, p, null, S_BEND, 1);
			}
		}
	}

	private void doHBendsPrv(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, PathElt p)
	{
		Cd cd2 = new Cd();
		int/*ACFixed*/ delta, strt, end;
		boolean ysame, ccw, above, doboth;
		if (y0 == y1) {
			return;
		}
		prvForBend(p, cd2);
		ysame = prodLt0(cd2.y - y0, y0 - y1);
		if (ysame ||
		    (testTan(x0 - cd2.x, y0 - cd2.y) &&
		     (prodLt0(cd2.x - x0, x0 - x1) ||
		      (isVertical(x0, y0, x1, y1) && testBend(cd2.x, cd2.y, x0, y0, x1, y1))))) {
			delta = fixHalfMul(mBendLength);
			doboth = false;
			if ((cd2.x < x0 && x0 <= x1) || (cd2.x <= x0 && x0 < x1)) {} else {
				if ((x1 < x0 && x0 <= cd2.x) || (x1 <= x0 && x0 < cd2.x)) {
					delta = -delta;
				} else if (ysame) {
					above = (cd2.y > y0);
					if (!Y_GOES_UP) {
						above = !above;
					}
					ccw = isCCW(cd2.x, cd2.y, x0, y0, x1, y1);
					if (above != ccw) {
						delta = -delta;
					}
				}
			}
			strt = x0 - delta;
			end = x0 + delta;
			addHSegment(strt, end, y0, p.prev, null, S_BEND, 2);
			if (doboth) {
				addHSegment(end, strt, y0, p.prev, null, S_BEND, 3);
			}
		}
	}

	private void doVBendsNxt(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, PathElt p)
	{
		Cd cd2 = new Cd();
		Cd cd3 = new Cd();
		int/*ACFixed*/ delta, strt, end;
		boolean xsame, ccw, right, doboth;
		if (x0 == x1) {
			return;
		}
		nxtForBend(p, cd2, cd3);
		xsame = prodLt0(cd2.x - x1, x1 - x0);
		if (xsame ||
		    (testTan(y1 - cd2.y, x1 - cd2.x) &&
		     (prodLt0(cd2.y - y1, y1 - y0) ||
		      (isHorizontal(x0, y0, x1, y1) && testBend(x0, y0, x1, y1, cd2.x, cd2.y))))) {
			delta = fixHalfMul(mBendLength);
			doboth = false;
			if ((y0 <= y1 && y1 < cd2.y) || (y0 < y1 && y1 <= cd2.y)) {} else {
				if ((cd2.y < y1 && y1 <= y0) || (cd2.y <= y1 && y1 < y0)) {
					delta = -delta;
				} else if (xsame) {
					right = x0 > x1;
					ccw = isCCW(x0, y0, x1, y1, cd2.x, cd2.y);
					if (right != ccw) {
						delta = -delta;
					}
					if (!Y_GOES_UP) {
						delta = -delta;
					}
				} else {
					doboth = true;
				}
			}
			strt = y1 - delta;
			end = y1 + delta;
			addVSegment(strt, end, x1, p, null, S_BEND, 0);
			if (doboth) {
				addVSegment(end, strt, x1, p, null, S_BEND, 1);
			}
		}
	}

	private void doVBendsPrv(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, PathElt p)
	{
		Cd cd2 = new Cd();
		int/*ACFixed*/ delta, strt, end;
		boolean xsame, ccw, right, doboth;
		if (x0 == x1) {
			return;
		}
		prvForBend(p, cd2);
		xsame = prodLt0(cd2.x - x0, x0 - x1);
		if (xsame ||
		    (testTan(y0 - cd2.y, x0 - cd2.x) &&
		     (prodLt0(cd2.y - y0, y0 - y1) ||
		      (isHorizontal(x0, y0, x1, y1) && testBend(cd2.x, cd2.y, x0, y0, x1, y1))))) {
			delta = fixHalfMul(mBendLength);
			doboth = false;
			if ((cd2.y < y0 && y0 <= y1) || (cd2.y <= y0 && y0 < y1)) {} else {
				if ((y1 < y0 && y0 <= cd2.y) || (y1 <= y0 && y0 < cd2.y)) {
					delta = -delta;
				} else if (xsame) {
					right = x0 > x1;
					ccw = isCCW(cd2.x, cd2.y, x0, y0, x1, y1);
					if (right != ccw) {
						delta = -delta;
					}
					if (!Y_GOES_UP) {
						delta = -delta;
					}
				}
			}
			strt = y0 - delta;
			end = y0 + delta;
			addVSegment(strt, end, x0, p.prev, null, S_BEND, 2);
			if (doboth) {
				addVSegment(end, strt, x0, p.prev, null, S_BEND, 3);
			}
		}
	}

	/* MA - traversing the global list lst, replace link to seg1 with seg2 */
	private void MergeLnkSegs(ClrSeg seg1, ClrSeg seg2, SegLnkLst lst)
	{
		/* replace lnk refs to seg1 by seg2 */
		SegLnk lnk;
		while (lst != null) {
			lnk = lst.lnk;
			if (lnk.seg == seg1) {
				lnk.seg = seg2;
			}
			lst = lst.next;
		}
	}

	private void mergeHSegs(ClrSeg seg1, ClrSeg seg2)
	{
		MergeLnkSegs(seg1, seg2, mHlnks);
	}

	private void mergeVSegs(ClrSeg seg1, ClrSeg seg2)
	{
		MergeLnkSegs(seg1, seg2, mVlnks);
	}

	/* Filters out bogus bend segments. */
	private void remExtraBends(int l0, int l1)
	{
		ClrSeg lst0, lst, n, p;
		ClrSeg nxt, prv;
		int/*ACFixed*/ loc0, loc;
		lst0 = mSegLists[l0];
		prv = null;
		while (lst0 != null) {
			nxt = lst0.sNxt;
			loc0 = lst0.sLoc;
			lst = mSegLists[l1];
			p = null;
			while (lst != null) {
				n = lst.sNxt;
				loc = lst.sLoc;
				if (loc > loc0) {
					break;		/* list in increasing order by sLoc */
				}
				if (loc == loc0 && lst.sMin < lst0.sMax && lst.sMax > lst0.sMin) {
					if (lst0.sType == S_BEND && lst.sType != S_BEND &&
					    lst.sType != S_GHOST &&
					    (lst.sMax - lst.sMin) > (lst0.sMax - lst0.sMin)*3) {
						/* delete lst0 */
						if (prv == null) {
							mSegLists[l0] = nxt;
						} else {
							prv.sNxt = nxt;
						}
						lst0 = prv;
						break;
					}
					if (lst.sType == S_BEND && lst0.sType != S_BEND &&
					    lst0.sType != S_GHOST &&
					    (lst0.sMax - lst0.sMin) > (lst.sMax - lst.sMin)*3) {
						/* delete lst */
						if (p == null) {
							mSegLists[l1] = n;
						} else {
							p.sNxt = n;
						}
						lst = p;
					}
				}
				p = lst;
				lst = n;
			}
			prv = lst0;
			lst0 = nxt;
		}
	}

	/* MA -
	 * i - segList index, vert - MergeVSegs
	 * remove segments completely contained in others by calling MergeHSegs or MergeVSegs
	 * a segList is sorted in increasing order by sLoc
	 */
	private void compactList(int i, boolean vert)
	{
		ClrSeg lst, prv, nxtprv, nxt;
		int/*ACFixed*/ lstmin, lstmax, nxtmin, nxtmax;
		boolean flg;
		lst = mSegLists[i];
		prv = null;
		while (lst != null) {
			nxt = lst.sNxt;
			nxtprv = lst;
			while (true) {
				if ((nxt == null) || (nxt.sLoc > lst.sLoc)) {
					flg = true;
					break;
				}
				lstmin = lst.sMin;
				lstmax = lst.sMax;
				nxtmin = nxt.sMin;
				nxtmax = nxt.sMax;
				if (lstmax >= nxtmin && lstmin <= nxtmax) {
					/* do not worry about Y_GOES_UP since "sMax" is really max in
					   device space, not in character space */
					if (Math.abs(lstmax - lstmin) > Math.abs(nxtmax - nxtmin)) {
						/* merge into lst and remove nxt */
						if (vert) {
							mergeVSegs(nxt, lst);
						} else {
							mergeHSegs(nxt, lst);
						}
						lst.sMin = Math.min(lstmin, nxtmin);
						lst.sMax = Math.max(lstmax, nxtmax);
						lst.sBonus = Math.max(lst.sBonus, nxt.sBonus);
						nxtprv.sNxt = nxt.sNxt;
					} else {
						/* merge into nxt and remove lst */
						if (vert) {
							mergeVSegs(lst, nxt);
						} else {
							mergeHSegs(lst, nxt);
						}
						nxt.sMin = Math.min(lstmin, nxtmin);
						nxt.sMax = Math.max(lstmax, nxtmax);
						nxt.sBonus = Math.max(lst.sBonus, nxt.sBonus);
						lst = lst.sNxt;
						if (prv == null) {
							mSegLists[i] = lst;
						} else {
							prv.sNxt = lst;
						}
					}
					flg = false;
					break;
				}
				nxtprv = nxt;
				nxt = nxt.sNxt;
			}
			if (flg) {
				prv = lst;
				lst = lst.sNxt;
			}
		}
	}

	/* MA - pick x0 or x1; similar to PickHSpot sans checking against zones */
	private int/*ACFixed*/ pickVSpot(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1,
					 int/*ACFixed*/ px1, int/*ACFixed*/ py1, int/*ACFixed*/ px2, int/*ACFixed*/ py2,
					 int/*ACFixed*/ prvx, int/*ACFixed*/ prvy, int/*ACFixed*/ nxtx, int/*ACFixed*/ nxty)
	{
		int/*ACFixed*/ a1, a2;
		if (x0 == px1 && x1 != px2) {
			return x0;
		}
		if (x0 != px1 && x1 == px2) {
			return x1;
		}
		if (x0 == prvx && x1 != nxtx) {
			return x0;
		}
		if (x0 != prvx && x1 == nxtx) {
			return x1;
		}
		a1 = Math.abs(py1 - y0);
		a2 = Math.abs(py2 - y1);
		if (a1 > a2) {
			return x0;
		}
		a1 = Math.abs(py2 - y1);
		a2 = Math.abs(py1 - y0);
		if (a1 > a2) {
			return x1;
		}
		if (x0 == prvx && x1 == nxtx) {
			a1 = Math.abs(y0 - prvy);
			a2 = Math.abs(y1 - nxty);
			if (a1 > a2) {
				return x0;
			}
			return x1;
		}
		return fixHalfMul(x0 + x1);
	}

	/* MA - returns d*q where 0<= q<= 1.0 */
	private int/*ACFixed*/ adjDist(int/*ACFixed*/ d, int/*ACFixed*/ q)
	{
		if (q == FIX_ONE) {
			return d;
		}
		return fTrunc(d * q);
	}

	/* serifs of ITCGaramond Ultra have points that are not quite horizontal
	   e.g., in H: (53, 51)(74, 52)(116, 54)
	   the following was added to let these through */
	private boolean tstFlat(int/*ACFixed*/ dmn, int/*ACFixed*/ dmx)
	{
		if (dmn < 0) {
			dmn = -dmn;
		}
		if (dmx < 0) {
			dmx = -dmx;
		}
		return (dmx >= psDist(50) && dmn <= psDist(4));
	}

	private boolean nxtHorz(int/*ACFixed*/ x, int/*ACFixed*/ y, PathElt p)
	{
		Cd cd2 = new Cd();
		Cd cd3 = new Cd();
		nxtForBend(p, cd2, cd3);
		return tstFlat(cd2.y - y, cd2.x - x);
	}

	private boolean prvHorz(int/*ACFixed*/ x, int/*ACFixed*/ y, PathElt p)
	{
		Cd cd2 = new Cd();
		prvForBend(p, cd2);
		return tstFlat(cd2.y - y, cd2.x - x);
	}

	private boolean nxtVert(int/*ACFixed*/ x, int/*ACFixed*/ y, PathElt p)
	{
		Cd cd2 = new Cd();
		Cd cd3 = new Cd();
		nxtForBend(p, cd2, cd3);
		return tstFlat(cd2.x - x, cd2.y - y);
	}

	private boolean prvVert(int/*ACFixed*/ x, int/*ACFixed*/ y, PathElt p)
	{
		Cd cd2 = new Cd();
		prvForBend(p, cd2);
		return tstFlat(cd2.x - x, cd2.y - y);
	}

	/* PrvSameDir and NxtSameDir were added to check the direction of a
	   path and not add a band if the point is not at an extreme and is
	   going in the same direction as the previous path. */
	private boolean tstSameDir(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, int/*ACFixed*/ x2, int/*ACFixed*/ y2)
	{
		if (prodLt0(y0 - y1, y1 - y2) || prodLt0(x0 - x1, x1 - x2)) {
			return false;
		}
		return !testBend(x0, y0, x1, y1, x2, y2);
	}

	private boolean prvSameDir(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, PathElt p)
	{
		Cd cd2 = new Cd();
		p = prvForBend(p, cd2);
		if (p != null && p.type == CURVETO && p.prev != null) {
			getEndPoint(p.prev, cd2);
		}
		return tstSameDir(x0, y0, x1, y1, cd2.x, cd2.y);
	}

	private boolean nxtSameDir(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, PathElt p)
	{
		Cd cd2 = new Cd();
		Cd cd3 = new Cd();
		p = nxtForBend(p, cd2, cd3);
		if (p != null && p.type == CURVETO) {
			cd2.x = p.x3;
			cd2.y = p.y3;
		}
		return tstSameDir(x0, y0, x1, y1, cd2.x, cd2.y);
	}

	private void genVPts(int specialCharType)
	{
		/* specialCharType 1 = upper; -1 = lower; 0 = neither */
		PathElt p, fl;
		boolean isVert, flex1, flex2;
		int/*ACFixed*/ flx0, fly0, yavg, yend, ydist, q, q2;
		int/*ACFixed*/ yd2;
		Cd prv = new Cd();
		Cd nxt = new Cd();
		Cd tmp = new Cd();
		Cd ll = new Cd();
		Cd ur = new Cd();
		p = mPathStart;
		flex1 = flex2 = false;
		mCpTo = mCPpercent;
		mCpFrom = 100 - mCpTo;
		flx0 = fly0 = 0;
		fl = null;
		while (p != null) {
			Cd cd0 = new Cd();
			Cd cd1 = new Cd();
			getEndPoints(p, cd0, cd1);
			if (p.type == CURVETO) {
				int/*ACFixed*/ px1, py1, px2, py2;
				isVert = false;
				if (p.isFlex) {
					if (flex1) {
						if (isVertical(flx0, fly0, cd1.x, cd1.y)) {
							addVSegment(fly0, cd1.y, cd1.x, fl.prev, p, S_LINE, 4);
						}
						flex1 = false;
						flex2 = true;
					} else {
						flex1 = true;
						flex2 = false;
						flx0 = cd0.x;
						fly0 = cd0.y;
						fl = p;
					}
				} else {
					flex1 = flex2 = false;
				}
				px1 = p.x1;
				py1 = p.y1;
				px2 = p.x2;
				py2 = p.y2;
				if (!flex2) {
					if ((q = vertQuo(px1, py1, cd0.x, cd0.y)) == 0) {
						/* first two not vertical */
						doVBendsPrv(cd0.x, cd0.y, px1, py1, p);
					} else {
						isVert = true;
						if (px1 == cd0.x ||
						    (px2 != cd1.x &&
						     (prvVert(px1, py1, p) || !prvSameDir(cd1.x, cd1.y, cd0.x, cd0.y, p)))) {
							if ((q2 = vertQuo(px2, py2, cd0.x, cd0.y)) > 0 &&
							    prodGe0(py1 - cd0.y, py2 - cd0.y) &&
							    Math.abs(py2 - cd0.y) > Math.abs(py1 - cd0.y)) {
								ydist = adjDist(cpTo(py1, py2) - cd0.y, q2);
								yend = adjDist(cpTo(cd0.y, py1) - cd0.y, q);
								if (Math.abs(yend) > Math.abs(ydist)) {
									ydist = yend;
								}
								addVSegment(cd0.y, cd0.y + ydist, cd0.x, p.prev, p, S_CURVE, 5);
							} else {
								ydist = adjDist(cpTo(cd0.y, py1) - cd0.y, q);
								addVSegment(cd0.y, cpTo(cd0.y, py1), cd0.x, p.prev, p, S_CURVE, 6);
							}
						}
					}
				}
				if (!flex1) {
					if ((q = vertQuo(px2, py2, cd1.x, cd1.y)) == 0) {
						/* last 2 not vertical */
						doVBendsNxt(px2, py2, cd1.x, cd1.y, p);
					} else if (px2 == cd1.x ||
						   (px1 != cd0.x &&
						    (nxtVert(px2, py2, p) || !nxtSameDir(cd0.x, cd0.y, cd1.x, cd1.y, p)))) {
						ydist = adjDist(cd1.y - cpFrom(py2, cd1.y), q);
						isVert = true;
						q2 = vertQuo(cd0.x, cd0.y, cd1.x, cd1.y);
						yd2 = (q2 > 0) ? adjDist(cd1.y - cd0.y, q2) : 0;
						if (isVert && q2 > 0 && Math.abs(yd2) > Math.abs(ydist)) {
							ydist = fixHalfMul(yd2);
							yavg = fixHalfMul(cd0.y + cd1.y);
							prvForBend(p, prv);
							nxtForBend(p, nxt, tmp);
							addVSegment(yavg - ydist, yavg + ydist,
								    pickVSpot(cd0.x, cd0.y, cd1.x, cd1.y, px1, py1, px2, py2, prv.x, prv.y, nxt.x, nxt.y),
								    p, null, S_CURVE, 7);
						} else {
							q2 = vertQuo(px1, py1, cd1.x, cd1.y);
							if (q2 > 0 && prodGe0(py1 - cd1.y, py2 - cd1.y) &&
							    Math.abs(py2 - cd1.y) < Math.abs(py1 - cd1.y)) {
								yend = adjDist(cd1.y - cpFrom(py1, py2), q2);
								if (Math.abs(yend) > Math.abs(ydist)) {
									ydist = yend;
								}
								addVSegment(cd1.y - ydist, cd1.y, cd1.x, p, null, S_CURVE, 8);
							} else {
								addVSegment(cd1.y - ydist, cd1.y, cd1.x, p, null, S_CURVE, 9);
							}
						}
					}
				}
				if (!flex1 && !flex2) {
					int/*ACFixed*/ minx, maxx;
					maxx = Math.max(cd0.x, cd1.x);
					minx = Math.min(cd0.x, cd1.x);
					if (px1 - maxx >= FIX_ONE || px2 - maxx >= FIX_ONE ||
					    px1 - minx <= FIX_ONE || px2 - minx <= FIX_ONE) {
						findCurveBBox(cd0.x, cd0.y, px1, py1, px2, py2, cd1.x, cd1.y, ll, ur);
						if (ur.x - maxx > FIX_ONE || minx - ll.x > FIX_ONE) {
							int/*ACFixed*/ loc;
							int[] frst = new int[1];
							int[] lst = new int[1];
							loc = (minx - ll.x > ur.x - maxx) ? ll.x : ur.x;
							checkBBoxEdge(p, true, loc, frst, lst);
							yavg = fixHalfMul(frst[0] + lst[0]);
							ydist = (frst[0] == lst[0]) ? (cd1.y - cd0.y)/10 : fixHalfMul(lst[0] - frst[0]);
							if (Math.abs(ydist) < mBendLength) {
								ydist = (ydist > 0) ?
									fixHalfMul(mBendLength) : fixHalfMul(-mBendLength);
							}
							addVSegment(yavg - ydist, yavg + ydist, loc, p, null, S_CURVE, 10);
						}
					}
				}
			} else {
				if (p.type == MOVETO) {
					mBonus = 0;
					if (specialCharType == -1) {
						if (isLower(p)) {
							mBonus = fixInt(200);
						}
					} else {
						if (specialCharType == 1) {
							if (isUpper(p)) {
								mBonus = fixInt(200);
							}
						}
					}
				} else {
					if (!isTiny(p)) {
						if ((q = vertQuo(cd0.x, cd0.y, cd1.x, cd1.y)) > 0) {
							if (cd0.x == cd1.x) {
								addVSegment(cd0.y, cd1.y, cd0.x, p.prev, p, S_LINE, 11);
							} else {
								if (q < FIX_QUARTER) {
									q = FIX_QUARTER;
								}
								ydist = fixHalfMul(adjDist(cd1.y - cd0.y, q));
								yavg = fixHalfMul(cd0.y + cd1.y);
								prvForBend(p, prv);
								nxtForBend(p, nxt, tmp);
								addVSegment(yavg - ydist, yavg + ydist,
									    pickVSpot(cd0.x, cd0.y, cd1.x, cd1.y, cd0.x, cd0.y, cd1.x, cd1.y, prv.x, prv.y, nxt.x, nxt.y),
									    p, null, S_LINE, 12);
							}
						} else {
							doVBendsNxt(cd0.x, cd0.y, cd1.x, cd1.y, p);
							doVBendsPrv(cd0.x, cd0.y, cd1.x, cd1.y, p);
						}
					}
				}
			}
			p = p.next;
		}
		compactList(0, true);
		compactList(1, true);
		remExtraBends(0, 1);
	}

	private boolean inBlueBand(int/*ACFixed*/ loc, int n, int/*ACFixed*/[] p)
	{
		int i;
		int/*ACFixed*/ y;
		if (n <= 0) {
			return false;
		}
		y = itfmy(loc);
		/* Augment the blue band by bluefuzz in each direction. This will
		   result in "near misses" being colored and so adjusted by the
		   PS interpreter. */
		for (i = 0; i < n; i += 2) {
			if ((p[i] - mBluefuzz) <= y && (p[i + 1] + mBluefuzz) >= y) {
				return true;
			}
		}
		return false;
	}

	/* MA - pick y0 or y1 or average of the two consulting blue zones */
	private int/*ACFixed*/ pickHSpot(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, int/*ACFixed*/ xdist,
					 int/*ACFixed*/ px1, int/*ACFixed*/ py1, int/*ACFixed*/ px2, int/*ACFixed*/ py2,
					 int/*ACFixed*/ prvx, int/*ACFixed*/ prvy, int/*ACFixed*/ nxtx, int/*ACFixed*/ nxty)
	{
		boolean topSeg = (xdist < 0) ? true : false;
		int/*ACFixed*/ upper, lower;
		boolean inBlue0, inBlue1;
		if (topSeg) {
			inBlue0 = inBlueBand(y0, mLenTopBands, mTopBands);
			inBlue1 = inBlueBand(y1, mLenTopBands, mTopBands);
		} else {
			inBlue0 = inBlueBand(y0, mLenBotBands, mBotBands);
			inBlue1 = inBlueBand(y1, mLenBotBands, mBotBands);
		}
		if (inBlue0 && !inBlue1) {
			return y0;
		}
		if (inBlue1 && !inBlue0) {
			return y1;
		}
		if (y0 == py1 && y1 != py2) {
			return y0;
		}
		if (y0 != py1 && y1 == py2) {
			return y1;
		}
		if (y0 == prvy && y1 != nxty) {
			return y0;
		}
		if (y0 != prvy && y1 == nxty) {
			return y1;
		}
		if (inBlue0 && inBlue1) {
			if (y0 > y1) {
				upper = y0;
				lower = y1;
			} else {
				upper = y1;
				lower = y0;
			}
			if (!Y_GOES_UP) {
				int/*ACFixed*/ tmp = lower;
				lower = upper;
				upper = tmp;
			}
			return topSeg ? upper : lower;
		}
		if (Math.abs(px1 - x0) > Math.abs(px2 - x1)) {
			return y0;
		}
		if (Math.abs(px2 - x1) > Math.abs(px1 - x0)) {
			return y1;
		}
		if (y0 == prvy && y1 == nxty) {
			if (Math.abs(x0 - prvx) > Math.abs(x1 - nxtx)) {
				return y0;
			}
			return y1;
		}
		return fixHalfMul(y0 + y1);
	}

	/* MA - walk all path elements generate H segments
	 */
	private void genHPts()
	{
		PathElt p, fl;
		boolean isHoriz, flex1, flex2;
		int/*ACFixed*/ flx0, fly0, xavg, xend, xdist, q, q2;
		int/*ACFixed*/ xd2;
		Cd prv = new Cd();
		Cd nxt = new Cd();
		Cd tmp = new Cd();
		Cd ll = new Cd();
		Cd ur = new Cd();
		p = mPathStart;
		mBonus = 0;
		flx0 = fly0 = 0;
		fl = null;
		flex1 = flex2 = false;
		mCpTo = mCPpercent;
		mCpFrom = 100 - mCpTo;
		while (p != null) {
			Cd cd0 = new Cd();
			Cd cd1 = new Cd();
			getEndPoints(p, cd0, cd1);
			if (p.type == CURVETO) {
				int/*ACFixed*/ px1, py1, px2, py2;
				isHoriz = false;
				if (p.isFlex) {
					if (flex1) {
						flex1 = false;
						flex2 = true;
						if (isHorizontal(flx0, fly0, cd1.x, cd1.y)) {
							addHSegment(flx0, cd1.x, cd1.y, fl.prev, p, S_LINE, 4);
						}
					} else {
						flex1 = true;
						flex2 = false;
						flx0 = cd0.x;
						fly0 = cd0.y;
						fl = p;
					}
				} else {
					flex1 = flex2 = false;
				}
				px1 = p.x1;
				py1 = p.y1;
				px2 = p.x2;
				py2 = p.y2;
				if (!flex2) {
					if ((q = horzQuo(px1, py1, cd0.x, cd0.y)) == 0) {
						doHBendsPrv(cd0.x, cd0.y, px1, py1, p);
					} else {
						isHoriz = true;
						if (py1 == cd0.y ||
						    (py2 != cd1.y &&
						     (prvHorz(px1, py1, p) || !prvSameDir(cd1.x, cd1.y, cd0.x, cd0.y, p)))) {
							if ((q2 = horzQuo(px2, py2, cd0.x, cd0.y)) > 0 &&
							    prodGe0(px1 - cd0.x, px2 - cd0.x) &&
							    Math.abs(px2 - cd0.x) > Math.abs(px1 - cd0.x)) {
								xdist = adjDist(cpTo(px1, px2) - cd0.x, q2);
								xend = adjDist(cpTo(cd0.x, px1) - cd0.x, q);
								if (Math.abs(xend) > Math.abs(xdist)) {
									xdist = xend;
								}
								addHSegment(cd0.x, cd0.x + xdist, cd0.y, p.prev, p, S_CURVE, 5);
							} else {
								xdist = adjDist(cpTo(cd0.x, px1) - cd0.x, q);
								addHSegment(cd0.x, cd0.x + xdist, cd0.y, p.prev, p, S_CURVE, 6);
							}
						}
					}
				}
				if (!flex1) {
					if ((q = horzQuo(px2, py2, cd1.x, cd1.y)) == 0) {
						doHBendsNxt(px2, py2, cd1.x, cd1.y, p);
					} else if (py2 == cd1.y ||
						   (py1 != cd0.y &&
						    (nxtHorz(px2, py2, p) || !nxtSameDir(cd0.x, cd0.y, cd1.x, cd1.y, p)))) {
						xdist = adjDist(cd1.x - cpFrom(px2, cd1.x), q);
						q2 = horzQuo(cd0.x, cd0.y, cd1.x, cd1.y);
						isHoriz = true;
						xd2 = (q2 > 0) ? adjDist(cd1.x - cd0.x, q2) : 0;
						if (isHoriz && q2 > 0 && Math.abs(xd2) > Math.abs(xdist)) {
							int/*ACFixed*/ hspot;
							prvForBend(p, prv);
							nxtForBend(p, nxt, tmp);
							xdist = fixHalfMul(xd2);
							xavg = fixHalfMul(cd0.x + cd1.x);
							hspot = pickHSpot(cd0.x, cd0.y, cd1.x, cd1.y, xdist, px1, py1, px2, py2, prv.x, prv.y, nxt.x, nxt.y);
							addHSegment(xavg - xdist, xavg + xdist, hspot, p, null, S_CURVE, 7);
						} else {
							q2 = horzQuo(px1, py1, cd1.x, cd1.y);
							if (q2 > 0 && prodGe0(px1 - cd1.x, px2 - cd1.x) &&
							    Math.abs(px2 - cd1.x) < Math.abs(px1 - cd1.x)) {
								xend = adjDist(cd1.x - cpFrom(px1, px2), q2);
								if (Math.abs(xend) > Math.abs(xdist)) {
									xdist = xend;
								}
								addHSegment(cd1.x - xdist, cd1.x, cd1.y, p, null, S_CURVE, 8);
							} else {
								addHSegment(cd1.x - xdist, cd1.x, cd1.y, p, null, S_CURVE, 9);
							}
						}
					}
				}
				if (!flex1 && !flex2) {
					int/*ACFixed*/ miny, maxy;
					maxy = Math.max(cd0.y, cd1.y);
					miny = Math.min(cd0.y, cd1.y);
					if (py1 - maxy >= FIX_ONE || py2 - maxy >= FIX_ONE ||
					    py1 - miny <= FIX_ONE || py2 - miny <= FIX_ONE) {
						findCurveBBox(cd0.x, cd0.y, px1, py1, px2, py2, cd1.x, cd1.y, ll, ur);
						if (ur.y - maxy > FIX_ONE || miny - ll.y > FIX_ONE) {
							int/*ACFixed*/ loc;
							int[] frst = new int[1];
							int[] lst = new int[1];
							loc = (miny - ll.y > ur.y - maxy) ? ll.y : ur.y;
							checkBBoxEdge(p, false, loc, frst, lst);
							xavg = fixHalfMul(frst[0] + lst[0]);
							xdist = (frst[0] == lst[0]) ? (cd1.x - cd0.x)/10 : fixHalfMul(lst[0] - frst[0]);
							if (Math.abs(xdist) < mBendLength) {
								xdist = (xdist > 0.0) ?
									fixHalfMul(mBendLength) : fixHalfMul(-mBendLength);
							}
							addHSegment(xavg - xdist, xavg + xdist, loc, p, null, S_CURVE, 10);
						}
					}
				}
			} else {
				if (p.type != MOVETO && !isTiny(p)) {
					if ((q = horzQuo(cd0.x, cd0.y, cd1.x, cd1.y)) > 0) {
						if (cd0.y == cd1.y) {
							addHSegment(cd0.x, cd1.x, cd0.y, p.prev, p, S_LINE, 11);
						} else {
							if (q < FIX_QUARTER) {
								q = FIX_QUARTER;
							}
							xdist = fixHalfMul(adjDist(cd1.x - cd0.x, q));
							xavg = fixHalfMul(cd0.x + cd1.x);
							prvForBend(p, prv);
							nxtForBend(p, nxt, tmp);
							tmp.y = pickHSpot(cd0.x, cd0.y, cd1.x, cd1.y, xdist, cd0.x, cd0.y, cd1.x, cd1.y, prv.x, prv.y, nxt.x, nxt.y);
							addHSegment(xavg - xdist, xavg + xdist, tmp.y, p.prev, p, S_LINE, 12);
						}
					} else {
						doHBendsNxt(cd0.x, cd0.y, cd1.x, cd1.y, p);
						doHBendsPrv(cd0.x, cd0.y, cd1.x, cd1.y, p);
					}
				}
			}
			p = p.next;
		}
		compactList(2, false);
		compactList(3, false);
		remExtraBends(2, 3);
	}

	private void preGenPts()
	{
		mHlnks = null;
		mSegLists[2] = null;
		mSegLists[3] = null;
		mVlnks = null;
		mSegLists[0] = null;
		mSegLists[1] = null;
	}

	/* Module ACcontrol.c */

	private void initAll()
	{
		initData();		/* must be first */
		initAuto();
		initFix();
		initGen();
		initPick();
	}

	private int ptLstLen(ClrPoint lst)
	{
		int cnt = 0;
		while (lst != null) {
			cnt++;
			lst = lst.next;
		}
		return cnt;
	}

	private int pointListCheck(ClrPoint newCP, ClrPoint lst)
	{
		/* -1 means not a member, 1 means already a member, 0 means conflicts */
		int/*ACFixed*/ l1 = 0, l2 = 0, n1 = 0, n2 = 0, tmp, halfMargin;
		char ch = newCP.c;
		halfMargin = fixHalfMul(mBandMargin);
		switch (ch) {
		case 'y': case 'm': {
			n1 = newCP.x0;
			n2 = newCP.x1;
			break;
		}
		case 'b': case 'v': {
			n1 = newCP.y0;
			n2 = newCP.y1;
			break;
		}
		}
		if (n1 > n2) {
			tmp = n1;
			n1 = n2;
			n2 = tmp;
		}
		while (true) {
			if (lst == null) {
				return -1;
			}
			if (lst.c == ch) {
				/* same kind of color */
				switch (ch) {
				case 'y': case 'm': {
					l1 = lst.x0;
					l2 = lst.x1;
					break;
				}
				case 'b': case 'v': {
					l1 = lst.y0;
					l2 = lst.y1;
					break;
				}
				}
				if (l1 > l2) {
					tmp = l1;
					l1 = l2;
					l2 = tmp;
				}
				if (l1 == n1 && l2 == n2) {
					return 1;
				}
				/* Add this extra margin to the band to fix a problem in
				   TimesEuropa/Italic/v, w, y where a main hstem hint was
				   being merged with newcolors. This main hstem caused
				   problems in rasterization so it shouldn't be included. */
				l1 -= halfMargin;
				l2 += halfMargin;
				if (l1 <= n2 && n1 <= l2) {
					return 0;
				}
			}
			lst = lst.next;
		}
	}

	private boolean sameColorLists(ClrPoint lst1, ClrPoint lst2)
	{
		if (ptLstLen(lst1) != ptLstLen(lst2)) {
			return false;
		}
		while (lst1 != null) {
			/* go through lst1 */
			if (pointListCheck(lst1, lst2) != 1) {
				return false;
			}
			lst1 = lst1.next;
		}
		return true;
	}

	private boolean sameColors(int cn1, int cn2)
	{
		if (cn1 == cn2) {
			return true;
		}
		return sameColorLists(mPtLstArray[cn1], mPtLstArray[cn2]);
	}

	private void mergeFromMainColors(char ch)
	{
		ClrPoint lst;
		for (lst = mPtLstArray[0]; lst != null; lst = lst.next) {
			if (lst.c != ch) {
				continue;
			}
			if (pointListCheck(lst, mPointList) == -1) {
				if (ch == 'b') {
					addColorPoint(0, lst.y0, 0, lst.y1, ch, lst.p0, lst.p1);
				} else {
					addColorPoint(lst.x0, 0, lst.x1, 0, ch, lst.p0, lst.p1);
				}
			}
		}
	}

	private void addColorPoint(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1, char ch, PathElt p0, PathElt p1)
	{
		ClrPoint pt;
		int chk;
		pt = new ClrPoint();
		pt.cptSN = ++mCptSN;
		if (pt.cptSN == 0) pt.cptSN++;
		pt.x0 = x0;
		pt.y0 = y0;
		pt.x1 = x1;
		pt.y1 = y1;
		pt.c = ch;
		pt.done = false;
		pt.next = null;
		pt.p0 = p0;
		pt.p1 = p1;
		chk = pointListCheck(pt, mPointList);
		if (chk == -1) {
			pt.next = mPointList;
			mPointList = pt;
		}
	}

	private void copyClrFromLst(char clr, ClrPoint lst)
	{
		boolean bvflg = (clr == 'b' || clr == 'v');
		while (lst != null) {
			if (lst.c == clr) {
				if (bvflg) {
					addColorPoint(0, lst.y0, 0, lst.y1, clr, lst.p0, lst.p1);
				} else {
					addColorPoint(lst.x0, 0, lst.x1, 0, clr, lst.p0, lst.p1);
				}
			}
			lst = lst.next;
		}
	}

	private void copyMainV()
	{
		copyClrFromLst('m', mPtLstArray[0]);
	}

	private void copyMainH()
	{
		copyClrFromLst('v', mPtLstArray[0]);
	}

	private void addHPair(ClrVal v, char ch)
	{
		int/*ACFixed*/ bot, top, tmp;
		PathElt p0, p1, p;
		bot = itfmy(v.vLoc1);
		top = itfmy(v.vLoc2);
		p0 = v.vBst.vSeg1.sElt;
		p1 = v.vBst.vSeg2.sElt;
		if (top < bot) {
			tmp = top;
			top = bot;
			bot = tmp;
			p = p0;
			p0 = p1;
			p1 = p;
		}
		if (v.vGhst) {
			if (v.vSeg1.sType == S_GHOST) {
				bot = top;
				p0 = p1;
				p1 = null;
				top = bot + fixInt(TOPGHST);		/* width == -20 iff bottom seg is ghost */
			} else {
				top = bot;
				p1 = p0;
				p0 = null;
				bot = top - fixInt(BOTGHST);		/* width == -21 iff top seg is ghost */
			}
		}
		addColorPoint(0, bot, 0, top, ch, p0, p1);
	}

	private void addVPair(ClrVal v, char ch)
	{
		int/*ACFixed*/ lft, rght, tmp;
		PathElt p0, p1, p;
		lft = itfmx(v.vLoc1);
		rght = itfmx(v.vLoc2);
		p0 = v.vBst.vSeg1.sElt;
		p1 = v.vBst.vSeg2.sElt;
		if (lft > rght) {
			tmp = lft;
			lft = rght;
			rght = tmp;
			p = p0;
			p0 = p1;
			p1 = p;
		}
		addColorPoint(lft, 0, rght, 0, ch, p0, p1);
	}

	/* MA - pasted from AC.doc
	   Check to see if H or V counter hints (hstem3, vstem3) should be used
	   instead of hstem or vstem.
	*/
	private boolean useCounter(ClrVal sLst, boolean mclr)
	{
		int cnt = 0;
		int/*ACFixed*/ minLoc, midLoc, maxLoc, prevBstVal, bestVal;
		int/*ACFixed*/ minDelta, midDelta, maxDelta, loc, delta, th;
		ClrVal lst, newLst;
		minLoc = midLoc = maxLoc = fixInt(20000);
		minDelta = midDelta = maxDelta = 0;
		lst = sLst;
		while (lst != null) {
			cnt++;
			lst = lst.vNxt;
		}
		if (cnt < 3) {
			return false;
		}
		cnt -= 3;
		prevBstVal = 0;
		while (cnt > 0) {
			cnt--;
			if (cnt == 0) {
				prevBstVal = sLst.vVal;
			}
			sLst = sLst.vNxt;
		}
		bestVal = sLst.vVal;
		if (prevBstVal > fixInt(1000) || bestVal < prevBstVal * 10) {
			return false;
		}
		newLst = sLst;
		while (sLst != null) {
			loc = sLst.vLoc1;
			delta = sLst.vLoc2 - loc;
			loc += fixHalfMul(delta);
			if (loc < minLoc) {
				maxLoc = midLoc;
				maxDelta = midDelta;
				midLoc = minLoc;
				midDelta = minDelta;
				minLoc = loc;
				minDelta = delta;
			} else {
				if (loc < midLoc) {
					maxLoc = midLoc;
					maxDelta = midDelta;
					midLoc = loc;
					midDelta = delta;
				} else {
					maxLoc = loc;
					maxDelta = delta;
				}
			}
			sLst = sLst.vNxt;
		}
		th = fixInt(5) / 100;
		if (Math.abs(minDelta - maxDelta) < th && Math.abs((maxLoc - midLoc) - (midLoc - minLoc)) < th) {
			if (mclr) {
				mVcoloring = newLst;
			} else {
				mHcoloring = newLst;
			}
			return true;
		}
		return false;
	}

	private void getNewPtLst()
	{
		if (mNumPtLsts >= mMaxPtLsts) {
			/* increase size */
			int i;
			mMaxPtLsts += 5;
			ClrPoint[] newArray = new ClrPoint[mMaxPtLsts];
			for (i = 0; i < mMaxPtLsts - 5; i++) {
				newArray[i] = mPtLstArray[i];
			}
			mPtLstArray = newArray;
		}
		mPtLstIndex = mNumPtLsts;
		mNumPtLsts++;
		mPointList = null;
		mPtLstArray[mPtLstIndex] = null;
	}

	private void xtraClrs(PathElt e)
	{
		/* this can be simplified for standalone coloring */
		mPtLstArray[mPtLstIndex] = mPointList;
		if (e.newcolors == 0) {
			getNewPtLst();
			e.newcolors = (short)mPtLstIndex;
		}
		mPtLstIndex = e.newcolors;
		mPointList = mPtLstArray[mPtLstIndex];
	}

	private void blues()
	{
		int/*ACFixed*/ pv = 0, pd = 0, pc = 0, pb = 0, pa = 0;
		ClrVal sLst;
		if (noBlueChar(mUnicode)) {
			mLenTopBands = mLenBotBands = 0;
		}
		genHPts();		/* MA - generate H segments */
		if (mDoCounters && !mCounterFailed && hColorChar(mUnicode)) {
			pv = mPruneValue;
			mPruneValue = fltToFix(mMinVal);
			pa = mPruneA;
			mPruneA = fltToFix(mMinVal);
			pd = mPruneD;
			mPruneD = fltToFix(mMinVal);
			pc = mPruneC;
			mPruneC = fltToFix(mMaxVal);
			pb = mPruneB;
			mPruneB = fltToFix(mMinVal);
		}
		evalH();
		pruneHVals();
		findBestHVals();
		mergeVals(false);
		markLinks(mValList, true);
		checkVals(mValList, false);
		doHStems (mValList);
		pickHVals(mValList);
		if (mDoCounters && !mCounterFailed && hColorChar(mUnicode)) {
			mPruneValue = pv;
			mPruneD = pd;
			mPruneC = pc;
			mPruneB = pb;
			mPruneA = pa;
			mUseH = useCounter(mHcoloring, false);
			if (!mUseH) {
				/* try to fix */
				addBBoxHV(true, true);
				mUseH = useCounter(mHcoloring, false);
				if (!mUseH) {
					/* still bad news */
					mCounterFailed = true;
				}
			}
		} else {
			mUseH = false;
		}
		if (mHcoloring == null) {
			addBBoxHV(true, false);
		}
		sLst = mHcoloring;
		while (sLst != null) {
			addHPair(sLst, mUseH ? 'v' : 'b');
			sLst = sLst.vNxt;
		}
	}

	private void doHStems (ClrVal sLst1)
	{
		int/*ACFixed*/ bot, top;
		int/*ACFixed*/ charTop = Integer.MIN_VALUE, charBot = Integer.MAX_VALUE;
		boolean curved;
		ACBBox bbox = new ACBBox();
		if (!mDoAligns) {
			return;
		}
		findPathBBox(bbox);
		if (Y_GOES_UP) {
			addCharExtreme(unScaleAbs(itfmy(bbox.ymin)), unScaleAbs(itfmy(bbox.ymax)));
		} else {
			addCharExtreme(unScaleAbs(itfmy(bbox.ymax)), unScaleAbs(itfmy(bbox.ymin)));
		}
		while (sLst1 != null) {
			bot = itfmy(sLst1.vLoc1);
			top = itfmy(sLst1.vLoc2);
			if (top < bot) {
				int/*ACFixed*/ tmp = top;
				top = bot;
				bot = tmp;
			}
			if (top > charTop) {
				charTop = top;
			}
			if (bot < charBot) {
				charBot = bot;
			}
			/* skip if ghost or not a line on top or bottom */
			if (sLst1.vGhst) {
				sLst1 = sLst1.vNxt;
				continue;
			}
			curved = !findLineSeg(sLst1.vLoc1, botList()) &&
				!findLineSeg(sLst1.vLoc2, topList());
			addHStem(unScaleAbs(top), unScaleAbs(bot), curved);
			sLst1 = sLst1.vNxt;
			if (top != Integer.MIN_VALUE || bot != Integer.MAX_VALUE) {
				addZone(unScaleAbs(bot), unScaleAbs(top));
			}
		}
		if (charTop != Integer.MIN_VALUE || charBot != Integer.MAX_VALUE) {
			addCharZone(unScaleAbs(charBot), unScaleAbs(charTop));
		}
	}

	private void yellows()
	{
		int/*ACFixed*/ pv = 0, pd = 0, pc = 0, pb = 0, pa = 0;
		ClrVal sLst;
		genVPts(specialCharType(mUnicode));
		if (mDoCounters && !mCounterFailed && vColorChar(mUnicode)) {
			pv = mPruneValue;
			mPruneValue = fltToFix(mMinVal);
			pa = mPruneA;
			mPruneA = fltToFix(mMinVal);
			pd = mPruneD;
			mPruneD = fltToFix(mMinVal);
			pc = mPruneC;
			mPruneC = fltToFix(mMaxVal);
			pb = mPruneB;
			mPruneB = fltToFix(mMinVal);
		}
		evalV();
		pruneVVals();
		findBestVVals();
		mergeVals(true);
		markLinks(mValList, false);
		checkVals(mValList, true);
		doVStems(mValList);
		pickVVals(mValList);
		if (mDoCounters && !mCounterFailed && vColorChar(mUnicode)) {
			mPruneValue = pv;
			mPruneD = pd;
			mPruneC = pc;
			mPruneB = pb;
			mPruneA = pa;
			mUseV = useCounter(mVcoloring, true);
			if (!mUseV) {
				/* try to fix */
				addBBoxHV(false, true);
				mUseV = useCounter(mVcoloring, true);
				if (!mUseV) {
					/* still bad news */
					mCounterFailed = true;
				}
			}
		} else {
			mUseV = false;
		}
		if (mVcoloring == null) {
			addBBoxHV(false, false);
		}
		sLst = mVcoloring;
		while (sLst != null) {
			addVPair(sLst, mUseV ? 'm' : 'y');
			sLst = sLst.vNxt;
		}
	}

	private void doVStems(ClrVal sLst)
	{
		int/*ACFixed*/ lft, rght;
		boolean curved;
		if (!mDoAligns) {
			return;
		}
		while (sLst != null) {
			curved = !findLineSeg(sLst.vLoc1, leftList()) &&
				!findLineSeg(sLst.vLoc2, rightList());
			lft = itfmx(sLst.vLoc1);
			rght = itfmx(sLst.vLoc2);
			if (lft > rght) {
				int/*ACFixed*/ tmp = lft;
				lft = rght;
				rght = tmp;
			}
			addVStem(unScaleAbs(rght), unScaleAbs(lft), curved);
			sLst = sLst.vNxt;
		}
	}

	private void removeRedundantFirstColors()
	{
		PathElt e;
		if (mNumPtLsts < 2) {
			return;
		}
		if (!sameColors(0, 1)) {
			return;
		}
		e = mPathStart;
		while (e != null) {
			if (e.newcolors == 1) {
				e.newcolors = 0;
				return;
			}
			e = e.next;
		}
	}

	private void addColorsSetup()
		throws ACException
	{
		int i;
		mVBigDist = 0;
		for (i = 0; i < mNumVStems; i++) {
			if (mVStems[i] > mVBigDist) {
				mVBigDist = mVStems[i];
			}
		}
		mVBigDist = dtfmx(mVBigDist);
		if (mVBigDist < mInitBigDist) {
			mVBigDist = mInitBigDist;
		}
		mVBigDist = (mVBigDist * 23) / 20;
		mVBigDistR = fixToFlt(mVBigDist);
		mHBigDist = 0;
		for (i = 0; i < mNumHStems; i++) {
			if (mHStems[i] > mHBigDist) {
				mHBigDist = mHStems[i];
			}
		}
		mHBigDist = Math.abs(dtfmy(mHBigDist));
		if (mHBigDist < mInitBigDist) {
			mHBigDist = mInitBigDist;
		}
		mHBigDist = (mHBigDist * 23) / 20;
		mHBigDistR = fixToFlt(mHBigDist);
		if (!mScalinghints) {
			roundPathCoords();
		}
		checkForMultiMoveTo();
		collectSubPathInfo();
		if (mFixWinding) {
			fixPathWind(Y_GOES_UP/*forceCCW*/);
		}
	}

	private void addColorsInnerLoop()
		throws ACException
	{
		int retryColoring = 0;
		while (true) {
			preGenPts();
			if (mDoSmoothing) {
				checkSmooth();
			}
			initShuffleSubpaths();
			blues();
			if (mGenerateVStems) {
				yellows();
			}
			if (!mDoWriteGlyph) {
				return;
			}
			if (mEditChar) {
				doShuffleSubpaths();
			}
			mHprimary = copyClrs(mHcoloring);
			if (mGenerateVStems) {
				mVprimary = copyClrs(mVcoloring);
			}
			pruneElementColorSegs();
			if (mExtracolor) {
				autoExtraColors(moveToNewClrs(mUnicode));
			}
			mPtLstArray[mPtLstIndex] = mPointList;
			retryColoring++;
			if (!(mCounterFailed && retryColoring == 1)) {
				if (!doFixes()) {
					break;
				}
				if (retryColoring > 2) {
					break;
				}
			}
			initAll();
			resetGlyph();
			addColorsSetup();
			if (!preCheckForColoring()) {
				break;
			}
			if (mFlexOK) {
				autoAddFlex();
			}
		}
	}

	private void addColorsCleanup()
	{
		removeRedundantFirstColors();
		if (mPathStart != null && mPathStart != mPathEnd) {
			writeGlyph();
		}
		initAll();
	}

	private void addColors()
		throws ACException
	{
		if (mPathStart == null || mPathStart == mPathEnd) {
			writeGlyph();	/* make sure it gets saved with no coloring */
			return;
		}
		mCounterFailed = false;
		addColorsSetup();
		if (!preCheckForColoring()) {
			/* write unhinted glyph */
			writeGlyph();
			return;
		}
		if (mFlexOK) {
			autoAddFlex();
		}
		addColorsInnerLoop();
		addColorsCleanup();
	}

	private boolean doGlyph()
		throws ACException
	{
		resetGlyph();
		addColors();
		return true;
	}

	/* Module ACeval.c */

	private boolean lePruneValue(int/*ACFixed*/ val)
	{
		return (val < FIX_ONE && (val << 10) <= mPruneValue);
	}

	/* MA - calculate the priority of the pair in range [1.0, 8000000.0] */
	private void adjustVal(int/*ACFixed*/[] pv, int/*ACFixed*/ l1, int/*ACFixed*/ l2, int/*ACFixed*/ dist, int/*ACFixed*/ d, boolean hFlg)
	{
		double v, q, r1, r2, rd;
		if (dist < FIX_ONE) {
			dist = FIX_ONE;
		}
		if (l1 < FIX_ONE) {
			l1 = FIX_ONE;
		}
		if (l2 < FIX_ONE) {
			l2 = FIX_ONE;
		}
		if (Math.abs(l1) < MAXF) {
			r1 = l1 * l1;
		} else {
			r1 = l1;
			r1 = r1*r1;
		}
		if (Math.abs(l2) < MAXF) {
			r2 = l2 * l2;
		} else {
			r2 = l2;
			r2 = r2*r2;
		}
		if (Math.abs(dist) < MAXF) {
			q = dist * dist;
		} else {
			q = dist;
			q = q*q;
		}
		v = (1000.0f * r1 * r2) / (q * q);
		while (true) {
			if (d <= (hFlg ? mHBigDist : mVBigDist)) {
				break;
			}
			rd = fixToFlt(d);
			/*	the following code is an optimized version of two lines:
			  q = (hFlg ? mHBigDistR : mVBigDistR) / rd;
			  if (q <= 0.5) {
			  v = 0.0;
			  break;
			  }
			*/
			q = (hFlg ? mHBigDistR : mVBigDistR);
			if (q + q <= rd) {
				v = 0.0;
				break;
			}
			q /= rd;
			/* 0 < q < 1.0 */
			q *= q;
			q *= q;
			q *= q;		/* raise q to 8th power */
			v = v * q;	/* if d is twice bigDist, value goes down by factor of 256 */
			break;
		}
		if (v > mMaxVal) {
			v = mMaxVal;
		} else if (v > 0.0 && v < mMinVal) {
			v = mMinVal;
		}
		pv[0] = fltToFix(v);
	}

	private int/*ACFixed*/ calcOverlapDist(int/*ACFixed*/ d, int/*ACFixed*/ overlaplen, int/*ACFixed*/ minlen)
	{
		double r = d, ro = overlaplen, rm = minlen;
		r = r * (1.0 + 0.4 * (1.0 - ro / rm));
		d = (int/*ACFixed*/)r;
		return d;
	}

	private int/*ACFixed*/ gapDist(int/*ACFixed*/ d)
	{
		if (d < fixInt(127)) {
			return fTrunc((d * d) / 20);
		}
		return fTrunc(d * (d / 20));
	}

	private void evalHPair(ClrSeg botSeg, ClrSeg topSeg, int/*ACFixed*/[] pspc, int/*ACFixed*/[] pv)
	{
		int/*ACFixed*/ brght, blft, bloc, tloc, trght, tlft, ldst, rdst;
		int/*ACFixed*/ mndist, dist, dx, dy, minlen, overlaplen;
		boolean inBotBand, inTopBand;
		int i;
		pspc[0] = 0;
		brght = botSeg.sMax;
		blft = botSeg.sMin;
		trght = topSeg.sMax;
		tlft = topSeg.sMin;
		bloc = botSeg.sLoc;
		tloc = topSeg.sLoc;
		dy = Math.abs(bloc - tloc);
		if (dy < mMinDist) {
			pv[0] = 0;
			return;
		}
		inBotBand = inBlueBand(bloc, mLenBotBands, mBotBands);
		inTopBand = inBlueBand(tloc, mLenTopBands, mTopBands);
		if (inBotBand && inTopBand) {
			/* delete these */
			pv[0] = 0;
			return;
		}
		if (inBotBand || inTopBand) {
			/* up the priority of these */
			pspc[0] = fixInt(2);
		}
		/* left is always < right */
		if ((tlft <= brght) && (trght >= blft)) {
			/* overlap */
			overlaplen = Math.min(trght, brght) - Math.max(tlft, blft);
			minlen = Math.min(trght - tlft, brght - blft);
			if (minlen == overlaplen) {
				dist = dy;
			} else {
				dist = calcOverlapDist(dy, overlaplen, minlen);
			}
		} else {
			/* no overlap; take closer ends */
			ldst = Math.abs(tlft - brght);
			rdst = Math.abs(trght - blft);
			dx = Math.min(ldst, rdst);
			dist = gapDist(dx) + (7*dy)/5;	/* extra penalty for nonoverlap
							 * changed from 7/5 to 12/5 for Perpetua/Regular/
							 * n, r, m and other lowercase serifs;
							 * undid change for Berthold/AkzidenzGrotesk 9/16/91;
							 * this did not make Perpetua any worse. */
			if (dx > dy) {
				dist *= dx / dy;
			}
		}
		mndist = fixTwoMul(mMinDist);
		dist = Math.max(dist, mndist);
		if (mNumHStems > 0) {
			int/*ACFixed*/ w = idtfmy(dy);
			w = Math.abs(w);
			for (i = 0; i < mNumHStems; i++) {
				if (w == mHStems[i]) {
					pspc[0] += FIX_ONE;
					break;
				}
			}
		}
		adjustVal(pv, brght - blft, trght - tlft, dist, dy, true);
	}

	private void evalVPair(ClrSeg leftSeg, ClrSeg rightSeg, int/*ACFixed*/[] pspc, int/*ACFixed*/[] pv)
	{
		int/*ACFixed*/ ltop, lbot, lloc, rloc, rtop, rbot, tdst, bdst;
		int/*ACFixed*/ mndist, dx, dy, dist, overlaplen, minlen;
		int/*ACFixed*/ bonus, lbonus, rbonus;
		int i;
		pspc[0] = 0;
		ltop = leftSeg.sMax;
		lbot = leftSeg.sMin;
		rtop = rightSeg.sMax;
		rbot = rightSeg.sMin;
		lloc = leftSeg.sLoc;
		rloc = rightSeg.sLoc;
		dx = Math.abs(lloc - rloc);
		if (dx < mMinDist) {
			pv[0] = 0;
			return;
		}
		/* top is always > bot, independent of Y_GOES_UP */
		if ((ltop >= rbot) && (lbot <= rtop)) {
			/* overlap */
			overlaplen = Math.min(ltop, rtop) - Math.max(lbot, rbot);
			minlen = Math.min(ltop - lbot, rtop - rbot);
			if (minlen == overlaplen) {
				dist = dx;
			} else {
				dist = calcOverlapDist(dx, overlaplen, minlen);
			}
		} else {
			/* no overlap; take closer ends */
			tdst = Math.abs(ltop - rbot);
			bdst = Math.abs(lbot - rtop);
			dy = Math.min(tdst, bdst);
			dist = (7*dx)/5 + gapDist(dy);		/* extra penalty for nonoverlap */
			if (dy > dx) {
				dist *= dy / dx;
			}
		}
		mndist = fixTwoMul(mMinDist);
		dist = Math.max(dist, mndist);
		lbonus = leftSeg.sBonus;
		rbonus = rightSeg.sBonus;
		bonus = Math.min(lbonus, rbonus);
		pspc[0] = (bonus > 0) ? fixInt(2): 0;		/* this is for sol-eol characters */
		if (mNumVStems > 0) {
			int/*ACFixed*/ w = idtfmx(dx);
			w = Math.abs(w);
			for (i = 0; i < mNumVStems; i++) {
				if (w == mVStems[i]) {
					pspc[0] += FIX_ONE;
					break;
				}
			}
		}
		adjustVal(pv, ltop - lbot, rtop - rbot, dist, dx, false);
	}

	private void insertVValue(int/*ACFixed*/ lft, int/*ACFixed*/ rght, int/*ACFixed*/ val, int/*ACFixed*/ spc, ClrSeg lSeg, ClrSeg rSeg)
	{
		ClrVal item, vlist, vprev;
		item = new ClrVal();
		item.cvlSN = ++mCvlSN;
		if (item.cvlSN == 0) item.cvlSN++;
		item.vVal = val;
		item.initVal = val;
		item.vLoc1 = lft;
		item.vLoc2 = rght;
		item.vSpc = spc;
		item.vSeg1 = lSeg;
		item.vSeg2 = rSeg;
		item.vGhst = false;
		vlist = mValList;
		vprev = null;
		while (vlist != null) {
			if (vlist.vLoc1 >= lft) {
				break;
			}
			vprev = vlist;
			vlist = vlist.vNxt;
		}
		while (vlist != null && vlist.vLoc1 == lft) {
			if (vlist.vLoc2 >= rght) {
				break;
			}
			vprev = vlist;
			vlist = vlist.vNxt;
		}
		if (vprev == null) {
			mValList = item;
		} else {
			vprev.vNxt = item;
		}
		item.vNxt = vlist;
	}

	private void addVValue(int/*ACFixed*/ lft, int/*ACFixed*/ rght, int/*ACFixed*/ val, int/*ACFixed*/ spc, ClrSeg lSeg, ClrSeg rSeg)
	{
		if (val == 0) {
			return;
		}
		if (lePruneValue(val) && spc <= 0) {
			return;
		}
		if (lSeg != null && lSeg.sType == S_BEND && rSeg != null && rSeg.sType == S_BEND) {
			return;
		}
		if (val <= mPruneD && spc <= 0 && lSeg != null && rSeg != null) {
			if (lSeg.sType == S_BEND || rSeg.sType == S_BEND || !checkBBoxes(lSeg.sElt, rSeg.sElt)) {
				return;
			}
		}
		insertVValue(lft, rght, val, spc, lSeg, rSeg);
	}

	/* MA - create a a new ClrVal and insert it into a sorted list vaList */
	private void insertHValue(int/*ACFixed*/ bot, int/*ACFixed*/ top, int/*ACFixed*/ val, int/*ACFixed*/ spc, ClrSeg bSeg, ClrSeg tSeg, boolean ghst)
	{
		ClrVal item, vlist, vprev, vl;
		vlist = mValList;
		vprev = null;
		while (vlist != null) {
			if (vlist.vLoc2 >= top) {
				break;
			}
			vprev = vlist;
			vlist = vlist.vNxt;
		}
		while (vlist != null && vlist.vLoc2 == top) {
			if (vlist.vLoc1 >= bot) {
				break;
			}
			vprev = vlist;
			vlist = vlist.vNxt;
		}
		/* prune ghost pair that is same as non ghost pair for same segment
		   only if val for ghost is less than an existing val with same
		   top and bottom segment (vl) */
		vl = vlist;
		while (ghst && vl != null && vl.vLoc2 == top && vl.vLoc1 == bot) {
			if (!vl.vGhst && (vl.vSeg1 == bSeg || vl.vSeg2 == tSeg) && vl.vVal > val) {
				return;
			}
			vl = vl.vNxt;
		}
		item = new ClrVal();
		item.cvlSN = ++mCvlSN;
		item.vVal = val;
		item.initVal = val;
		item.vSpc = spc;
		item.vLoc1 = bot;
		item.vLoc2 = top;
		item.vSeg1 = bSeg;
		item.vSeg2 = tSeg;
		item.vGhst = ghst;
		if (vprev == null) {
			mValList = item;
		} else {
			vprev.vNxt = item;
		}
		item.vNxt = vlist;
	}

	/* MA - if the top & bot pair looks reasonable insert it to vaList */
	private void addHValue(int/*ACFixed*/ bot, int/*ACFixed*/ top, int/*ACFixed*/ val, int/*ACFixed*/ spc, ClrSeg bSeg, ClrSeg tSeg)
	{
		boolean ghst;
		if (val == 0) {
			return;
		}
		if (lePruneValue(val) && spc <= 0) {
			return;
		}
		if (bSeg.sType == S_BEND && tSeg.sType == S_BEND) {
			return;
		}
		ghst = bSeg.sType == S_GHOST || tSeg.sType == S_GHOST;
		if (!ghst && val <= mPruneD && spc <= 0) {
			if (bSeg.sType == S_BEND || tSeg.sType == S_BEND || !checkBBoxes(bSeg.sElt, tSeg.sElt)) {
				return;
			}
		}
		insertHValue(bot, top, val, spc, bSeg, tSeg, ghst);
	}

	private double mfabs(double in)
	{
		if (in > 0) {
			return in;
		}
		return -in;
	}

	/* MA - calculate v1 + v2 + 2*sqrt(v1*v2) */
	private int/*ACFixed*/ combVals(int/*ACFixed*/ v1, int/*ACFixed*/ v2)
	{
		int i;
		double r1, r2;
		double x, a, xx = 0.0;
		r1 = fixToFlt(v1);
		r2 = fixToFlt(v2);
		/* home brew sqrt */
		a = r1 * r2;
		x = a;
		for (i = 0; i < 16; i++) {
			xx = 0.5 * (x + a / x);
			if (i >= 8 && mfabs(xx - x) <= mfabs(xx) * 0.0000001) {
				break;
			}
			x = xx;
		}
		r1 += r2 + 2.0 * xx;
		if (r1 > mMaxVal) {
			r1 = mMaxVal;
		} else if (r1 > 0 && r1 < mMinVal) {
			r1 = mMinVal;
		}
		return fltToFix(r1);
	}

	/* MA - segments at the same top & bot get increased values */
	private void combineValues()
	{
		/* works for both H and V */
		ClrVal vlist, v1;
		int/*ACFixed*/ loc1, loc2;
		int/*ACFixed*/ val;
		boolean match;
		vlist = mValList;
		while (vlist != null) {
			v1 = vlist.vNxt;
			loc1 = vlist.vLoc1;
			loc2 = vlist.vLoc2;
			val = vlist.vVal;
			match = false;
			while (v1 != null && v1.vLoc1 == loc1 && v1.vLoc2 == loc2) {
				if (v1.vGhst) {
					val = v1.vVal;
				} else {
					val = combVals(val, v1.vVal);
				}
				/* increase value to compensate for length squared effect */
				match = true;
				v1 = v1.vNxt;
			}
			if (match) {
				while (vlist != v1) {
					vlist.vVal = val;
					vlist = vlist.vNxt;
				}
			} else {
				vlist = v1;
			}
		}
	}

	private void evalV()
	{
		ClrSeg lList, rList;
		int/*ACFixed*/ lft, rght;
		int/*ACFixed*/[] val = new int[1];
		int/*ACFixed*/[] spc = new int[1];
		mValList = null;
		lList = leftList();
		while (lList != null) {
			rList = rightList();
			while (rList != null) {
				lft = lList.sLoc;
				rght = rList.sLoc;
				if (lft < rght) {
					evalVPair(lList, rList, spc, val);
					addVValue(lft, rght, val[0], spc[0], lList, rList);
				}
				rList = rList.sNxt;
			}
			lList = lList.sNxt;
		}
		combineValues();
	}

	private void evalH()
	{
		ClrSeg bList, tList, lst, ghostSeg;
		int/*ACFixed*/ lstLoc, tempLoc, cntr;
		int/*ACFixed*/ val, spc;
		/* MA - find every reasonable segment pair and insert it as a stem */
		mValList = null;
		bList = botList();
		while (bList != null) {
			tList = topList();
			while (tList != null) {
				int/*ACFixed*/ bot, top;
				bot = bList.sLoc;
				top = tList.sLoc;
				if ((bot < top && Y_GOES_UP) || (bot > top && !Y_GOES_UP)) {
					int[] spca = new int[1];
					int[] vala = new int[1];
					evalHPair(bList, tList, spca, vala);
					spc = spca[0];
					val = vala[0];
					addHValue(bot, top, val, spc, bList, tList);
				}
				tList = tList.sNxt;
			}
			bList = bList.sNxt;
		}
		ghostSeg = new ClrSeg();
		ghostSeg.csgSN = ++mCsgSN;
		ghostSeg.sType = S_GHOST;
		ghostSeg.sElt = null;
		if (mLenBotBands >= 2 || mLenTopBands >= 2) {
			/* MA - generate bottom ghost stems */
			lst = botList();
			while (lst != null) {
				lstLoc = lst.sLoc;
				if (inBlueBand(lstLoc, mLenBotBands, mBotBands)) {
					tempLoc = lstLoc;
					if (Y_GOES_UP) {
						tempLoc += mGhostWidth;
					} else {
						tempLoc -= mGhostWidth;
					}
					ghostSeg.sLoc = tempLoc;
					cntr = (lst.sMax + lst.sMin)/2;
					ghostSeg.sMax = cntr + mGhostLength/2;
					ghostSeg.sMin = cntr - mGhostLength/2;
					spc = fixInt(2);
					val = fixInt(20);
					addHValue(lstLoc, tempLoc, val, spc, lst, ghostSeg);
				}
				lst = lst.sNxt;
			}
			/* MA - create top ghost stems */
			lst = topList();
			while (lst != null) {
				lstLoc = lst.sLoc;
				if (inBlueBand(lstLoc, mLenTopBands, mTopBands)) {
					tempLoc = lstLoc;
					if (!Y_GOES_UP) {
						tempLoc += mGhostWidth;
					} else {
						tempLoc -= mGhostWidth;
					}
					ghostSeg.sLoc = tempLoc;
					cntr = (lst.sMin + lst.sMax)/2;
					ghostSeg.sMax = cntr + mGhostLength/2;
					ghostSeg.sMin = cntr - mGhostLength/2;
					spc = fixInt(2);
					val = fixInt(20);
					addHValue(tempLoc, lstLoc, val, spc, ghostSeg, lst);
				}
				lst = lst.sNxt;
			}
			/* MA - as a fallback generate ghost stems at both character extremes */
			if (mLenTopBands != 0 || mLenBotBands != 0) {
				ACBBox bbox = new ACBBox();
				int/*ACFixed*/ top, bot;
				ClrSeg	seg;
				findPathBBox(bbox);
				if (Y_GOES_UP) {
					top = bbox.ymax;
					bot = bbox.ymin;
				} else {
					top = bbox.ymin;
					bot = bbox.ymax;
				}
				seg = new ClrSeg();
				seg.csgSN = ++mCsgSN;
				seg.sType = S_LINE;
				seg.sElt = null;
				seg.sMin = FIXED_NEG_INF;
				seg.sMax = FIXED_POS_INF;
				ghostSeg.sMax = mGhostLength/2;
				ghostSeg.sMin = -mGhostLength/2;
				spc = fixInt(2);
				val = fixInt(20);
				if (inBlueBand(bot, mLenBotBands, mBotBands)) {
					seg.sLoc = bot;
					tempLoc = bot;
					if (Y_GOES_UP) {
						tempLoc += mGhostWidth;
					} else {
						tempLoc -= mGhostWidth;
					}
					ghostSeg.sLoc = tempLoc;
					addHValue(bot, tempLoc, val, spc, seg, ghostSeg);
				}
				if (inBlueBand(top, mLenTopBands, mTopBands)) {
					seg.sLoc = top;
					tempLoc = top;
					if (!Y_GOES_UP) {
						tempLoc += mGhostWidth;
					} else {
						tempLoc -= mGhostWidth;
					}
					ghostSeg.sLoc = tempLoc;
					addHValue(tempLoc, top, val, spc, ghostSeg, seg);
				}
			}
		}
		combineValues();
	}

	/* Module ACwind.c */

	private boolean inRange(int x, int x1, int x2)
	{
		return (x >= x1 && x <= x2);
	}

	private void flatReportProc(int frp, Object param, Cd c)
	{
		switch (frp) {
		case FRP_NZWIND:
			nzWindProc(param, c);
			break;
		case FRP_YEXTREME:
			findYExtremeProc(param, c);
			break;
		case FRP_CHKDT:
			chkDT(param, c);
			break;
		case FRP_CHKBBDT:
			chkBBDT(param, c);
			break;
		case FRP_FPBBOXPT:
			fpBBoxPt(param, c);
			break;
		}
	}

	/* walk the sub-path and call the proc for each point
	 */
	private void walkSubPath(PathElt sp, boolean yExtreme, Object param)
	{
		PathElt e = sp;
		Cd curPt = new Cd();
		FltnRec fltnrec = new FltnRec();
		Cd c1 = new Cd();
		Cd c2 = new Cd();
		Cd c3 = new Cd();
		boolean shortcut;
		while (e != null) {
			switch (e.type) {
			case CURVETO:
				flatReportProc((yExtreme) ? FRP_YEXTREME : FRP_NZWIND, param, curPt);
				fltnrec.param = param;
				c1.x = e.x1;
				c1.y = e.y1;
				c2.x = e.x2;
				c2.y = e.y2;
				c3.x = e.x3;
				c3.y = e.y3;
				if (yExtreme) {
					fltnrec.report = FRP_YEXTREME;
					shortcut = yExtremeShortCutProc(curPt, c1, c2, c3, param, fltnrec.report);
				} else {
					fltnrec.report = FRP_NZWIND;
					shortcut = nzWindShortCutProc(curPt, c1, c2, c3, param, fltnrec.report);
				}
				if (!shortcut) {
					fltnCurve(curPt, c1, c2, c3, fltnrec);
				}
				curPt.copyFrom(c3);
				break;
			case LINETO:
				flatReportProc((yExtreme) ? FRP_YEXTREME : FRP_NZWIND, param, curPt);
				curPt.x = e.x;
				curPt.y = e.y;
				break;
			case MOVETO:
				curPt.x = e.x;
				curPt.y = e.y;
				break;
			case CLOSEPATH:
				flatReportProc((yExtreme) ? FRP_YEXTREME : FRP_NZWIND, param, curPt);
				return;
			}
			e = e.next;
		}
	}

	/* given the y extreme point and the two adjacent points
	   determine the winding direction of the subpath
	*/
	private int windTest(Cd prevPt, Cd topPt, Cd nextPt)
	{
		double px, py, tx, ty, ny;
		double x, nx;
		/* check simple cases first */
		if (prevPt.x > topPt.x && topPt.x > nextPt.x) {
			return WD_CCW;
		}
		if (prevPt.x < topPt.x && topPt.x < nextPt.x) {
			return WD_CW;
		}
		if (prevPt.y == topPt.y) {
			if (prevPt.x >= topPt.x) {
				return WD_CCW;
			} else {
				return WD_CW;
			}
		}
		if (topPt.y == nextPt.y) {
			if (topPt.x >= nextPt.x) {
				return WD_CCW;
			} else {
				return WD_CW;
			}
		}
		/* hard case */
		px = fixToFlt(prevPt.x);
		py = fixToFlt(prevPt.y);
		tx = fixToFlt(topPt.x);
		ty = fixToFlt(topPt.y);
		nx = fixToFlt(nextPt.x);
		ny = fixToFlt(nextPt.y);
		x = ((tx - px)*ny + (px*ty - tx*py))/(ty - py);
		if (nx <= x) {
			return WD_CCW;
		} else {
			return WD_CW;
		}
	}

	/* while walking around a sub-path collect at most four points around the y extreme.
	 * we need to examine the beginning of the y extreme (risePt) and the ending (lastPt)
	 * because an irregular path may move back and forth horizontally at the y extreme
	 * before determining the actual direction of the path.
	 * when the current point starts sinking, determine the winding direction by examining:
	 * if risePt == lastPt, then three points: prerisePt, risePt, and lastPt
	 * if risePt != lastPt, then two points: risePt, lastPt
	 */
	private void findYExtremeProc(Object param, Cd curPt)
	{
		FindExParam xp = (FindExParam)param;
		/* if this point is the same as the previous, skip it */
		if (xp.hasLastPt && xp.lastPt.x == curPt.x && xp.lastPt.y == curPt.y) {
			return;
		}
		/* check if the current point is sinking from the current max y
		 * and have enough points. if so, determine the winding direction.
		 */
		if (xp.lastWasExtreme && (curPt.y < xp.yMax)) {
			xp.plaeauEnded = true;
			if (xp.hasRisePt) {
				if (xp.risePt.x != xp.lastPt.x) {
					xp.windDir = (xp.risePt.x > xp.lastPt.x) ? WD_CCW: WD_CW;
				} else {
					if (xp.hasPrerisePt) {
						xp.windDir = windTest(xp.prerisePt, xp.risePt, curPt);
						xp.firstWasExtreme = false;
					}
				}
			} else {
				if (!xp.firstWasExtreme) {
					/* no rise point means that the first point is at the extreme
					 * save the sinking point for the exception case later
					 */
					xp.firstWasExtreme = true;
					xp.risePt.copyFrom(xp.firstPt);
					xp.beforeFirstSinkPt.copyFrom(xp.lastPt);
					xp.firstSinkPt.copyFrom(curPt);
				}
			}
		}
		/* update the prerisePt and the risePt
		 */
		if (xp.hasLastPt) {
			/* if the current point is the new y extreme, then it becomes
			 * the new risePt. the lastPt becomes the prerisePt
			 */
			if ((curPt.y > xp.yMax) || xp.plaeauEnded && (curPt.y == xp.yMax)) {
				xp.plaeauEnded = false;
				xp.risePt.copyFrom(curPt);
				xp.hasRisePt = true;
				xp.prerisePt.copyFrom(xp.lastPt);
				xp.hasPrerisePt = true;
				xp.firstWasExtreme = false;
			}
		} else {
			/* save the first point as the default sink point */
			xp.firstSinkPt.copyFrom(curPt);
			xp.firstPt.copyFrom(curPt);
		}
		/* memorize the prerise point and the rise point for the first point at extreme */
		if (xp.firstWasExtreme) {
			if (curPt.y < xp.yMax) {
				xp.prerisePt.copyFrom(curPt);
				xp.hasPrerisePt = true;
			}
			if (xp.hasPrerisePt && curPt.y == xp.yMax && !xp.hasRisePtForFirst) {
				xp.risePt.copyFrom(curPt);
				xp.hasRisePtForFirst = true;
			}
		}
		/* update the last point and the yMax */
		xp.lastPt.copyFrom(curPt);
		xp.hasLastPt = true;
		xp.lastWasExtreme = (curPt.y >= xp.yMax);
		if (xp.lastWasExtreme) {
			xp.yMax = curPt.y;
		}
	}

	// returns true if this curve is below the current y max
	private boolean yExtremeShortCutProc(Cd c0, Cd c1, Cd c2, Cd c3, Object param, int frp)
	{
		FindExParam xp = (FindExParam)param;
		int/*ACFixed*/ yMax = xp.yMax;
		if (c0.y < yMax && c1.y < yMax && c2.y < yMax && c3.y < yMax) {
			flatReportProc(frp, param, c3);	// pretend a line to the end point
			return true;
		} else {
			return false;
		}
	}

	/* collect the head of sub-path, bbox etc */
	private void collectSubPathInfo()
		throws ACException
	{
		PathElt e;
		Cd cd = new Cd();
		SubPathInfo sp;
		int i;
		int[] elemCount = new int[1];
		mNumSubPaths = countSubPaths(elemCount);
		if (mNumSubPaths > MAX_NUM_SUBPATHS || elemCount[0] > MAX_NUM_PATH_ELEMENTS)
			throw new ACException("Glyph too complex");
		mSubPaths = new SubPathInfo[mNumSubPaths];
		/* collect the sub-path information */
		for (i = 0, e = mPathEnd; i < mNumSubPaths && e != null;) {
			e = getDest(e);
			if (e != null) {
				ACBBox bbox = new ACBBox();
				calcSubpathBBox(bbox, e);
				sp = mSubPaths[i++] = new SubPathInfo();
				sp.head = e;
				sp.startPt.x = e.x;
				sp.startPt.y = e.y;
				sp.bbox = bbox;
				/* determine the last point */
				sp.tail = getClosedBy(e);
				if (sp.tail == null) {
					continue;
				}
				getEndPoint(sp.tail.prev, cd);
				sp.endPt.x = cd.x;
				sp.endPt.y = cd.y;
				e = e.prev;
			}
		}
	}

	/* determine the winding direction of each sub-path by
	 * checking the path direction at the Y extreme
	 * while iterate over flattened points of the sub-path we examine
	 * points at the y extreme of the sub-path
	 * if the start point or the end point are at the extreme,
	 * they are examined later after we have the necessary points
	 * for the test
	 */
	private void determineCurrentWind(SubPathInfo[] subPaths, int numSubPaths)
	{
		int i;
		for (i = 0; i < numSubPaths; i++) {
			SubPathInfo subPath = subPaths[i];
			PathElt e = subPath.head;
			FindExParam param = new FindExParam();
			param.yMax = FIXED_NEG_INF;
			walkSubPath(e, true, param);
			/* handle exceptional cases:
			 * 1. the first point was at the extreme, and we didn't see the prerisePt (the last point is the candidate)
			 * 2. the last point was at the extreme, and we didn't see the sinking point (the first point is the candidate)
			 */
			if (param.firstWasExtreme) {
				/* the first point was at the extreme.
				 * determine using prerisePt, risePt, beforeFirstSinkPt, firstSinkPt
				 */
				if (param.risePt.x != param.beforeFirstSinkPt.x) {
					subPath.currentWind = (param.risePt.x > param.beforeFirstSinkPt.x) ? WD_CCW: WD_CW;
				} else {
					subPath.currentWind = windTest(param.prerisePt, param.beforeFirstSinkPt, param.firstSinkPt);
				}
			} else {
				if (param.lastWasExtreme) {
					/* the last point was at the extreme. the first point was not so it must be the sinking point.
					 * test using the four points: prerisePt, risePt, lastPt, firstSinkPt
					 */
					if (param.risePt.x != param.lastPt.x) {
						subPath.currentWind = (param.risePt.x > param.lastPt.x) ? WD_CCW: WD_CW;
					} else {
						subPath.currentWind = windTest(param.prerisePt, param.lastPt, param.firstSinkPt);
					}
				} else {
					/* otherwise, findYExtremeProc must have deterined the current direction */
					subPath.currentWind = param.windDir;
				}
			}
		}
	}

	private int testHighLow(int/*ACFixed*/ y1, int/*ACFixed*/ y2)
	{
		if (y1 == y2) {
			return 0;
		}
		if (y1 < y2) {
			return 1;
		} else {
			return -1;
		}
	}

	private void nzWindProc(Object param, Cd pt)
	{
		NZWindParam eop = (NZWindParam)param;
		int test1, test2;
		double x1, x2, y1, y2, x;
		if (eop.hasLastPt) {
			/* quick test: see if the two points are above and
			 * below of the test line
			 */
			test1 = testHighLow(eop.testPt.y, eop.lastPt.y);
			test2 = testHighLow(eop.testPt.y, pt.y);
			if (((test1 * test2) <= 0) /* one point is above and another is below? */
			    /* another quick test: if both points are located on the opposite of the test point
			     * on the test line, it can't cross the test line
			     */
			    && (eop.testToRight && (eop.lastPt.x > eop.testPt.x || pt.x > eop.testPt.x)
				|| (!eop.testToRight && (eop.lastPt.x < eop.testPt.x || pt.x < eop.testPt.x)))
			    /* horizontal line segment cannot cross with the test line */
			    && (eop.lastPt.y != pt.y)) {
				/* special case: this point is on the test line
				 */
				if (pt.y == eop.testPt.y && (eop.testToRight ^ (pt.x > eop.testPt.x)) == false) {
					eop.onLine = true;
					eop.fromAbove = eop.lastPt.y > eop.testPt.y;
					eop.lastPt.copyFrom(pt);
					eop.hasLastPt = true;
					return;
				}
				/* special case: last point was on the test line
				 */
				if (eop.onLine) {
					if (eop.fromAbove && (test2 < 0) ||(!eop.fromAbove && (test2 > 0))) {
						eop.windCount += (eop.testToRight ^ (test2 < 0)) ? 1: -1;
					}
					eop.onLine = false;
					eop.lastPt.copyFrom(pt);
					eop.hasLastPt = true;
					return;
				}
				/* determine the coordinates where the line segment and the test line crosses
				 * do this math in floating point as a fixed math may overflow
				 */
				x1 = fixToFlt(eop.lastPt.x);
				y1 = fixToFlt(eop.lastPt.y);
				x2 = fixToFlt(pt.x);
				y2 = fixToFlt(pt.y);
				/* calculate the x coordinate where the line segment crosses the test line */
				x = (eop.testy*(x2 - x1) + x1*y2 - x2*y1)/(y2 - y1);
				/* if it is to the same side as the test line, two lines are crossing */
				if ((eop.testToRight ^ (x > eop.testx)) == false) {
					eop.windCount += (eop.testToRight ^ (y1 > y2)) ? 1: -1;
				}
			}
		} else {
			eop.firstPt.copyFrom(pt);
		}
		eop.lastPt.copyFrom(pt);
		eop.hasLastPt = true;
	}

	// returns true if this curve does not cross the horizontal test line vertically
	private boolean nzWindShortCutProc(Cd c0, Cd c1, Cd c2, Cd c3, Object param, int frp)
	{
		NZWindParam eop = (NZWindParam)param;
		int/*ACFixed*/ testy = eop.testPt.y;
		if ( (c0.y > testy && c1.y > testy && c2.y > testy && c3.y > testy)
		     || (c0.y < testy && c1.y < testy && c2.y < testy && c3.y < testy)) {
			flatReportProc(frp, param, c3);	// pretend a line to the end point
			return true;
		} else {
			return false;
		}
	}

	/* do non-zero winding test and return the number of crosses with the sub path
	 */
	private int nzWindTest(Cd pt, SubPathInfo sp)
	{
		NZWindParam param = new NZWindParam();
		param.testPt.copyFrom(pt);
		param.testx = fixToFlt(pt.x);
		param.testy = fixToFlt(pt.y);
		/* in order to simplify the test against the line segment
		 * from the end of the subpath to the first,
		 * draw the test line to the other side of the subpath start point
		 * in the x coordinate
		 */
		param.testToRight = sp.startPt.x < pt.x;
		walkSubPath(sp.head, false, param);
		/* draw another line segment from the subpath end to the start */
		if (param.hasLastPt) {
			nzWindProc(param, param.firstPt);
		}
		return param.windCount;
	}

	/* determine if each sub-path should be filled or unfilled
	 * this is done by the non-zero winding test of the startPt of a sub-path
	 * against other sub-paths. note that this test may err if
	 * sub-paths are intersecting or touching at the sub-path startPt
	 */
	private void determineFillOrUnfill(SubPathInfo[] subPaths, int numSubPaths)
	{
		int i, j;
		for (i = 0; i < numSubPaths; i++) {
			int windCount = 0;
			Cd startPt = (Cd)subPaths[i].startPt.clone();
			for (j = 0; j < numSubPaths; j++) {
				if (i == j) {
					continue;
				}
				/* quick test: if the point is completely outside of the bbox
				 * of this sub-path, no need to test against it
				 */
				if (!(inRange(startPt.x, subPaths[j].bbox.xmin, subPaths[j].bbox.xmax) &&
				      inRange(startPt.y, subPaths[j].bbox.ymin, subPaths[j].bbox.ymax))) {
					continue;
				}
				windCount += nzWindTest(startPt, subPaths[j]);
			}
			/* add the self winding count */
			if (subPaths[i].currentWind == WD_CCW) {
				windCount++;
			} else {
				windCount--;
			}
			/* the sub-path should be filled if the winding count is non-zero */
			subPaths[i].filled = (windCount != 0);
		}
	}

	/* flip the winding direction of the subpath.
	 * each path element is changed as follows:
	 * MOVETO = > CLOSEPATH
	 * CLOSEPATH = > MOVETO
	 * LINETO, CURVETO = > points are flipped
	 */
	private void flipSubPath(SubPathInfo sp)
	{
		PathElt e, next, prev;
		PathElt prevSubPath, nextSubPath;
		PathElt newHead, newTail;
		int/*ACFixed*/ cx = 0, cy = 0;		/* current point */
		int/*ACFixed*/ ncx, ncy;
		int/*ACFixed*/ x1, y1;
		/* save the linkage to other subpaths for later re-linking */
		e = sp.head;
		prevSubPath = e.prev;
		nextSubPath = sp.tail.next;
		newHead = null;
		newTail = null;
		for (; e != null; e = next) {
			next = e.next;
			prev = e.prev;
			switch (e.type) {
			case CURVETO:
				x1 = e.x1;
				y1 = e.y1;
				ncx = e.x3;
				ncy = e.y3;
				e.x1 = e.x2;
				e.y1 = e.y2;
				e.x2 = x1;
				e.y2 = y1;
				e.x3 = cx;
				e.y3 = cy;
				cx = ncx;
				cy = ncy;
				break;
			case LINETO:
				ncx = e.x;
				ncy = e.y;
				e.x = cx;
				e.y = cy;
				cx = ncx;
				cy = ncy;
				break;
			case MOVETO:
				newTail = e;
				cx = e.x;
				cy = e.y;
				e.type = CLOSEPATH;
				e.next = nextSubPath;
				/* re-link the next subpath to this new end */
				if (nextSubPath != null) {
					nextSubPath.prev = e;
				} else {
					mPathEnd = e;
				}
				e.prev = next;
				e.next = nextSubPath;
				continue;
			case CLOSEPATH:
				newHead = e;
				e.type = MOVETO;
				e.x = cx;
				e.y = cy;
				e.x = sp.tail.x;
				e.y = sp.tail.y;
				/* re-link the previous subpath to this new start */
				if (prevSubPath != null) {
					prevSubPath.next = e;
				} else {
					mPathStart = e;
				}
				e.next = prev;
				e.prev = prevSubPath;
				sp.head = newHead;		/* exit: */
				sp.tail = newTail;
				return;
			}
			e.next = prev;
			e.prev = next;
		}
		sp.head = newHead;
		sp.tail = newTail;
	}

	/* if a subpath is not in the correct winding direction, flip it */
	private void flipSubPaths(SubPathInfo[] subPaths, int numSubPaths, boolean forceCCW)
	{
		int i;
		for (i = 0; i < numSubPaths; i++) {
			SubPathInfo sp = subPaths[i];
			boolean correct = (sp.currentWind == WD_CCW) ^ (sp.filled != false) ^ (forceCCW != false);
			if (!correct) {
				flipSubPath(sp);
			}
		}
	}

	/* Note: this code assumes Y_GOES_UP. If it isn't the case, the caller
	 * should flip "forceCCW" to compensate it
	 */
	private void fixPathWind(boolean forceCCW)
	{
		determineCurrentWind(mSubPaths, mNumSubPaths);
		determineFillOrUnfill(mSubPaths, mNumSubPaths);
		flipSubPaths(mSubPaths, mNumSubPaths, forceCCW);
	}

	/* Module ACmerge.c */

	/* true iff you can go from e1 to e2 without going out of band loc1..loc2 */
	/* if vert is true, then band is vert (test x values) */
	/* else band is horizontal (test y values) */
	/* band is expanded by CLSMRG in each direction */
	private boolean closeElements(PathElt e1, PathElt e2, int/*ACFixed*/ loc1, int/*ACFixed*/ loc2, boolean vert)
	{
		int/*ACFixed*/ tmp;
		Cd cd = new Cd();
		PathElt e;
		int/*ACFixed*/ clsmrg;
		if (e1 == e2) {
			return true;
		}
		clsmrg = psDist(20);
		if (loc1 < loc2) {
			loc1 -= clsmrg;
			loc2 += clsmrg;
		} else {
			tmp = loc1;
			loc1 = loc2-clsmrg;
			loc2 = tmp + clsmrg;
		}
		e = e1;
		while (true) {
			if (e == e2) {
				return true;
			}
			getEndPoint(e, cd);
			tmp = vert ? cd.x : cd.y;
			if (tmp > loc2 || tmp < loc1) {
				return false;
			}
			if (e.type == CLOSEPATH) {
				e = getDest(e);
			} else {
				e = e.next;
			}
			if (e == e1) {
				return false;
			}
		}
	}

	private boolean closeSegs(ClrSeg s1, ClrSeg s2, boolean vert)
	{
		/* true iff the elements for these segs are "close" in the path */
		PathElt e1, e2;
		int/*ACFixed*/ loc1, loc2;
		if (s1 == s2) {
			return true;
		}
		e1 = s1.sElt;
		e2 = s2.sElt;
		if (e1 == null || e2 == null) {
			return true;
		}
		loc1 = s1.sLoc;
		loc2 = s2.sLoc;
		return (closeElements(e1, e2, loc1, loc2, vert) ||
			closeElements(e2, e1, loc2, loc1, vert)) ? true : false;
	}

	private void doPrune()
	{
		ClrVal vL = mValList, vPrv;
		while (vL != null && vL.pruned) {
			vL = vL.vNxt;
		}
		mValList = vL;
		if (vL == null) {
			return;
		}
		vPrv = vL;
		vL = vL.vNxt;
		while (vL != null) {
			if (vL.pruned) {
				vPrv.vNxt = vL = vL.vNxt;
			} else {
				vPrv = vL;
				vL = vL.vNxt;
			}
		}
	}

	private ClrVal pruneOne(ClrVal sLst, boolean hFlg, ClrVal sL, int i)
	{
		sLst.pruned = true;
		return sLst.vNxt;
	}

	private boolean pruneLt(int/*ACFixed*/ val, int/*ACFixed*/ v)
	{
		if (v < FIXED_POS_INF / 10 && val < FIXED_POS_INF / PRNFCTR) {
			return (val * PRNFCTR < v * 10);
		}
		return (val / 10 < v / PRNFCTR);
	}

	private boolean pruneLe(int/*ACFixed*/ val, int/*ACFixed*/ v)
	{
		if (val < FIXED_POS_INF / PRNFCTR) {
			return (v <= val * PRNFCTR);
		}
		return (v / PRNFCTR <= val);
	}

	private boolean pruneGt(int/*ACFixed*/ val, int/*ACFixed*/ v)
	{
		if (val < FIXED_POS_INF / PRNFCTR) {
			return (v > val * PRNFCTR);
		}
		return (v / PRNFCTR > val);
	}

	private boolean pruneMuchGt(int/*ACFixed*/ val, int/*ACFixed*/ v)
	{
		if (val < FIXED_POS_INF / MUCHFCTR) {
			return (v > val * MUCHFCTR);
		}
		return (v / MUCHFCTR > val);
	}

	private boolean pruneVeryMuchGt(int/*ACFixed*/ val, int/*ACFixed*/ v)
	{
		if (val < FIXED_POS_INF / VERYMUCHFCTR) {
			return (v > val * VERYMUCHFCTR);
		}
		return (v / VERYMUCHFCTR > val);
	}

	/* The changes made here and in PruneHVals are to fix a bug in
	   MinisterLight/E where the top left point was not getting colored. */
	private void pruneVVals()
	{
		ClrVal sLst, sL;
		ClrSeg seg1, seg2, sg1, sg2;
		int/*ACFixed*/ lft, rht, l, r, prndist;
		int/*ACFixed*/ val, v;
		boolean flg, otherLft, otherRht;
		sLst = mValList;
		prndist = psDist(10);
		while (sLst != null) {
			flg = true;
			otherLft = otherRht = false;
			val = sLst.vVal;
			lft = sLst.vLoc1;
			rht = sLst.vLoc2;
			seg1 = sLst.vSeg1;
			seg2 = sLst.vSeg2;
			sL = mValList;
			while (sL != null) {
				v = sL.vVal;
				sg1 = sL.vSeg1;
				sg2 = sL.vSeg2;
				l = sL.vLoc1;
				r = sL.vLoc2;
				if ((l!= lft || r!= rht) && !pruneLe(val, v)) {
					if (rht + prndist >= r && lft - prndist <= l &&
					    (val < fixInt(100) && pruneMuchGt(val, v) ?
					     (closeSegs(seg1, sg1, true) || closeSegs(seg2, sg2, true)) :
					     (closeSegs(seg1, sg1, true) && closeSegs(seg2, sg2, true)))) {
						sLst = pruneOne(sLst, false, sL, 1);
						flg = false;
						break;
					}
					if (seg1 != null && seg2 != null) {
						if (Math.abs(l - lft) < FIX_ONE) {
							if (!otherLft && pruneLt(val, v) && Math.abs(l - r) < Math.abs(lft - rht) &&
							    closeSegs(seg1, sg1, true)) {
								otherLft = true;
							}
							if (seg2.sType == S_BEND && closeSegs(seg1, sg1, true)) {
								sLst = pruneOne(sLst, false, sL, 2);
								flg = false;
								break;
							}
						}
						if (Math.abs(r - rht) < FIX_ONE) {
							if (!otherRht && pruneLt(val, v) && Math.abs(l - r) < Math.abs(lft - rht) &&
							    closeSegs(seg2, sg2, true)) {
								otherRht = true;
							}
							if (seg1.sType == S_BEND && closeSegs(seg2, sg2, true)) {
								sLst = pruneOne(sLst, false, sL, 3);
								flg = false;
								break;
							}
						}
						if (otherLft && otherRht) {
							sLst = pruneOne(sLst, false, sL, 4);
							flg = false;
							break;
						}
					}
				}
				sL = sL.vNxt;
			}
			if (flg) {
				sLst = sLst.vNxt;
			}
		}
		doPrune();
	}

	private void pruneHVals()
	{
		ClrVal sLst, sL;
		ClrSeg seg1, seg2, sg1, sg2;
		int/*ACFixed*/ bot, top, t, b;
		int/*ACFixed*/ val, v, prndist;
		boolean flg, otherTop, otherBot, topInBlue, botInBlue, ghst;
		sLst = mValList;
		prndist = psDist(10);
		while (sLst != null) {
			flg = true;
			otherTop = otherBot = false;
			seg1 = sLst.vSeg1;
			seg2 = sLst.vSeg2;		/* seg1 is bottom, seg2 is top */
			ghst = sLst.vGhst;
			val = sLst.vVal;
			bot = sLst.vLoc1;
			top = sLst.vLoc2;
			topInBlue = inBlueBand(top, mLenTopBands, mTopBands);
			botInBlue = inBlueBand(bot, mLenBotBands, mBotBands);
			sL = mValList;
			while (sL != null) {
				sg1 = sL.vSeg1;
				sg2 = sL.vSeg2;	/* sg1 is b, sg2 is t */
				v = sL.vVal;
				if (!ghst && sL.vGhst && !pruneVeryMuchGt(val, v)) {
					sL = sL.vNxt;
					continue;
				}
				b = sL.vLoc1;
				t = sL.vLoc2;
				if (t == top && b == bot) {
					sL = sL.vNxt;
					continue;
				}
				if (pruneGt(val, v) &&
				    ((Y_GOES_UP	&& top + prndist >= t && bot - prndist <= b) ||
				     (!Y_GOES_UP && top - prndist <= t && bot + prndist >= b)) &&
				    (val < fixInt(100) && pruneMuchGt(val, v) ?
				     (closeSegs(seg1, sg1, false) || closeSegs(seg2, sg2, false)) :
				     (closeSegs(seg1, sg1, false) && closeSegs(seg2, sg2, false))) &&
				    (val < FIX_16
				     /* needs to be greater than FIX_ONE << 3 for
					HelveticaNeue 95 Black G has val == 2.125
					Poetica/ItalicOne H has val == .66	 */ ||
				     ((!topInBlue || top == t) && (!botInBlue || bot == b)))) {
					sLst = pruneOne(sLst, true, sL, 5);
					flg = false;
					break;
				}
				if (seg1 != null && seg2 != null) {
					if (Math.abs(b - bot) < FIX_ONE) {
						if (pruneGt(val, v) &&
						    !topInBlue && seg2.sType == S_BEND && closeSegs(seg1, sg1, false)) {
							sLst = pruneOne(sLst, true, sL, 6);
							flg = false;
							break;
						}
						if (!otherBot && pruneLt(val, v) && Math.abs(t - b) < Math.abs(top - bot)) {
							if (closeSegs(seg1, sg1, false)) {
								otherBot = true;
							}
						}
					}
					if (Math.abs(t - top) < FIX_ONE) {
						if (pruneGt(val, v) &&
						    !botInBlue && seg2.sType == S_BEND && closeSegs(seg1, sg1, false)) {
							sLst = pruneOne(sLst, true, sL, 7);
							flg = false;
							break;
						}
						if (!otherTop && pruneLt(val, v) && Math.abs(t - b) < Math.abs(top - bot)) {
							if (closeSegs(seg2, sg2, false)) {
								otherTop = true;
							}
						}
					}
					if (otherBot && otherTop) {
						sLst = pruneOne(sLst, true, sL, 8);
						flg = false;
						break;
					}
				}
				sL = sL.vNxt;
			}
			if (flg) {
				sLst = sLst.vNxt;
			}
		}
		doPrune();
	}

	private void findBestVals(ClrVal vL)
	{
		int/*ACFixed*/ bV, bS;
		int/*ACFixed*/ t, b;
		ClrVal vL2, vPrv, bstV;
		for (; vL != null; vL = vL.vNxt) {
			if (vL.vBst != null) {
				continue;	/* already assigned */
			}
			bV = vL.vVal;
			bS = vL.vSpc;
			bstV = vL;
			b = vL.vLoc1;
			t = vL.vLoc2;
			vL2 = vL.vNxt;
			vPrv = vL;
			for (; vL2 != null; vL2 = vL2.vNxt) {
				if (vL2.vBst != null || vL2.vLoc1 != b || vL2.vLoc2 != t) {
					continue;
				}
				if ((vL2.vSpc == bS && vL2.vVal > bV) || (vL2.vSpc > bS)) {
					bS = vL2.vSpc;
					bV = vL2.vVal;
					bstV = vL2;
				}
				vL2.vBst = vPrv;
				vPrv = vL2;
			}
			while (vPrv != null) {
				vL2 = vPrv.vBst;
				vPrv.vBst = bstV;
				vPrv = vL2;
			}
		}
	}

	/* The following changes were made to fix a problem in Ryumin-Light and
	   possibly other fonts as well. The old version causes bogus coloring
	   and extra newcolors. */
	private void replaceVals(int/*ACFixed*/ oldB, int/*ACFixed*/ oldT, int/*ACFixed*/ newB, int/*ACFixed*/ newT, ClrVal newBst, boolean vert)
	{
		ClrVal vL;
		for (vL = mValList; vL != null; vL = vL.vNxt) {
			if (vL.vLoc1 != oldB || vL.vLoc2 != oldT || vL.merge) {
				continue;
			}
			vL.vLoc1 = newB;
			vL.vLoc2 = newT;
			vL.vVal = newBst.vVal;
			vL.vSpc = newBst.vSpc;
			vL.vBst = newBst;
			vL.merge = true;
		}
	}

	private void mergeVals(boolean vert)
	{
		ClrVal vLst, vL;
		ClrVal bstV, bV;
		ClrSeg seg1, seg2, sg1, sg2;
		int/*ACFixed*/ bot, top, b, t;
		int/*ACFixed*/ val, v, spc, s;
		boolean ghst;
		findBestVals(mValList);
		for (vL = mValList; vL != null; vL = vL.vNxt) {
			vL.merge = false;
		}
		while (true) {
			/* pick best from mValList with merge field still set to false */
			vLst = mValList; vL = null;
			while (vLst != null) {
				if (!vLst.merge) {
					if (vL == null || compareValues(vLst.vBst, vL.vBst, SFACTOR, 0)) {
						vL = vLst;
					}
				}
				vLst = vLst.vNxt;
			}
			if (vL == null) {
				break;
			}
			vL.merge = true;
			ghst = vL.vGhst;
			b = vL.vLoc1;
			t = vL.vLoc2;
			sg1 = vL.vSeg1;	/* left or bottom */
			sg2 = vL.vSeg2;	/* right or top */
			bV = vL.vBst;
			v = bV.vVal;
			s = bV.vSpc;
			for (vLst = mValList; vLst != null; vLst = vLst.vNxt) {
				/* consider replacing vLst by vL */
				if (vLst.merge || ghst != vLst.vGhst) {
					continue;
				}
				bot = vLst.vLoc1;
				top = vLst.vLoc2;
				if (bot == b && top == t) {
					continue;
				}
				bstV = vLst.vBst;
				val = bstV.vVal;
				spc = bstV.vSpc;
				if ((top == t && closeSegs(sg2, vLst.vSeg2, vert) &&
				     (vert || (
					       !inBlueBand(t, mLenTopBands, mTopBands) &&
					       !inBlueBand(bot, mLenBotBands, mBotBands) &&
					       !inBlueBand(b, mLenBotBands, mBotBands)))) ||
				    (bot == b && closeSegs(sg1, vLst.vSeg1, vert) &&
				     (vert || (
					       !inBlueBand(b, mLenBotBands, mBotBands) &&
					       !inBlueBand(t, mLenTopBands, mTopBands) &&
					       !inBlueBand(top, mLenTopBands, mTopBands)))) ||
				    (Math.abs(top - t) <= mMaxMerge && Math.abs(bot - b) <= mMaxMerge &&
				     (vert || (t == top || !inBlueBand(top, mLenTopBands, mTopBands))) &&
				     (vert || (b == bot || !inBlueBand(bot, mLenBotBands, mBotBands))))) {
					if (s == spc && val == v && !vert) {
						if (inBlueBand(t, mLenTopBands, mTopBands)) {
							if ((Y_GOES_UP && t > top) || (!Y_GOES_UP && t < top)) {
								replaceVals(bot, top, b, t, bV, vert);
								continue;
							}
						} else {
							if (inBlueBand(b, mLenBotBands, mBotBands)) {
								if ((Y_GOES_UP && b < bot) || (!Y_GOES_UP && b > bot)) {
									replaceVals(bot, top, b, t, bV, vert);
									continue;
								}
							}
						}
					} else {
						replaceVals(bot, top, b, t, bV, vert);
						continue;
					}
				} else {
					if (s == spc && sg1 != null && sg2 != null) {
						seg1 = vLst.vSeg1;
						seg2 = vLst.vSeg2;
						if (seg1 != null && seg2 != null) {
							if (Math.abs(bot - b) <= FIX_ONE && Math.abs(top - t) <= mMaxBendMerge) {
								if (seg2.sType == S_BEND &&
								    (vert || !inBlueBand(top, mLenTopBands, mTopBands))) {
									replaceVals(bot, top, b, t, bV, vert);
									continue;
								}
							} else {
								if (Math.abs(top - t) <= FIX_ONE && Math.abs(bot - b) <= mMaxBendMerge) {
									if (v > val && seg1.sType == S_BEND &&
									    (vert || !inBlueBand(bot, mLenBotBands, mBotBands))) {
										replaceVals(bot, top, b, t, bV, vert);
										continue;
									}
								}
							}
						}
					}
				}
			}
			vL = vL.vNxt;
		}
	}

	/* Module ACcheck.c */

	private void initCheckData(CheckData d)
	{
		d.forMultiMaster = false;
	}

	private void chkBad(CheckData d)
	{
		d.reCheckSmooth = resolveConflictBySplit(d.e, false, null, null);
	}

	private boolean grTan(int/*ACFixed*/ n, int/*ACFixed*/ d)
	{
		return (Math.abs(n) * 100 > Math.abs(d) * S_CURVE_TAN);
	}

	private boolean lsTan(int/*ACFixed*/ n, int/*ACFixed*/ d)
	{
		return (Math.abs(n) * 100 < Math.abs(d) * S_CURVE_TAN);
	}

	private void chkYDIR(CheckData d)
	{
		if (d.y > d.yloc) {
			/* going up */
			if (d.ystate == GOINGUP) {
				return;
			}
			if (d.ystate == STARTING) {
				d.ystart = d.ystate = GOINGUP;
			} else /*if (d.ystate == GOINGDOWN)*/ {
				if (d.ystart == GOINGUP) {
					d.yflatendx = d.xloc;
					d.yflatendy = d.yloc;
				} else {
					if (!d.yflat) {
						d.yflatstartx = d.xloc;
						d.yflatstarty = d.yloc;
						d.yflat = true;
					}
				}
				d.ystate = GOINGUP;
			}
		} else {
			if (d.y < d.yloc) {
				/* going down */
				if (d.ystate == GOINGDOWN) {
					return;
				}
				if (d.ystate == STARTING) {
					d.ystart = d.ystate = GOINGDOWN;
				} else /*if (d.ystate == GOINGUP)*/ {
					if (d.ystart == GOINGDOWN) {
						d.yflatendx = d.xloc;
						d.yflatendy = d.yloc;
					} else {
						if (!d.yflat) {
							d.yflatstartx = d.xloc;
							d.yflatstarty = d.yloc;
							d.yflat = true;
						}
					}
					d.ystate = GOINGDOWN;
				}
			}
		}
	}

	private void chkYFLAT(CheckData d)
	{
		if (!d.yflat) {
			if (lsTan(d.y - d.yloc, d.x - d.xloc)) {
				d.yflat = true;
				d.yflatstartx = d.xloc;
				d.yflatstarty = d.yloc;
			}
			return;
		}
		if (d.ystate != d.ystart) {
			return;
		}
		if (grTan(d.y - d.yloc, d.x - d.xloc)) {
			d.yflatendx = d.xloc;
			d.yflatendy = d.yloc;
			d.ydone = true;
		}
	}

	private void chkXFLAT(CheckData d)
	{
		if (!d.xflat) {
			if (lsTan(d.x - d.xloc, d.y - d.yloc)) {
				d.xflat = true;
				d.xflatstartx = d.xloc;
				d.xflatstarty = d.yloc;
			}
			return;
		}
		if (d.xstate != d.xstart) {
			return;
		}
		if (grTan(d.x - d.xloc, d.y - d.yloc)) {
			d.xflatendx = d.xloc;
			d.xflatendy = d.yloc;
			d.xdone = true;
		}
	}

	private void chkXDIR(CheckData d)
	{
		if (d.x > d.xloc) {
			/* going up */
			if (d.xstate == GOINGUP) {
				return;
			}
			if (d.xstate == STARTING) {
				d.xstart = d.xstate = GOINGUP;
			} else /*if (d.xstate == GOINGDOWN)*/ {
				if (d.xstart == GOINGUP) {
					d.xflatendx = d.xloc;
					d.xflatendy = d.yloc;
				} else {
					if (!d.xflat) {
						d.xflatstartx = d.xloc;
						d.xflatstarty = d.yloc;
						d.xflat = true;
					}
				}
				d.xstate = GOINGUP;
			}
		} else {
			if (d.x < d.xloc) {
				if (d.xstate == GOINGDOWN) {
					return;
				}
				if (d.xstate == STARTING) {
					d.xstart = d.xstate = GOINGDOWN;
				} else /*if (d.xstate == GOINGUP)*/ {
					if (d.xstart == GOINGDOWN) {
						d.xflatendx = d.xloc;
						d.xflatendy = d.yloc;
					} else {
						if (!d.xflat) {
							d.xflatstartx = d.xloc;
							d.xflatstarty = d.yloc;
							d.xflat = true;
						}
					}
					d.xstate = GOINGDOWN;
				}
			}
		}
	}

	private void chkDT(Object param, Cd c)
	{
		CheckData d = (CheckData)param;
		d.x = c.x;
		d.y = c.y;
		d.ynxt = d.y;
		d.xnxt = d.x;
		if (!d.ydone) {
			chkYDIR(d);
			chkYFLAT(d);
			if (d.ydone && d.yflat &&
			    Math.abs(d.yflatstarty - d.cy0) > fixInt(4) &&
			    Math.abs(d.cy1 - d.yflatendy) > fixInt(4)) {
				if ((d.ystart == GOINGUP && d.yflatstarty - d.yflatendy > fixInt(4)) ||
				    (d.ystart == GOINGDOWN && d.yflatendy - d.yflatstarty > fixInt(4))) {
					if (mEditChar && !d.forMultiMaster) {
						chkBad(d);
					}
					return;
				}
				if (Math.abs(d.yflatstartx - d.yflatendx) > fixInt(5)) {
					if (!d.forMultiMaster) {
						addHSegment(d.yflatstartx, d.yflatendx, (d.yflatstarty + d.yflatendy)/2,
							    d.e, null, S_CURVE, 13);
					}
				}
			}
		}
		if (!d.xdone) {
			chkXDIR(d);
			chkXFLAT(d);
			if (d.xdone && d.xflat &&
			    Math.abs(d.xflatstartx - d.x0) > fixInt(4) &&
			    Math.abs(d.x1 - d.xflatendx) > fixInt(4)) {
				if ((d.xstart == GOINGUP && d.xflatstartx - d.xflatendx > fixInt(4)) ||
				    (d.xstart == GOINGDOWN && d.xflatendx - d.xflatstartx > fixInt(4))) {
					if (mEditChar && !d.forMultiMaster) {
						chkBad(d);
					}
					return;
				}
				if (Math.abs(d.xflatstarty - d.xflatendy) > fixInt(5)) {
					if (!d.forMultiMaster) {
						addVSegment(d.xflatstarty, d.xflatendy, (d.xflatstartx + d.xflatendx)/2,
							    d.e, null, S_CURVE, 13);
					}
				}
			}
		}
		d.xloc = d.xnxt;
		d.yloc = d.ynxt;
	}

	private int cpDirection(int/*ACFixed*/ x1, int/*ACFixed*/ cy1, int/*ACFixed*/ x2, int/*ACFixed*/ y2, int/*ACFixed*/ x3, int/*ACFixed*/ y3)
	{
		int q, q1, q2, q3;
		q1 = (x2 >> 6) * ((y3 - cy1) >> 6);
		q2 = (x1 >> 6) * ((y2 - y3) >> 6);
		q3 = (x3 >> 6) * ((cy1 - y2) >> 6);
		q = q1 + q2 + q3;
		if (q > 0) {
			return 1;
		}
		if (q < 0) {
			return -1;
		}
		return 0;
	}

	private void rMovePoint(int/*ACFixed*/ dx, int/*ACFixed*/ dy, int whichcp, PathElt e)
	{
		if (whichcp == CP_START) {
			e = e.prev;
			whichcp = CP_END;
		}
		if (whichcp == CP_END) {
			if (e.type == CLOSEPATH) {
				e = getDest(e);
			}
			if (e.type == CURVETO) {
				e.x3 += dx;
				e.y3 += dy;
			} else {
				e.x += dx;
				e.y += dy;
			}
			return;
		}
		if (whichcp == CP_CURVE1) {
			e.x1 += dx;
			e.y1 += dy;
			return;
		}
		if (whichcp == CP_CURVE2) {
			e.x2 += dx;
			e.y2 += dy;
			return;
		}
	}

	private void delete(PathElt e)
	{
		PathElt nxt, prv;
		nxt = e.next;
		prv = e.prev;
		if (nxt != null) {
			nxt.prev = prv;
		} else {
			mPathEnd = prv;
		}
		if (prv != null) {
			prv.next = nxt;
		} else {
			mPathStart = nxt;
		}
	}

	private void checkSCurve(CheckData d, PathElt ee)
	{
		FltnRec fr = new FltnRec();
		Cd c0 = new Cd();
		Cd c1 = new Cd();
		Cd c2 = new Cd();
		Cd c3 = new Cd();
		getEndPoint(ee.prev, c0);
		fr.report = FRP_CHKDT;
		fr.param = d;
		c1.x = ee.x1;
		c1.y = ee.y1;
		c2.x = ee.x2;
		c2.y = ee.y2;
		c3.x = ee.x3;
		c3.y = ee.y3;
		d.xstate = d.ystate = STARTING;
		d.xdone = d.ydone = d.xflat = d.yflat = false;
		d.x0 = c0.x;
		d.cy0 = c0.y;
		d.x1 = c3.x;
		d.cy1 = c3.y;
		d.xloc = d.x0;
		d.yloc = d.cy0;
		d.e = ee;
		d.forMultiMaster = false;
		fltnCurve(c0, c1, c2, c3, fr);
	}

	private void checkZeroLength(CheckData d)
	{
		PathElt e, NxtE;
		Cd cd0 = new Cd();
		Cd cd1 = new Cd();
		int/*ACFixed*/ x2, y2, x3, y3;
		e = mPathStart;
		while (e != null) {
			/* delete zero length elements */
			NxtE = e.next;
			getEndPoints(e, cd0, cd1);
			if (e.type == LINETO && cd0.x == cd1.x && cd0.y == cd1.y) {
				delete(e);
			} else {
				if (e.type == CURVETO) {
					x2 = e.x1;
					y2 = e.y1;
					x3 = e.x2;
					y3 = e.y2;
					if (cd0.x == cd1.x && cd0.y == cd1.y && x2 == cd1.x && x3 == cd1.x && y2 == cd1.y && y3 == cd1.y) {
						delete(e);
					}
				}
			}
			e = NxtE;
		}
	}

	private void checkSmooth()
	{
		PathElt e, nxt, NxtE;
		int/*ACFixed*/ x3, y3;
		Cd cd0 = new Cd();
		Cd cd1 = new Cd();
		Cd cd2 = new Cd();
		Cd tmp = new Cd();
		CheckData data = new CheckData();
		initCheckData(data);
		checkZeroLength(data);
		do {
			data.reCheckSmooth = false;
			for (e = mPathStart; e != null; e = NxtE) {
				NxtE = e.next;
				if (e.type == MOVETO || isTiny(e) || e.isFlex) {
					continue;
				}
				getEndPoint(e, cd1);
				if (e.type == CURVETO) {
					int cpd0, cpd1;
					cd2.x = e.x1;
					cd2.y = e.y1;
					x3 = e.x2;
					y3 = e.y2;
					getEndPoint(e.prev, cd0);
					cpd0 = cpDirection(cd0.x, cd0.y, cd2.x, cd2.y, x3, y3);
					cpd1 = cpDirection(cd2.x, cd2.y, x3, y3, cd1.x, cd1.y);
					if (prodLt0(cpd0, cpd1)) {
						checkSCurve(data, e);
					}
				}
				nxt = nxtForBend(e, cd2, tmp);
				if (nxt.isFlex) {
					continue;
				}
				prvForBend(nxt, cd0);
			}
		} while (data.reCheckSmooth);
	}

	private void chkBBDT(Object param, Cd c)
	{
		CheckData d = (CheckData)param;
		int/*ACFixed*/ x = c.x, y = c.y;
		if (d.bbquit) {
			return;
		}
		if (d.vert) {
			d.lst = y;
			if (!d.started && Math.abs(x - d.loc) <= fixInt(10)) {
				d.started = true;
				d.frst = y;
			} else {
				if (d.started && Math.abs(x - d.loc) > fixInt(10)) {
					d.bbquit = true;
				}
			}
		} else {
			d.lst = x;
			if (!d.started && Math.abs(y - d.loc) <= fixInt(10)) {
				d.started = true;
				d.frst = x;
			} else {
				if (d.started && Math.abs(y - d.loc) > fixInt(10)) {
					d.bbquit = true;
				}
			}
		}
	}

	private void checkForMultiMoveTo()
	{
		PathElt e = mPathStart;
		boolean moveto;
		moveto = false;
		while (e != null) {
			if (e.type != MOVETO) {
				moveto = false;
			} else if (!moveto) {
				moveto = true;
			} else {
				delete(e.prev);	/* delete previous moveto */
			}
			e = e.next;
		}
	}

	private void checkBBoxEdge(PathElt e, boolean vrt, int/*ACFixed*/ lc, int/*ACFixed*/[] pf, int/*ACFixed*/[] pl)
	{
		CheckData data = new CheckData();
		FltnRec fr = new FltnRec();
		Cd c0 = new Cd();
		Cd c1 = new Cd();
		Cd c2 = new Cd();
		Cd c3 = new Cd();
		getEndPoint(e.prev, c0);
		fr.report = FRP_CHKBBDT;
		fr.param = data;
		data.bbquit = false;
		c1.x = e.x1;
		c1.y = e.y1;
		c2.x = e.x2;
		c2.y = e.y2;
		c3.x = e.x3;
		c3.y = e.y3;
		data.loc = lc;
		data.vert = vrt;
		data.started = false;
		chkBBDT(data, c0);
		fltnCurve(c0, c1, c2, c3, fr);
		pf[0] = data.frst;
		pl[0] = data.lst;
	}

	private void moveSubpathToEnd(PathElt e)
	{
		PathElt subEnd, subStart, subNext, subPrev;
		subEnd = (e.type == CLOSEPATH) ? e : getClosedBy(e);
		subStart = getDest(subEnd);
		if (subEnd == mPathEnd) {
			return;		/* already at end */
		}
		subNext = subEnd.next;
		if (subStart == mPathStart) {
			mPathStart = subNext;
			subNext.prev = null;
		} else {
			subPrev = subStart.prev;
			subPrev.next = subNext;
			subNext.prev = subPrev;
		}
		mPathEnd.next = subStart;
		subStart.prev = mPathEnd;
		subEnd.next = null;
		mPathEnd = subEnd;
	}

	/* Module ACpick.c */

	private void initPick()
	{
	}

	private boolean ltPruneB(int/*ACFixed*/ val)
	{
		return (val < FIX_ONE && (val << 10) < mPruneB);
	}

	private boolean considerPicking(int/*ACFixed*/ bestSpc, int/*ACFixed*/ bestVal, ClrVal colorList, int/*ACFixed*/ prevBestVal)
	{
		if (bestSpc > 0) {
			return true;
		}
		if (colorList == null) {
			return bestVal >= mPruneD;
		}
		if (bestVal > mPruneA) {
			return true;
		}
		if (ltPruneB(bestVal)) {
			return false;
		}
		return (bestVal < FIXED_POS_INF / mPruneC) ?
			(prevBestVal <= bestVal * mPruneC) : (prevBestVal / mPruneC <= bestVal);
	}

	private void pickVVals(ClrVal valList)
	{
		ClrVal colorList, rejectList, vlist, prev, best, bestPrev, nxt;
		int/*ACFixed*/ bestVal = 0, prevBestVal;
		int/*ACFixed*/ lft, rght, vlft, vrght;
		colorList = rejectList = null;
		prevBestVal = 0;
		while (true) {
			vlist = valList;
			prev = bestPrev = best = null;
			while (vlist != null) {
				if ((best == null || compareValues(vlist, best, SPCBONUS, 0)) &&
				    considerPicking(vlist.vSpc, vlist.vVal, colorList, prevBestVal)) {
					best = vlist;
					bestPrev = prev;
					bestVal = vlist.vVal;
				}
				prev = vlist;
				vlist = vlist.vNxt;
			}
			if (best == null) {
				break;					/* no more */
			}
			if (bestPrev == null) {
				valList = best.vNxt;
			} else {
				bestPrev.vNxt = best.vNxt;
			}
			/* have now removed best from valList */
			best.vNxt = colorList;				/* add best to front of list */
			colorList = best;
			prevBestVal = bestVal;
			lft = best.vLoc1 - mBandMargin;
			rght = best.vLoc2 + mBandMargin;
			/* remove segments from valList that overlap lft..rght */
			vlist = valList;
			prev = null;
			while (vlist != null) {
				vlft = vlist.vLoc1;
				vrght = vlist.vLoc2;
				if ((vlft <= rght) && (vrght >= lft)) {
					nxt = vlist.vNxt;
					vlist.vNxt = rejectList;
					rejectList = vlist;
					vlist = nxt;
					if (prev == null) {
						valList = vlist;
					} else {
						prev.vNxt = vlist;
					}
				} else {
					prev = vlist;
					vlist = vlist.vNxt;
				}
			}
		}
		vlist = valList;					/* move rest of valList to rejectList */
		while (vlist != null) {
			nxt = vlist.vNxt;
			vlist.vNxt = rejectList;
			rejectList = vlist;
			vlist = nxt;
		}
		if (colorList == null) {
			clrVBnds(mBBox);
		}
		mVcoloring = colorList;
	}

	private boolean inSerifBand(int/*ACFixed*/ y0, int/*ACFixed*/ y1, int n, int/*ACFixed*/[] p)
	{
		int i;
		if (n <= 0) {
			return false;
		}
		y0 = itfmy(y0);
		y1 = itfmy(y1);
		if (y0 > y1) {
			int/*ACFixed*/ tmp = y1;
			y1 = y0;
			y0 = tmp;
		}
		for (i = 0; i < n; i += 2) {
			if (p[i] <= y0 && p[i + 1] >= y1) {
				return true;
			}
		}
		return false;
	}

	private boolean considerValForSeg(ClrVal val, ClrSeg seg, int/*ACFixed*/ loc, int nb, int/*ACFixed*/[] b, int ns, int/*ACFixed*/[] s, boolean primary)
	{
		if (primary && val.vSpc > 0.0) {
			return true;
		}
		if (inBlueBand(loc, nb, b)) {
			return true;
		}
		if (val.vSpc <= 0.0 && inSerifBand(seg.sMax, seg.sMin, ns, s)) {
			return false;
		}
		if (ltPruneB(val.vVal)) {
			return false;
		}
		return true;
	}

	private ClrVal fndBstVal(ClrSeg seg, boolean seg1Flg, ClrVal cList, ClrVal rList,
				 int nb, int/*ACFixed*/[] b, int ns, int/*ACFixed*/[] s, boolean locFlg, boolean hFlg)
	{
		int/*ACFixed*/ loc, vloc;
		ClrVal best, vList, initLst;
		ClrSeg vseg;
		best = null;
		loc = seg.sLoc;
		vList = cList;
		while (true) {
			initLst = vList;
			while (vList != null) {
				if (seg1Flg) {
					vseg = vList.vSeg1;
					vloc = vList.vLoc1;
				} else {
					vseg = vList.vSeg2;
					vloc = vList.vLoc2;
				}
				if (Math.abs(loc - vloc) <= mMaxMerge &&
				    (locFlg ? !vList.vGhst :
				     (vseg == seg || closeSegs(seg, vseg, !hFlg))) &&
				    (best == null ||
				     (vList.vVal == best.vVal && vList.vSpc == best.vSpc &&
				      vList.initVal > best.initVal) ||
				     compareValues(vList, best, SPCBONUS, 3)) &&
				    /* last arg is "ghostshift" that penalizes ghost values */
				    /* ghost values are set to 20 */
				    /* so ghostshift of 3 means prefer nonghost if its
				       value is > (20 >> 3) */
				    considerValForSeg(vList, seg, loc, nb, b, ns, s, true)) {
					best = vList;
				}
				vList = vList.vNxt;
			}
			if (initLst == rList) {
				break;
			}
			vList = rList;
		}
		return best;
	}

	private ClrVal findBestValForSeg(ClrSeg seg, boolean seg1Flg, ClrVal cList, ClrVal rList,
					 int nb, int/*ACFixed*/[] b, int ns, int/*ACFixed*/[] s, boolean hFlg)
	{
		ClrVal best, nonghst, ghst = null;
		best = fndBstVal(seg, seg1Flg, cList, rList, nb, b, ns, s, false, hFlg);
		if (best != null && best.vGhst) {
			nonghst = fndBstVal(seg, seg1Flg, cList, rList, nb, b, ns, s, true, hFlg);
			/* If nonghst hints are "better" use it instead of ghost band. */
			if (nonghst != null && nonghst.vVal >= fixInt(2)) {
				/* threshold must be greater than 1.004 for ITC Garamond Ultra "q" */
				ghst = best;
				best = nonghst;
			}
		}
		if (best != null) {
			/* threshold must be > .035 for Monotype/Plantin/Bold Thorn and < .08 for Bookman2/Italic asterisk */
			if (best.vVal < FIX_SIXTEENTH && (ghst == null || ghst.vVal < FIX_SIXTEENTH)) {
				best = null;
			} else {
				best.pruned = false;
			}
		}
		return best;
	}

	private boolean membValList(ClrVal val, ClrVal vList)
	{
		while (vList != null) {
			if (val == vList) {
				return true;
			}
			vList = vList.vNxt;
		}
		return false;
	}

	private ClrVal prevVal(ClrVal val, ClrVal vList)
	{
		ClrVal prev;
		if (val == vList) {
			return null;
		}
		prev = vList;
		while (true) {
			vList = vList.vNxt;
			if (vList == val) {
				return prev;
			}
			prev = vList;
		}
	}

	private void pickHVals(ClrVal valList)
	{
		ClrVal vlist, colorList, rejectList, bestPrev, prev, best, nxt;
		int/*ACFixed*/ bestVal = 0, prevBestVal;
		int/*ACFixed*/ bot, top, vtop, vbot;
		ClrVal newBst;
		ClrSeg seg1, seg2;
		colorList = rejectList = null;
		prevBestVal = 0;
		while (true) {
			vlist = valList;
			prev = bestPrev = best = null;
			while (vlist != null) {
				if ((best == null || compareValues(vlist, best, SPCBONUS, 0)) &&
				    considerPicking(vlist.vSpc, vlist.vVal, colorList, prevBestVal)) {
					best = vlist;
					bestPrev = prev;
					bestVal = vlist.vVal;
				}
				prev = vlist;
				vlist = vlist.vNxt;
			}
			if (best != null) {
				seg1 = best.vSeg1;
				seg2 = best.vSeg2;
				if (best.vGhst) {
					/* find real segments at same loc as best */
					vlist = valList;
					while (vlist != null) {
						if (vlist.vLoc2 == best.vLoc2 && vlist.vLoc1 == best.vLoc1 && !vlist.vGhst) {
							seg1 = vlist.vSeg1;
							seg2 = vlist.vSeg2;
							break;
						}
						vlist = vlist.vNxt;
					}
				}
				if (seg1.sType == S_GHOST) {
					newBst = seg2.sLnk;
					if (newBst != null && newBst != best && membValList(newBst, valList)) {
						best = newBst;
						bestPrev = prevVal(best, valList);
					}
				} else {
					if (seg2.sType == S_GHOST) {
						newBst = seg2.sLnk;
						if (newBst != null && newBst != best && membValList(newBst, valList)) {
							best = newBst;
							bestPrev = prevVal(best, valList);
						}
					}
				}
			}
			if (best == null) {
				break;
			}
			prevBestVal = bestVal;
			if (bestPrev == null) {
				valList = best.vNxt;
			} else {
				bestPrev.vNxt = best.vNxt;
			}
			/* have now removed best from valList */
			best.vNxt = colorList;
			colorList = best;		/* add best to front of list */
			bot = best.vLoc1;
			top = best.vLoc2;
			/* The next if statement was added so that ghost bands are given
			   0 width for doing the conflict tests for bands too close together.
			   This was a problem in Minion/DisplayItalic onequarter and onehalf. */
			if (best.vGhst) {
				/* collapse width */
				if (best.vSeg1.sType == S_GHOST) {
					bot = top;
				} else {
					top = bot;
				}
			}
			if (Y_GOES_UP) {
				bot -= mBandMargin;
				top += mBandMargin;
			} else {
				bot += mBandMargin;
				top -= mBandMargin;
			}
			/* remove segments from valList that overlap bot..top */
			vlist = valList;
			prev = null;
			while (vlist != null) {
				vbot = vlist.vLoc1;
				vtop = vlist.vLoc2;
				/* The next if statement was added so that ghost bands are given
				   0 width for doing the conflict tests for bands too close together. */
				if (vlist.vGhst) {
					/* collapse width */
					if (vlist.vSeg1.sType == S_GHOST) {
						vbot = vtop;
					} else {
						vtop = vbot;
					}
				}
				if ((Y_GOES_UP && (vbot <= top) && (vtop >= bot)) ||
				    ((!Y_GOES_UP && (vbot >= top) && (vtop <= bot)))) {
					nxt = vlist.vNxt;
					vlist.vNxt = rejectList;
					rejectList = vlist;
					vlist = nxt;
					if (prev == null) {
						valList = vlist;
					} else {
						prev.vNxt = vlist;
					}
				} else {
					prev = vlist;
					vlist = vlist.vNxt;
				}
			}
		}
		vlist = valList;			/* move rest of valList to rejectList */
		while (vlist != null) {
			nxt = vlist.vNxt;
			vlist.vNxt = rejectList;
			rejectList = vlist;
			vlist = nxt;
		}
		if (colorList == null) {
			clrHBnds(mBBox);
		}
		mHcoloring = colorList;
	}

	private void findBestValForSegs(ClrSeg sList, boolean seg1Flg, ClrVal cList, ClrVal rList,
					int nb, int/*ACFixed*/[] b, int ns, int/*ACFixed*/[] s, boolean hFlg)
	{
		ClrVal best;
		while (sList != null) {
			best = findBestValForSeg(sList, seg1Flg, cList, rList, nb, b, ns, s, hFlg);
			sList.sLnk = best;
			sList = sList.sNxt;
		}
	}

	private void setPruned()
	{
		ClrVal vL = mValList;
		while (vL != null) {
			vL.pruned = true;
			vL = vL.vNxt;
		}
	}

	private void findBestHVals()
	{
		setPruned();
		findBestValForSegs(topList(), false, mValList, null,
				   mLenTopBands, mTopBands, 0, null, true);
		findBestValForSegs(botList(), true, mValList, null,
				   mLenBotBands, mBotBands, 0, null, true);
		doPrune();
	}

	private void findBestVVals()
	{
		setPruned();
		findBestValForSegs(leftList(), true, mValList, null,
				   0, null, mNumSerifs, mSerifs, false);
		findBestValForSegs(rightList(), false, mValList, null,
				   0, null, mNumSerifs, mSerifs, false);
		doPrune();
	}

	/* Module ACbbox.c */

	private void fpBBoxPt(Object param, Cd c)
	{
		ACBBox bb = (ACBBox)param;
		if (c.x < bb.xmin) {
			bb.xmin = c.x;
			bb.pxmn = bb.pe;
		}
		if (c.x > bb.xmax) {
			bb.xmax = c.x;
			bb.pxmx = bb.pe;
		}
		if (c.y < bb.ymin) {
			bb.ymin = c.y;
			bb.pymn = bb.pe;
		}
		if (c.y > bb.ymax) {
			bb.ymax = c.y;
			bb.pymx = bb.pe;
		}
	}

	/* returns true if the give curve does not have to be flattened for calculating the bbox */
	private boolean fpBBoxCurveShortCut(Cd c0, Cd c1, Cd c2, Cd c3, Object param)
	{
		ACBBox bb = (ACBBox)param;
		/* incorporate the end points in the existing bbox first */
		fpBBoxPt(param, c0);
		fpBBoxPt(param, c3);
		/* if both control points are within the existing bbox, then no need to flatten the curve */
		return (inRange(c1.x, bb.xmin, bb.xmax) && inRange(c1.y, bb.ymin, bb.ymax) &&
			inRange(c2.x, bb.xmin, bb.xmax) && inRange(c2.y, bb.ymin, bb.ymax));
	}

	private void findPathBBox(ACBBox bb)
	{
		if (mPathBBox != null) {
			bb.copyFrom(mPathBBox);
			return;
		}
		{
			FltnRec fr = new FltnRec();
			PathElt e;
			Cd c0 = new Cd();
			Cd c1 = new Cd();
			Cd c2 = new Cd();
			Cd c3 = new Cd();
			if (mPathStart == null) {
				bb.xmin = bb.ymin = bb.xmax = bb.ymax = 0;
				bb.pxmn = bb.pxmx = bb.pymn = bb.pymx = null;
				return;
			}
			fr.report = FRP_FPBBOXPT;
			fr.param = bb;
			bb.xmin = bb.ymin = fixInt(10000);
			bb.xmax = bb.ymax = -bb.xmin;
			e = mPathStart;
			while (e != null) {
				switch (e.type) {
				case MOVETO:
				case LINETO:
					c0.x = e.x;
					c0.y = e.y;
					bb.pe = e;
					fpBBoxPt(bb, c0);
					break;
				case CURVETO:
					c1.x = e.x1;
					c1.y = e.y1;
					c2.x = e.x2;
					c2.y = e.y2;
					c3.x = e.x3;
					c3.y = e.y3;
					bb.pe = e;
					if (!fpBBoxCurveShortCut(c0, c1, c2, c3, bb)) {
						fltnCurve(c0, c1, c2, c3, fr);
					}
					c0.copyFrom(c3);
					break;
				case CLOSEPATH:
					break;
				}
				e = e.next;
			}
			bb.xmin = fHalfRnd(bb.xmin);
			bb.ymin = fHalfRnd(bb.ymin);
			bb.xmax = fHalfRnd(bb.xmax);
			bb.ymax = fHalfRnd(bb.ymax);
			mPathBBox = new ACBBox();
			mPathBBox.copyFrom(bb);
		}
	}

	private PathElt calcSubpathBBox(ACBBox bb, PathElt e)
	{
		FltnRec fr = new FltnRec();
		Cd c0 = new Cd();
		Cd c1 = new Cd();
		Cd c2 = new Cd();
		Cd c3 = new Cd();
		if (e == null) {
			bb.xmin = bb.ymin = bb.xmax = bb.ymax = 0;
			bb.pxmn = bb.pxmx = bb.pymn = bb.pymx = null;
			return null;
		}
		fr.report = FRP_FPBBOXPT;
		fr.param = bb;
		bb.xmin = bb.ymin = fixInt(10000);
		bb.xmax = bb.ymax = -bb.xmin;
		if (e.type != MOVETO) {
			e = getDest(e);		/* back up to moveto */
		}
		while (e != null) {
			switch (e.type) {
			case MOVETO:
			case LINETO:
				c0.x = e.x;
				c0.y = e.y;
				bb.pe = e;
				fpBBoxPt(bb, c0);
				break;
			case CURVETO:
				c1.x = e.x1;
				c1.y = e.y1;
				c2.x = e.x2;
				c2.y = e.y2;
				c3.x = e.x3;
				c3.y = e.y3;
				bb.pe = e;
				if (!fpBBoxCurveShortCut(c0, c1, c2, c3, bb)) {
					fltnCurve(c0, c1, c2, c3, fr);
				}
				c0.copyFrom(c3);
				break;
			case CLOSEPATH:
				e = e.next;
				bb.xmin = fHalfRnd(bb.xmin);
				bb.ymin = fHalfRnd(bb.ymin);
				bb.xmax = fHalfRnd(bb.xmax);
				bb.ymax = fHalfRnd(bb.ymax);
				return e;
			}
			e = e.next;
		}
		bb.xmin = fHalfRnd(bb.xmin);
		bb.ymin = fHalfRnd(bb.ymin);
		bb.xmax = fHalfRnd(bb.xmax);
		bb.ymax = fHalfRnd(bb.ymax);
		return e;
	}

	private PathElt findSubpathBBox(ACBBox bb, PathElt e)
	{
		if (mSubPaths != null) {
			int i;
			for (i = 0; i < mNumSubPaths; i++) {
				if (mSubPaths[i].head == e) {
					bb.copyFrom(mSubPaths[i].bbox);
					e = mSubPaths[i].tail;
					if (e != null) {
						e = e.next;
					}
					return e;
				}
			}
		}
		return calcSubpathBBox(bb, e);
	}

	private void findCurveBBox(int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ px1, int/*ACFixed*/ py1,
				   int/*ACFixed*/ px2, int/*ACFixed*/ py2, int/*ACFixed*/ x1, int/*ACFixed*/ y1, Cd ll, Cd ur)
	{
		FltnRec fr = new FltnRec();
		Cd c0 = new Cd();
		Cd c1 = new Cd();
		Cd c2 = new Cd();
		Cd c3 = new Cd();
		ACBBox bbox = new ACBBox();
		fr.report = FRP_FPBBOXPT;
		fr.param = bbox;
		bbox.xmin = bbox.ymin = fixInt(10000);
		bbox.xmax = bbox.ymax = -bbox.xmin;
		c0.x = x0;
		c0.y = y0;
		c1.x = px1;
		c1.y = py1;
		c2.x = px2;
		c2.y = py2;
		c3.x = x1;
		c3.y = y1;
		fpBBoxPt(bbox, c0);
		fltnCurve(c0, c1, c2, c3, fr);
		ll.x = fHalfRnd(bbox.xmin);
		ll.y = fHalfRnd(bbox.ymin);
		ur.x = fHalfRnd(bbox.xmax);
		ur.y = fHalfRnd(bbox.ymax);
	}

	private void clrVBnds(ACBBox bb)
	{
		int/*ACFixed*/ tmp;
		PathElt p;
		if (mPathStart == null || (mDoCounters && vColorChar(mUnicode))) {
			return;
		}
		findPathBBox(bb);
		bb.vMn = itfmx(bb.xmin);
		bb.vMx = itfmx(bb.xmax);
		bb.pvMn = bb.pxmn;
		bb.pvMx = bb.pxmx;
		if (bb.vMn > bb.vMx) {
			tmp = bb.vMn;
			bb.vMn = bb.vMx;
			bb.vMx = tmp;
			p = bb.pvMn;
			bb.pvMn = bb.pvMx;
			bb.pvMx = p;
		}
		addColorPoint(bb.vMn, 0, bb.vMx, 0, 'y', bb.pvMn, bb.pvMx);
	}

	private void reClrVBnds(ACBBox bb)
	{
		addColorPoint(bb.vMn, 0, bb.vMx, 0, 'y', bb.pvMn, bb.pvMx);
	}

	private void clrHBnds(ACBBox bb)
	{
		int/*ACFixed*/ tmp;
		PathElt p;
		if (mPathStart == null || (mDoCounters && hColorChar(mUnicode))) {
			return;
		}
		findPathBBox(bb);
		bb.hMn = itfmy(bb.ymin);
		bb.hMx = itfmy(bb.ymax);
		bb.phMn = bb.pymn;
		bb.phMx = bb.pymx;
		if (bb.hMn > bb.hMx) {
			tmp = bb.hMn;
			bb.hMn = bb.hMx;
			bb.hMx = tmp;
			p = bb.phMn;
			bb.phMn = bb.phMx;
			bb.phMx = p;
		}
		addColorPoint(0, bb.hMn, 0, bb.hMx, 'b', bb.phMn, bb.phMx);
	}

	private void reClrHBnds(ACBBox bb)
	{
		addColorPoint(0, bb.hMn, 0, bb.hMx, 'b', bb.phMn, bb.phMx);
	}

	private boolean checkValOverlaps(int/*ACFixed*/ lft, int/*ACFixed*/ rht, ClrVal lst, boolean xflg)
	{
		int/*ACFixed*/ lft2, rht2, tmp;
		if (xflg) {
			lft = itfmx(lft);
			rht = itfmx(rht);
		} else {
			lft = itfmy(lft);
			rht = itfmy(rht);
		}
		if (lft > rht) {
			tmp = lft;
			lft = rht;
			rht = tmp;
		}
		while (lst != null) {
			lft2 = lst.vLoc1;
			rht2 = lst.vLoc2;
			if (xflg) {
				lft2 = itfmx(lft2);
				rht2 = itfmx(rht2);
			} else {
				lft2 = itfmy(lft2);
				rht2 = itfmy(rht2);
			}
			if (lft2 > rht2) {
				tmp = lft2;
				lft2 = rht2;
				rht2 = tmp;
			}
			if (lft2 <= rht && lft <= rht2) {
				return true;
			}
			lst = lst.vNxt;
		}
		return false;
	}

	private void addBBoxHV(boolean Hflg, boolean subs)
	{
		PathElt e;
		ClrVal val;
		ClrSeg seg1, seg2;
		e = mPathStart;
		while (e != null) {
			ACBBox bbox = new ACBBox();
			if (subs) {
				e = findSubpathBBox(bbox, e);
			} else {
				findPathBBox(bbox);
				e = null;
			}
			if (!Hflg) {
				if (!checkValOverlaps(bbox.xmin, bbox.xmax, mVcoloring, true)) {
					val = new ClrVal();
					val.cvlSN = ++mCvlSN;
					seg1 = new ClrSeg();
					seg1.csgSN = ++mCsgSN;
					seg1.sLoc = bbox.xmin;
					seg1.sElt = bbox.pxmn;
					seg1.sBonus = 0;
					seg1.sType = S_LINE;
					seg1.sMin = bbox.ymin;
					seg1.sMax = bbox.ymax;
					seg1.sNxt = null;
					seg1.sLnk = null;
					seg2 = new ClrSeg();
					seg2.csgSN = ++mCsgSN;
					seg2.sLoc = bbox.xmax;
					seg2.sElt = bbox.pxmx;
					seg2.sBonus = 0;
					seg2.sType = S_LINE;
					seg2.sMin = bbox.ymin;
					seg2.sMax = bbox.ymax;
					seg2.sNxt = null;
					seg2.sLnk = null;
					val.vVal = 100;
					val.vSpc = 0;
					val.vLoc1 = bbox.xmin;
					val.vLoc2 = bbox.xmax;
					val.vSeg1 = seg1;
					val.vSeg2 = seg2;
					val.vGhst = false;
					val.vNxt = mVcoloring;
					val.vBst = val;
					mVcoloring = val;
				}
			} else {
				if (!checkValOverlaps(bbox.ymin, bbox.ymax, mHcoloring, false)) {
					val = new ClrVal();
					val.cvlSN = ++mCvlSN;
					seg1 = new ClrSeg();
					seg1.csgSN = ++mCsgSN;
					seg1.sLoc = bbox.ymax;
					seg1.sElt = bbox.pymx;
					seg1.sBonus = 0;
					seg1.sType = S_LINE;
					seg1.sMin = bbox.xmin;
					seg1.sMax = bbox.xmax;
					seg1.sNxt = null;
					seg1.sLnk = null;
					seg2 = new ClrSeg();
					seg2.csgSN = ++mCsgSN;
					seg2.sLoc = bbox.ymin;
					seg2.sElt = bbox.pymn;
					seg2.sBonus = 0;
					seg2.sType = S_LINE;
					seg2.sMin = bbox.xmin;
					seg2.sMax = bbox.xmax;
					seg2.sNxt = null;
					seg2.sLnk = null;
					val.vVal = 100;
					val.vSpc = 0;
					val.vLoc1 = bbox.ymax;		/* bot is > top because y axis is reversed */
					val.vLoc2 = bbox.ymin;
					val.vSeg1 = seg1;
					val.vSeg2 = seg2;
					val.vGhst = false;
					val.vNxt = mHcoloring;
					val.vBst = val;
					mHcoloring = val;
				}
			}
		}
	}

	private boolean checkBBoxes(PathElt e1, PathElt e2)
	{
		/* return true if e1 and e2 in same subpath or i
		   the bbox for one is inside the bbox of the other */
		int/*ACFixed*/ xmn, xmx, ymn, ymx;
		ACBBox bbox = new ACBBox();
		e1 = getDest(e1);
		e2 = getDest(e2);
		if (e1 == e2) {
			return true;		/* same subpath */
		}
		findSubpathBBox(bbox, e1);
		xmn = bbox.xmin;
		xmx = bbox.xmax;
		ymn = bbox.ymin;
		ymx = bbox.ymax;
		findSubpathBBox(bbox, e2);
		return ((xmn <= bbox.xmin && bbox.xmax <= xmx && ymn <= bbox.ymin && bbox.ymax <= ymx) ||
			(xmn >= bbox.xmin && bbox.xmax >= xmx && ymn >= bbox.ymin && bbox.ymax >= ymx));
	}

	/* Module ACwrite.c */

	private void newBest(WriteData wd, ClrPoint lst)
	{
		int/*ACFixed*/ x0, x1, y0, y1;
		wd.bst = lst;
		wd.bch = lst.c;
		if (wd.bch == 'y' || wd.bch == 'm') {
			wd.bstB = true;
			x0 = lst.x0;
			x1 = lst.x1;
			wd.bx = Math.min(x0, x1);
		} else {
			wd.bstB = false;
			y0 = lst.y0;
			y1 = lst.y1;
			wd.by = Math.min(y0, y1);
		}
	}

	private void writePointItem(WriteData wd, ClrPoint lst)
	{
		int flags = 0;
		int/*ACFixed*/ v1, v2;
		int/*ACFixed*/ width;
		switch (lst.c) {
		case 'v':
			flags = AC_GC_STEM3_STEM;
			v1 = lst.y0;
			v2 = lst.y1;
			break;
		case 'b':
			v1 = lst.y0;
			v2 = lst.y1;
			break;
		case 'm':
			flags = AC_GC_STEM3_STEM + AC_GC_VERT_STEM;
			v1 = lst.x0;
			v2 = lst.x1;
			break;
		case 'y':
			flags = AC_GC_VERT_STEM;
			v1 = lst.x0;
			v2 = lst.x1;
			break;
		default:
			return;
		}
		/* don't unscale ghost stem widths */
		width = v2 - v1;
		if (width == fixInt(TOPGHST)) {
			v1 = unScaleAbs(v1);
			v2 = v1 + width;
		} else {
			if (width == fixInt(BOTGHST)) {
				v2 = unScaleAbs(v2);
				v1 = v2 - width;
			} else {
				v1 = unScaleAbs(v1);
				v2 = unScaleAbs(v2);
			}
		}
		if (wd.hintArray == null)
			wd.hintArray = new ArrayList();
		HintData hp = new HintData();
		hp.flags = flags;
		hp.v1 = v1;
		hp.v2 = v2;
		wd.hintArray.add(hp);
	}

	private void flushHintArray(ArrayList ha)
	{
		if (mOutlineConsumer != null) {
			for (int i = 0; i < ha.size(); i++) {
				double fv1, fv2;
				HintData hp = (HintData)ha.get(i);
				int flags = hp.flags;
				if (i == 0) {
					flags |= AC_GC_NEW_HINTS;
				}
				fv1 = fixToFlt(hp.v1);
				fv2 = fixToFlt(hp.v2);
				mOutlineConsumer.stem(fv1, fv2, (flags & AC_GC_NEW_HINTS) != 0, (flags & AC_GC_VERT_STEM) != 0, false);
			}
		}
	}

	private boolean compareHintArray(ArrayList ha1, ArrayList ha2)
	{
		int i;
		int size1 = (ha1 == null) ? 0 : ha1.size();
		int size2 = (ha2 == null) ? 0 : ha2.size();
		if (size1 != size2) {
			return false;
		}
		for (i = 0; i < size1; i++) {
			HintData h1 = (HintData)ha1.get(i);
			HintData h2 = (HintData)ha2.get(i);
			if (h1.flags != h2.flags || h1.v1 != h2.v1 || h1.v2 != h2.v2) {
				return false;
			}
		}
		return true;
	}

	private void wrtPntLst(WriteData wd, ClrPoint lst)
	{
		ClrPoint ptLst;
		char ch;
		int/*ACFixed*/ x0, x1, y0, y1;
		ptLst = lst;
		while (lst != null) {
			/* mark all as not yet done */
			lst.done = false;
			lst = lst.next;
		}
		while (true) {
			/* write in sort order */
			lst = ptLst;
			wd.bst = null;
			while (lst != null) {
				/* find first not yet done as init best */
				if (!lst.done) {
					newBest(wd, lst);
					break;
				}
				lst = lst.next;
			}
			if (wd.bst == null) {
				break;				/* finished with entire list */
			}
			lst = wd.bst.next;
			while (lst != null) {
				/* search for best */
				if (!lst.done) {
					ch = lst.c;
					if (ch > wd.bch) {
						newBest(wd, lst);
					} else if (ch == wd.bch) {
						if (wd.bstB) {
							x0 = lst.x0;
							x1 = lst.x1;
							if (Math.min(x0, x1) < wd.bx) {
								newBest(wd, lst);
							}
						} else {
							y0 = lst.y0;
							y1 = lst.y1;
							if (Math.min(y0, y1) < wd.by) {
								newBest(wd, lst);
							}
						}
					}
				}
				lst = lst.next;
			}
			wd.bst.done = true;			/* mark as having been done */
			writePointItem(wd, wd.bst);
		}
	}

	private void wrtNewClrs(WriteData wd, PathElt e)
	{
		if (!wd.wrtColorInfo) {
			return;
		}
		wd.hintArray = null;
		wrtPntLst(wd, mPtLstArray[e.newcolors]);
		if (!compareHintArray(wd.prevHintArray, wd.hintArray)) {
			flushHintArray(wd.hintArray);
			wd.prevHintArray = new ArrayList(wd.hintArray);
		}
	}

	private boolean isFlex(WriteData wd, PathElt e)
	{
		PathElt e0, e1;
		if (wd.firstFlex) {
			e0 = e;
			e1 = e.next;
		} else {
			e0 = e.prev;
			e1 = e;
		}
		return (e0 != null && e0.isFlex && e1 != null && e1.isFlex);
	}

	/* moveto */
	private void mt(WriteData wd, Cd c, PathElt e)
	{
		if (mOutlineConsumer != null) {
			double fcx, fcy;
			if (e.newcolors != 0) {
				wrtNewClrs(wd, e);
			}
			fcx = fixToFlt(c.x);
			fcy = fixToFlt(c.y);
			mOutlineConsumer.moveto(fcx, fcy);
		}
	}

	/* lineto */
	private void dt(WriteData wd, Cd c, PathElt e)
	{
		if (mOutlineConsumer != null) {
			double fcx, fcy;
			if (e.newcolors != 0) {
				wrtNewClrs(wd, e);
			}
			fcx = fixToFlt(c.x);
			fcy = fixToFlt(c.y);
			mOutlineConsumer.lineto(fcx, fcy);
		}
	}

	/* flex */
	private void wrtFlex(WriteData wd, Cd c1, Cd c2, Cd c3, PathElt e)
	{
		if (mOutlineConsumer != null) {
			double fdmin, fc1x, fc1y, fc2x, fc2y, fc3x, fc3y;
			double c1x, c1y, c2x, c2y, c3x, c3y;
			if (wd.firstFlex) {
				wd.fc1.copyFrom(c1);
				wd.fc2.copyFrom(c2);
				wd.fc3.copyFrom(c3);
				wd.firstFlex = false;
				return;
			}
			fdmin = fixToFlt(mDMIN);
			fc1x = fixToFlt(wd.fc1.x);
			fc1y = fixToFlt(wd.fc1.y);
			fc2x = fixToFlt(wd.fc2.x);
			fc2y = fixToFlt(wd.fc2.y);
			fc3x = fixToFlt(wd.fc3.x);
			fc3y = fixToFlt(wd.fc3.y);
			c1x = fixToFlt(c1.x);
			c1y = fixToFlt(c1.y);
			c2x = fixToFlt(c2.x);
			c2y = fixToFlt(c2.y);
			c3x = fixToFlt(c3.x);
			c3y = fixToFlt(c3.y);
			mOutlineConsumer.flex(fdmin, fc1x, fc1y, fc2x, fc2y, fc3x, fc3y, c1x, c1y, c2x, c2y, c3x, c3y);
			wd.firstFlex = true;
		}
	}

	/* curveto */
	private void ct(WriteData wd, Cd c1, Cd c2, Cd c3, PathElt e)
	{
		if (mOutlineConsumer != null) {
			double fc1x, fc1y, fc2x, fc2y, fc3x, fc3y;
			if (e.newcolors != 0) {
				wrtNewClrs(wd, e);
			}
			if (e.isFlex && isFlex(wd, e)) {
				wrtFlex(wd, c1, c2, c3, e);
			}
			fc1x = fixToFlt(c1.x);
			fc1y = fixToFlt(c1.y);
			fc2x = fixToFlt(c2.x);
			fc2y = fixToFlt(c2.y);
			fc3x = fixToFlt(c3.x);
			fc3y = fixToFlt(c3.y);
			mOutlineConsumer.curveto(fc1x, fc1y, fc2x, fc2y, fc3x, fc3y);
		}
	}

	/* closepath */
	private void cp(WriteData wd, PathElt e)
	{
		if (e.newcolors != 0) {
			wrtNewClrs(wd, e);
		}
		/* the next move or end automatically closes the last sub-path automatically */
	}

	/* begin glyph */
	private void beginGlyph()
	{
	}

	/* end glyph */
	private void endGlyph()
	{
		if (mOutlineConsumer != null) {
			mOutlineConsumer.endchar();
		}
	}

	/* write glyph width */
	private boolean writeWidth()
	{
		if (mOutlineConsumer != null) {
			return mOutlineConsumer.width(mGlyphWidth);
		}
		return true;
	}

	private void numberPath()
	{
		short cnt;
		PathElt e;
		e = mPathStart;
		cnt = 1;
		while (e != null) {
			e.count = cnt++;
			e = e.next;
		}
	}

	private void writeGlyph()
	{
		if (mDoWriteGlyph) {
			PathElt e = mPathStart;
			Cd c1 = new Cd();
			Cd c2 = new Cd();
			Cd c3 = new Cd();
			WriteData writeData = new WriteData();
			writeData.wrtColorInfo = (mPathStart != null && mPathStart != mPathEnd);
			numberPath();
			beginGlyph();
			if (!writeWidth()) {
				return;
			}
			if (writeData.wrtColorInfo && e.newcolors == 0) {
				wrtPntLst(writeData, mPtLstArray[0]);
				flushHintArray(writeData.hintArray);
				writeData.prevHintArray = new ArrayList(writeData.hintArray);
			}
			writeData.firstFlex = true;
			while (e != null) {
				switch (e.type) {
				case CURVETO:
					c1.x = unScaleAbs(itfmx(e.x1));
					c1.y = unScaleAbs(itfmy(e.y1));
					c2.x = unScaleAbs(itfmx(e.x2));
					c2.y = unScaleAbs(itfmy(e.y2));
					c3.x = unScaleAbs(itfmx(e.x3));
					c3.y = unScaleAbs(itfmy(e.y3));
					ct(writeData, c1, c2, c3, e);
					break;
				case LINETO:
					c1.x = unScaleAbs(itfmx(e.x));
					c1.y = unScaleAbs(itfmy(e.y));
					dt(writeData, c1, e);
					break;
				case MOVETO:
					c1.x = unScaleAbs(itfmx(e.x));
					c1.y = unScaleAbs(itfmy(e.y));
					mt(writeData, c1, e);
					break;
				case CLOSEPATH:
					cp(writeData, e);
					break;
				}
				e = e.next;
			}
			endGlyph();
			writeData.hintArray = null;
			writeData.prevHintArray = null;
		}
	}

	/* ACfix.c */
	private void initFix()
	{
		mHFixCount = mVFixCount = 0;
	}

	private void recordHFix(int/*ACFixed*/ y, int/*ACFixed*/ dy)
	{
		mHFixYs[mHFixCount] = y;
		mHFixDYs[mHFixCount] = dy;
		mHFixCount++;
	}

	private void recordVFix(int/*ACFixed*/ x, int/*ACFixed*/ dx)
	{
		mVFixXs[mVFixCount] = x;
		mVFixDXs[mVFixCount] = dx;
		mVFixCount++;
	}

	private void recordForFix(boolean vert, int/*ACFixed*/ w, int/*ACFixed*/ minW, int/*ACFixed*/ b, int/*ACFixed*/ t)
	{
		int/*ACFixed*/ mn, mx, delta;
		if (b < t) {
			mn = b;
			mx = t;
		} else {
			mn = t;
			mx = b;
		}
		if (!vert && mHFixCount + 4 < MAXFIXES && mAutoHFix) {
			int/*ACFixed*/ fixdy = w - minW;
			if (Math.abs(fixdy) <= FIX_ONE) {
				recordHFix(mn, fixdy);
				recordHFix(mx - fixdy, fixdy);
			} else {
				delta = fixHalfMul(fixdy);
				recordHFix(mn, delta);
				recordHFix(mn + fixdy, -delta);
				recordHFix(mx, -delta);
				recordHFix(mx - fixdy, delta);
			}
		} else {
			if (vert && mVFixCount + 4 < MAXFIXES && mAutoVFix) {
				int/*ACFixed*/ fixdx = w - minW;
				if (Math.abs(fixdx) <= FIX_ONE) {
					recordVFix(mn, fixdx);
					recordVFix(mx - fixdx, fixdx);
				} else {
					delta = fixHalfMul(fixdx);
					recordVFix(mn, delta);
					recordVFix(mn + fixdx, -delta);
					recordVFix(mx, -delta);
					recordVFix(mx - fixdx, delta);
				}
			}
		}
	}

	private boolean findLineSeg(int/*ACFixed*/ loc, ClrSeg sL)
	{
		while (sL != null) {
			if (sL.sLoc == loc && sL.sType == S_LINE) {
				return true;
			}
			sL = sL.sNxt;
		}
		return false;
	}

	private void checkVal(ClrVal val, boolean vert)
	{
		int/*ACFixed*/[] stems;
		int numstems, i;
		int/*ACFixed*/ wd, diff, minDiff, minW, b, t, w;
		if (vert) {
			stems = mVStems;
			numstems = mNumVStems;
			b = itfmx(val.vLoc1);
			t = itfmx(val.vLoc2);
		} else {
			stems = mHStems;
			numstems = mNumHStems;
			b = itfmy(val.vLoc1);
			t = itfmy(val.vLoc2);
		}
		w = Math.abs(t - b);
		minDiff = fixInt(1000);
		minW = 0;
		for (i = 0; i < numstems; i++) {
			wd = stems[i];
			diff = Math.abs(wd - w);
			if (diff < minDiff) {
				minDiff = diff;
				minW = wd;
				if (minDiff == 0) {
					break;
				}
			}
		}
		if (minDiff == 0 || minDiff > fixInt(2)) {
			return;
		}
		if ((vert && mAutoVFix) || (!vert && mAutoHFix)) {
			recordForFix(vert, w, minW, b, t);
		}
	}

	private void checkVals(ClrVal vlst, boolean vert)
	{
		while (vlst != null) {
			checkVal(vlst, vert);
			vlst = vlst.vNxt;
		}
	}

	private void fixH(PathElt e, int/*ACFixed*/ fixy, int/*ACFixed*/ fixdy)
	{
		PathElt prev, nxt;
		rMovePoint(0, fixdy, CP_START, e);
		rMovePoint(0, fixdy, CP_END, e);
		prev = e.prev;
		if (prev != null && prev.type == CURVETO && prev.y2 == fixy) {
			rMovePoint(0, fixdy, CP_CURVE2, prev);
		}
		if (e.type == CLOSEPATH) {
			e = getDest(e);
		}
		nxt = e.next;
		if (nxt != null && nxt.type == CURVETO && nxt.y1 == fixy) {
			rMovePoint(0, fixdy, CP_CURVE1, nxt);
		}
	}

	private void fixHs(int/*ACFixed*/ fixy, int/*ACFixed*/ fixdy)
	{
		/* y dy in user space */
		PathElt e;
		int/*ACFixed*/ xlst = 0, ylst = 0, xinit = 0, yinit = 0;
		fixy = tfmy(fixy);
		fixdy = dtfmy(fixdy);
		e = mPathStart;
		while (e != null) {
			switch (e.type) {
			case MOVETO:
				xlst = xinit = e.x;
				ylst = yinit = e.y;
				break;
			case LINETO:
				if (e.y == fixy && ylst == fixy) {
					fixH(e, fixy, fixdy);
				}
				xlst = e.x;
				ylst = e.y;
				break;
			case CURVETO:
				xlst = e.x3;
				ylst = e.y3;
				break;
			case CLOSEPATH:
				if (yinit == fixy && ylst == fixy && xinit != xlst) {
					fixH(e, fixy, fixdy);
				}
				break;
			}
			e = e.next;
		}
	}

	private void fixV(PathElt e, int/*ACFixed*/ fixx, int/*ACFixed*/ fixdx)
	{
		PathElt prev, nxt;
		rMovePoint(fixdx, 0, CP_START, e);
		rMovePoint(fixdx, 0, CP_END, e);
		prev = e.prev;
		if (prev != null && prev.type == CURVETO && prev.x2 == fixx) {
			rMovePoint(fixdx, 0, CP_CURVE2, prev);
		}
		if (e.type == CLOSEPATH) {
			e = getDest(e);
		}
		nxt = e.next;
		if (nxt != null && nxt.type == CURVETO && nxt.x1 == fixx) {
			rMovePoint(fixdx, 0, CP_CURVE1, nxt);
		}
	}

	private void fixVs(int/*ACFixed*/ fixx, int/*ACFixed*/ fixdx)
	{
		/* x dx in user space */
		PathElt e;
		int/*ACFixed*/ xlst = 0, ylst = 0, xinit = 0, yinit = 0;
		fixx = tfmx(fixx);
		fixdx = dtfmx(fixdx);
		e = mPathStart;
		while (e != null) {
			switch (e.type) {
			case MOVETO:
				xlst = xinit = e.x;
				ylst = yinit = e.y;
				break;
			case LINETO:
				if (e.x == fixx && xlst == fixx) {
					fixV(e, fixx, fixdx);
				}
				xlst = e.x;
				ylst = e.y;
				break;
			case CURVETO:
				xlst = e.x3;
				ylst = e.y3;
				break;
			case CLOSEPATH:
				if (xinit == fixx && xlst == fixx && yinit != ylst) {
					fixV(e, fixx, fixdx);
				}
				break;
			}
			e = e.next;
		}
	}

	private boolean doFixes()
	{
		boolean didfixes = false;
		int i;
		if (mHFixCount > 0 && mAutoHFix) {
			didfixes = true;
			for (i = 0; i < mHFixCount; i++) {
				fixHs(mHFixYs[i], mHFixDYs[i]);
			}
		}
		if (mVFixCount > 0 && mAutoVFix) {
			didfixes = true;
			for (i = 0; i < mVFixCount; i++) {
				fixVs(mVFixXs[i], mVFixDXs[i]);
			}
		}
		return didfixes;
	}

	/* Module AChead.c */

	private PathElt getDest(PathElt cldest)
	{
		if (cldest == null) {
			return null;
		}
		while (true) {
			cldest = cldest.prev;
			if (cldest == null) {
				return mPathStart;
			}
			if (cldest.type == MOVETO) {
				return cldest;
			}
		}
	}

	private PathElt getClosedBy(PathElt clsdby)
	{
		if (clsdby == null) {
			return null;
		}
		if (clsdby.type == CLOSEPATH) {
			return clsdby;
		}
		while (true) {
			clsdby = clsdby.next;
			if (clsdby == null) {
				return null;
			}
			if (clsdby.type == MOVETO) {
				return null;
			}
			if (clsdby.type == CLOSEPATH) {
				return clsdby;
			}
		}
	}

	private void getEndPoint(PathElt e, Cd endPoint)
	{
		if (e == null) {
			endPoint.x = 0;
			endPoint.y = 0;
			return;
		}
		while (true) {
			switch (e.type) {
			case MOVETO:
			case LINETO:
				endPoint.x = e.x;
				endPoint.y = e.y;
				return;
			case CURVETO:
				endPoint.x = e.x3;
				endPoint.y = e.y3;
				return;
			case CLOSEPATH:
				e = getDest(e);
				continue;
			}
		}
	}

	private void getEndPoints(PathElt p, Cd p0, Cd p1)
	{
		getEndPoint(p, p1);
		getEndPoint(p.prev, p0);
	}

	private double interpolate(double q, double v0, double q0, double v1, double q1)
	{
		return v0 + ((q - q0) * ((v1 - v0) / (q1 - q0)));
	}

	private int/*ACFixed*/ hvNess(double q)
	{
		double result;
		/* approximately == 2 q neg exp */
		/* as q . 0, result goes to 1.0 */
		/* as q . inf, result goes to 0.0 */
		if (q < .25) {
			result = interpolate(q, 1.0, 0.0, .841, .25);
		} else if (q < .5) {
			result = interpolate(q, .841, .25, .707, .5);
		} else if (q < 1) {
			result = interpolate(q, .707, .5, .5, 1.0);
		} else if (q < 2) {
			result = interpolate(q, .5, 1.0, .25, 2.0);
		} else if (q < 4) {
			result = interpolate(q, .25, 2.0, 0.0, 4.0);
		} else {
			result = 0.0;
		}
		return fltToFix(result);
	}

	private int/*ACFixed*/ vertQuo(int/*ACFixed*/ xk, int/*ACFixed*/ yk, int/*ACFixed*/ xl, int/*ACFixed*/ yl)
	{
		/* FIX_ONE means exactly vertical. 0 means not vertical */
		/* intermediate values mean almost vertical */
		int/*ACFixed*/ xabs, yabs;
		double rx, ry, q;
		xabs = xk - xl;
		if (xabs < 0) {
			xabs = -xabs;
		}
		if (xabs == 0) {
			return FIX_ONE;
		}
		yabs = yk - yl;
		if (yabs < 0) {
			yabs = -yabs;
		}
		if (yabs == 0) {
			return 0;
		}
		rx = fixToFlt(xabs);
		ry = fixToFlt(yabs);
		q = (2.0*rx*rx)/(THETA*ry);
		return hvNess(q);
	}

	private int/*ACFixed*/ horzQuo(int/*ACFixed*/ xk, int/*ACFixed*/ yk, int/*ACFixed*/ xl, int/*ACFixed*/ yl)
	{
		int/*ACFixed*/ xabs, yabs;
		double rx, ry, q;
		yabs = yk - yl;
		if (yabs < 0) {
			yabs = -yabs;
		}
		if (yabs == 0) {
			return FIX_ONE;
		}
		xabs = xk - xl;
		if (xabs < 0) {
			xabs = -xabs;
		}
		if (xabs == 0) {
			return 0;
		}
		rx = fixToFlt(xabs);
		ry = fixToFlt(yabs);
		q = (2.0*ry*ry)/(THETA*rx);
		return hvNess(q);
	}

	private boolean isTiny(PathElt e)
	{
		Cd cd0 = new Cd();
		Cd cd1 = new Cd();
		getEndPoints(e, cd0, cd1);
		return ((Math.abs(cd0.x - cd1.x) < FIX_ONE) && (Math.abs(cd0.y - cd1.y) < FIX_ONE)) ? true : false;
	}

	private boolean isShort(PathElt e)
	{
		Cd cd0 = new Cd();
		Cd cd1 = new Cd();
		int/*ACFixed*/ dx, dy, mn, mx;
		getEndPoints(e, cd0, cd1);
		dx = Math.abs(cd0.x - cd1.x);
		dy = Math.abs(cd0.y - cd1.y);
		if (dx > dy) {
			mn = dy;
			mx = dx;
		} else {
			mn = dx;
			mx = dy;
		}
		return ((mx + (mn*42)/125) < fixInt(3)) ? true : false;
	}

	private PathElt nxtForBend(PathElt p, Cd p2, Cd p3)
	{
		PathElt nxt, nxtMT = null;
		Cd ep = new Cd();
		int/*ACFixed*/ x2, y2;
		nxt = p;
		getEndPoint(p, ep);
		while (true) {
			if (nxt.type == CLOSEPATH) {
				nxt = getDest(nxt);
				/* The following test was added to prevent an infinite loop. */
				if (nxtMT != null && nxtMT == nxt) {
					nxt = null;
				} else {
					nxtMT = nxt;
					nxt = nxt.next;
				}
			} else {
				nxt = nxt.next;
			}
			if (nxt == null) {
				/* forget it */
				p2.x = p2.y = p3.x = p3.y = fixInt(-9999);
				return nxt;
			}
			if (!isTiny(nxt)) {
				break;
			}
		}
		if (nxt.type == CURVETO) {
			x2 = nxt.x1;
			y2 = nxt.y1;
			if (x2 == ep.x && y2 == ep.y) {
				x2 = nxt.x2;
				y2 = nxt.y2;
			}
			p2.x = x2;
			p2.y = y2;
		} else {
			getEndPoint(nxt, p2);
		}
		getEndPoint(nxt, p3);
		return nxt;
	}

	private PathElt prvForBend(PathElt p, Cd p2)
	{
		PathElt prv, prvCP = null;
		int/*ACFixed*/ x2, y2;
		prv = p;
		while (true) {
			prv = prv.prev;
			if (prv == null) {
				p2.x = p2.y = fixInt(-9999);
				return prv;
			}
			if (prv.type == MOVETO) {
				prv = getClosedBy(prv);
				/* The following test was added to prevent an infinite loop. */
				if (prv == null || (prvCP != null && prvCP == prv)) {
					p2.x = p2.y = fixInt(-9999);
					return prv;
				}
				prvCP = prv;
			}
			if (!isTiny(prv)) {
				break;
			}
		}
		if (prv.type == CURVETO) {
			x2 = prv.x2;
			y2 = prv.y2;
			if (x2 == prv.x3 && y2 == prv.y3) {
				x2 = prv.x1;
				y2 = prv.y1;
			}
			p2.x = x2;
			p2.y = y2;
		} else {
			p = prv.prev;
			if (p == null) {
				p2.x = p2.y = fixInt(-9999);
				return prv;
			}
			getEndPoint(p, p2);
		}
		return prv;
	}

	private boolean checkHeight(boolean upperFlag, PathElt p)
	{
		PathElt ee;
		int/*ACFixed*/ y, yy;
		ee = mPathStart;
		y = itfmy(p.y);
		while (ee != null) {
			if (ee.type == MOVETO && ee != p) {
				yy = itfmy(ee.y);
				if ((upperFlag && yy > y) || (!upperFlag && yy < y)) {
					return false;
				}
			}
			ee = ee.next;
		}
		return true;
	}

	/* MA - returns true if there is likely no sub-path below this sub-path */
	private boolean isLower(PathElt p)
	{
		return checkHeight(false, p);
	}

	/* MA - returns true if there is likely no sub-path above this sub-path */
	private boolean isUpper(PathElt p)
	{
		return checkHeight(true, p);
	}

	/* Module ACflat.c */

	private short sh_abs(short a)
	{
		return (a < 0) ? (short)-a : a;
	}

	private short mdpt(int a, int b)
	{
		int result ;
		result = (a + b) >> 1;
		return (short)result;
	}

	private void fMiniFltn(Cd f0, Cd f1, Cd f2, Cd f3, FltnRec fr, boolean inside)
	{
		/* Like FFltnCurve, but assumes abs(deltas) <= 127 pixels */
		/* 8 bits of fraction gives enough precision for splitting curves */
		short[] cds = new short[MINIBLKSZ*MINIFLTNMAXDEPTH];
		short dpth, eps, bbLLX = 0, bbLLY = 0, bbURX = 0, bbURY = 0;
		int idx = 0;
		dpth = 1;
		cds[idx++] = inside ? (short)1 : (short)0;	/* initial value of inrect */
		cds[idx++] = (short)0;				/* inbbox starts out false */
		{
			int/*ACFixed*/ llx, lly;
			llx = fr.llx;
			lly = fr.lly;
			cds[idx++] = (short)(f0.x - llx);
			cds[idx++] = (short)(f0.y - lly);
			cds[idx++] = (short)(f1.x - llx);
			cds[idx++] = (short)(f1.y - lly);
			cds[idx++] = (short)(f2.x - llx);
			cds[idx++] = (short)(f2.y - lly);
			cds[idx++] = (short)(f3.x - llx);
			cds[idx++] = (short)(f3.y - lly);
		}
		if (cds[idx - 10]/*inrect*/ == 0) {
			int/*ACFixed*/ c, f128;
			c = fr.ll.x;
			bbLLX = (c <= 0) ? 0 : (short)c;
			c = fr.ll.y;
			bbLLY = (c <= 0) ? 0 : (short)c;
			f128 = fixInt(128);
			c = fr.ur.x;
			bbURX = (c >= f128) ? 0x7fff : (short)c;
			c = fr.ur.y;
			bbURY = (c >= f128) ? 0x7fff : (short)c;
		}
		eps = (short)(fr.feps);
		if (eps < 8) {
			eps = 8;	 /* Brotz patch */
		}
		while (true) {
			while (true) {
				if (dpth == MINIFLTNMAXDEPTH) {
					break;
				}
				if (cds[idx - 10]/*inrect*/ == 0) {
					short llx, lly, urx, ury, c;
					llx = urx = cds[idx - 8]/*c0x*/;
					if ((c = cds[idx - 6]/*c1x*/) < llx) {
						llx = c;
					} else if (c > urx) {
						urx = c;
					}
					if ((c = cds[idx - 4]/*c2x*/) < llx) {
						llx = c;
					} else if (c > urx) {
						urx = c;
					}
					if ((c = cds[idx - 2]/*c3x*/) < llx) {
						llx = c;
					} else if (c > urx) {
						urx = c;
					}
					if (urx < bbLLX || llx > bbURX) {
						break;
					}
					lly = ury = cds[idx - 7]/*c0y*/;
					if ((c = cds[idx - 5]/*c1y*/) < lly) {
						lly = c;
					} else if (c > ury) {
						ury = c;
					}
					if ((c = cds[idx - 3]/*c2y*/) < lly) {
						lly = c;
					} else if (c > ury) {
						ury = c;
					}
					if ((c = cds[idx - 1]/*c3y*/) < lly) {
						lly = c;
					} else if (c > ury) {
						ury = c;
					}
					if (ury < bbLLY || lly > bbURY) {
						break;
					}
					if (urx <= bbURX && ury <= bbURY && llx >= bbLLX && lly >= bbLLY) {
						cds[idx - 10]/*inrect*/ = 1;
					}
				}
				if (cds[idx - 9]/*inbbox*/ == 0) {
					short mrgn = eps, r0, r3, ll, ur, c;
					r0 = cds[idx - 8]/*c0x*/;
					r3 = cds[idx - 2]/*c3x*/;
					if (r0 < r3) {
						ll = (short)(r0 - mrgn);
						ur = (short)(r3 + mrgn);
					} else {
						ll = (short)(r3 - mrgn);
						ur = (short)(r0 + mrgn);
					}
					if (ur < 0) {
						ur = (128 << 8) - 1;
					}
					c = cds[idx - 6]/*c1x*/;
					if (c > ll && c < ur) {
						c = cds[idx - 4]/*c2x*/;
						if (c > ll && c < ur) {
							r0 = cds[idx - 7]/*c0y*/;
							r3 = cds[idx - 1]/*c3y*/;
							if (r0 < r3) {
								ll = (short)(r0 - mrgn);
								ur = (short)(r3 + mrgn);
							} else {
								ll = (short)(r3 - mrgn);
								ur = (short)(r0 + mrgn);
							}
							if (ur < 0) {
								ur = (128 << 8) - 1;
							}
							c = cds[idx - 5]/*c1y*/;
							if (c > ll && c < ur) {
								c = cds[idx - 3]/*c2y*/;
								if (c > ll && c < ur) {
									cds[idx - 9]/*inbbox*/ = 1;
								}
							}
						}
					}
				}
				if (cds[idx - 9]/*inbbox*/ != 0) {
					short eqa, eqb, x, y;
					int/*ACFixed*/ EPS, d;
					x = cds[idx - 8]/*c0x*/;
					y = cds[idx - 7]/*c0y*/;
					eqa = (short)(cds[idx - 1]/*c3y*/ - y);
					eqb = (short)(x - cds[idx - 2]/*c3x*/);
					if (eqa == 0 && eqb == 0) {
						break;
					}
					EPS = ((sh_abs(eqa) > sh_abs(eqb)) ? (int)eqa : (int)eqb)*eps;
					if (EPS < 0) {
						EPS = -EPS;
					}
					d = eqa*(cds[idx - 6]/*c1x*/ - x);
					d += eqb*(cds[idx - 5]/*c1y*/ - y);
					if (Math.abs(d) < EPS) {
						d = eqa*(cds[idx - 4]/*c2x*/ - x);
						d += eqb*(cds[idx - 3]/*c2y*/ - y);
						if (Math.abs(d) < EPS) {
							break;
						}
					}
				}
				{
					/* Bezier divide */
					short c0, c1, c2, d1, d2, d3;
					cds[idx + 2]/*d0x*/ = c0 = cds[idx - 8]/*c0x*/;
					c1 = cds[idx - 6]/*c1x*/;
					c2 = cds[idx - 4]/*c2x*/;
					cds[idx + 4]/*d1x*/ = d1 = mdpt(c0, c1);
					d3 = mdpt(c1, c2);
					cds[idx + 6]/*d2x*/ = d2 = mdpt(d1, d3);
					cds[idx - 4]/*c2x*/ = c2 = mdpt(c2, cds[idx - 2]/*c3x*/);
					cds[idx - 6]/*c1x*/ = c1 = mdpt(d3, c2);
					cds[idx - 8]/*c0x*/ = cds[idx + 8]/*d3x*/ = mdpt(d2, c1);
					cds[idx + 3]/*d0y*/ = c0 = cds[idx - 7]/*c0y*/;
					c1 = cds[idx - 5]/*c1y*/;
					c2 = cds[idx - 3]/*c2y*/;
					cds[idx + 5]/*d1y*/ = d1 = mdpt(c0, c1);
					d3 = mdpt(c1, c2);
					cds[idx + 7]/*d2y*/ = d2 = mdpt(d1, d3);
					cds[idx - 3]/*c2y*/ = c2 = mdpt(c2, cds[idx - 1]/*c3y*/);
					cds[idx - 5]/*c1y*/ = c1 = mdpt(d3, c2);
					cds[idx - 7]/*c0y*/ = cds[idx + 9]/*d3y*/ = mdpt(d2, c1);
					cds[idx + 1]/*bbox2*/ = cds[idx - 9]/*inbbox*/;
					cds[idx + 0]/*inrect2*/ = cds[idx - 10]/*inrect*/;
					idx += MINIBLKSZ;
					dpth++;
					continue;
				}
			}
			{
				Cd c = new Cd();
				if (--dpth == 0) {
					c.copyFrom(f3);
				} else {
					c.x = cds[idx - 2]/*c3x*/ + fr.llx;
					c.y = cds[idx - 1]/*c3y*/ + fr.lly;
				}
				flatReportProc(fr.report, fr.param, c);
				if (dpth == 0) {
					return;
				}
				idx -= MINIBLKSZ;
			}
		}
	}

	/* inrect = !testRect */
	/* Like FltnCurve, but works in the ACFixed domain. */
	/* abs values of coords must be < 2^14 so will not overflow when
	   find midpoint by add and shift */
	private void fFltnCurve(Cd c0, Cd c1, Cd c2, Cd c3, FltnRec fr, boolean inrect)
	{
		c0 = (Cd)c0.clone();
		c1 = (Cd)c1.clone();
		c2 = (Cd)c2.clone();
		c3 = (Cd)c3.clone();
		Cd d0 = new Cd();
		Cd d1 = new Cd();
		Cd d2 = new Cd();
		Cd d3 = new Cd();
		int/*ACFixed*/ llx, lly, urx, ury;
		if ((c0.x == c1.x && c0.y == c1.y && c2.x == c3.x && c2.y == c3.y) || fr.limit <= 0) {
			flatReportProc(fr.report, fr.param, c3);
			return;
		}
		{
			int/*ACFixed*/ c;
			llx = urx = c0.x;
			if ((c = c1.x) < llx) {
				llx = c;
			} else if (c > urx) {
				urx = c;
			}
			if ((c = c2.x) < llx) {
				llx = c;
			} else if (c > urx) {
				urx = c;
			}
			if ((c = c3.x) < llx) {
				llx = c;
			} else if (c > urx) {
				urx = c;
			}
			lly = ury = c0.y;
			if ((c = c1.y) < lly) {
				lly = c;
			} else if (c > ury) {
				ury = c;
			}
			if ((c = c2.y) < lly) {
				lly = c;
			} else if (c > ury) {
				ury = c;
			}
			if ((c = c3.y) < lly) {
				lly = c;
			} else if (c > ury) {
				ury = c;
			}
		}
		if (!inrect) {
			if (urx < fr.ll.x || llx > fr.ur.x || ury < fr.ll.y || lly > fr.ur.y) {
				flatReportProc(fr.report, fr.param, c3);
				return;
			}
			if (urx <= fr.ur.x && ury <= fr.ur.y && llx >= fr.ll.x && lly >= fr.ll.y) {
				inrect = true;
			}
		}
		{
			int/*ACFixed*/ th;
			th = fixInt(127);		/* delta threshhold of 127 pixels */
			if (urx - llx >= th || ury - lly >= th) {
				d3.copyFrom(c3);
				d2.x = (c2.x + c3.x) >> 1;
				d2.y = (c2.y + c3.y) >> 1;
				c3.x = (c1.x + c2.x) >> 1;
				c3.y = (c1.y + c2.y) >> 1;
				c1.x = (c0.x + c1.x) >> 1;
				c1.y = (c0.y + c1.y) >> 1;
				c2.x = (c1.x + c3.x) >> 1;
				c2.y = (c1.y + c3.y) >> 1;
				d1.x = (c3.x + d2.x) >> 1;
				d1.y = (c3.y + d2.y) >> 1;
				d0.x = (c2.x + d1.x) >> 1;
				d0.y = (c2.y + d1.y) >> 1;
				c3.copyFrom(d0);
				fr.limit--;
				fFltnCurve(c0, c1, c2, c3, fr, inrect);
				fFltnCurve(d0, d1, d2, d3, fr, inrect);
				fr.limit++;
				return;
			}
		}
		fr.llx = llx;
		fr.lly = lly;
		if (!inrect) {
			fr.ll.x -= llx;
			fr.ur.x -= llx;
			fr.ll.y -= lly;
			fr.ur.y -= lly;
		}
		fMiniFltn(c0, c1, c2, c3, fr, inrect);
		if (!inrect) {
			fr.ll.x += llx;
			fr.ur.x += llx;
			fr.ll.y += lly;
			fr.ur.y += lly;
		}
	}

	private void fltnCurve(Cd c0, Cd c1, Cd c2, Cd c3, FltnRec fr)
	{
		fr.limit = 6;
		fr.feps = FIX_HALF;
		fFltnCurve(c0, c1, c2, c3, fr, true);
	}

	/* Module ACmisc.c */

	private int countSubPaths(int[] elemCount)
	{
		int numSubPaths = 0;
		int numElem = 0;
		PathElt e = mPathEnd;
		while (e != null) {
			while (true) {
				e = e.prev;
				numElem++ ;
				if (e == null || e.type == MOVETO) {
					break;
				}
			}
			numSubPaths++;
			if (e != null) {
				e = e.prev;
			}
		}
		if (elemCount != null) {
			elemCount[0] = numElem;
		}
		return numSubPaths;
	}

	private void roundPathCoords()
	{
		PathElt e;
		e = mPathStart;
		while (e != null) {
			if (e.type == CURVETO) {
				e.x1 = fHalfRnd(e.x1);
				e.y1 = fHalfRnd(e.y1);
				e.x2 = fHalfRnd(e.x2);
				e.y2 = fHalfRnd(e.y2);
				e.x3 = fHalfRnd(e.x3);
				e.y3 = fHalfRnd(e.y3);
			} else {
				if (e.type == LINETO || e.type == MOVETO) {
					e.x = fHalfRnd(e.x);
					e.y = fHalfRnd(e.y);
				}
			}
			e = e.next;
		}
	}

	private int checkForClr()
	{
		PathElt mt, cp;
		mt = mPathStart;
		while (mt != null) {
			cp = getClosedBy(mt);
			mt = cp.next;
		}
		return 0;
	}

	private boolean preCheckForColoring()
	{
		PathElt e, nxt;
		int cnt = 0;
		int chk;
		while (mPathEnd != null) {
			if (mPathEnd.type == MOVETO) {
				delete(mPathEnd);
			} else if (mPathEnd.type != CLOSEPATH) {
				return false;
			} else {
				break;
			}
		}
		e = mPathStart;
		while (e != null) {
			if (e.type == CLOSEPATH) {
				if (e == mPathEnd) {
					break;
				}
				nxt = e.next;
				if (nxt.type == MOVETO) {
					e = nxt;
					continue;
				}
				if (nxt.type == CLOSEPATH) {
					/* remove double closepath */
					delete(nxt);
					continue;
				}
			}
			e = e.next;
		}
		while (true) {
			chk = checkForClr();
			if (chk == -1) {
				return false;
			}
			if (chk == 0) {
				break;
			}
			if (++cnt > 10) {
				break;
			}
		}
		return true;
	}

	private PathElt getSubpathNext(PathElt e)
	{
		while (true) {
			if (e.type == CLOSEPATH) {
				e = getDest(e);
			}
			e = e.next;
			if (e == null) {
				break;
			}
			if (!isTiny(e)) {
				break;
			}
		}
		return e;
	}

	private PathElt getSubpathPrev(PathElt e)
	{
		while (true) {
			e = e.prev;
			if (e == null) {
				break;
			}
			if (e.type == MOVETO) {
				e = getClosedBy(e);
			}
			if (!isTiny(e)) {
				break;
			}
		}
		return e;
	}

	private boolean addAutoFlexProp(PathElt e, boolean yflag)
	{
		PathElt e0 = e, e1 = e.next;
		/* Don't add flex to linear curves. */
		if (yflag && e0.y3 == e1.y1 && e1.y1 == e1.y2 && e1.y2 == e1.y3) {
			return false;
		} else if (e0.x3 == e1.x1 && e1.x1 == e1.x2 && e1.x2 == e1.x3) {
			return false;
		}
		e0.isFlex = true;
		e1.isFlex = true;
		return true;
	}

	private void tryYFlex(PathElt e, PathElt n, int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1)
	{
		Cd cd2 = new Cd();
		Cd cd3 = new Cd();
		Cd cd4 = new Cd();
		PathElt p, q;
		boolean top, dwn;
		double d0sq, d1sq, quot, dx, dy;
		getEndPoint(n, cd2);
		if (Math.abs(y0 - cd2.y) > mFlexCand) {
			return;					/* too big diff in bases */
		}
		if (prodLt0(y1 - y0, y1 - cd2.y)) {
			return;					/* y0 and y2 not on same side of y1 */
		}
		/* check the ratios of the "lengths" of 'e' and 'n'	 */
		dx = (x1 - x0);
		dy = (y1 - y0);
		d0sq = dx*dx + dy*dy;
		dx = (cd2.x - x1);
		dy = (cd2.y - y1);
		d1sq = dx*dx + dy*dy;
		quot = (d0sq > d1sq) ? (d1sq / d0sq) : (d0sq / d1sq);
		if (quot < LENGTHRATIOCUTOFF) {
			return;
		}
		if (mFlexStrict) {
			q = getSubpathNext(n);
			getEndPoint(q, cd3);
			if (prodLt0(cd3.y - cd2.y, y1 - cd2.y)) {
				return;				/* y1 and y3 not on same side of y2 */
			}
			p = getSubpathPrev(e);
			getEndPoint(p.prev, cd4);
			if (prodLt0(cd4.y - y0, y1 - y0)) {
				return;				/* y1 and y4 not on same side of y0 */
			}
			top = (x0 > x1) ? true: false;
			if (Y_GOES_UP) {
				dwn = (y1 < y0) ? true: false;
			} else {
				dwn = (y1 > y0) ? true: false;
			}
			if ((top && !dwn) || (!top && dwn)) {
				return;				/* concave */
			}
		}
		if (n != e.next) {
			/* something in the way */
			return;
		}
		if (y0 != cd2.y) {
			return;
		}
		addAutoFlexProp(e, true);
	}

	private void tryXFlex(PathElt e, PathElt n, int/*ACFixed*/ x0, int/*ACFixed*/ y0, int/*ACFixed*/ x1, int/*ACFixed*/ y1)
	{
		Cd cd2 = new Cd();
		Cd cd3 = new Cd();
		Cd cd4 = new Cd();
		PathElt p, q;
		boolean lft;
		double d0sq, d1sq, quot, dx, dy;
		getEndPoint(n, cd2);
		if (Math.abs(x0 - cd2.x) > mFlexCand) {
			return;					/* too big diff in bases */
		}
		if (prodLt0(x1 - x0, x1 - cd2.x)) {
			return;					/* x0 and x2 not on same side of x1 */
		}
		/* check the ratios of the "lengths" of 'e' and 'n'	 */
		dx = (x1 - x0);
		dy = (y1 - y0);
		d0sq = dx*dx + dy*dy;
		dx = (cd2.x - x1);
		dy = (cd2.y - y1);
		d1sq = dx*dx + dy*dy;
		quot = (d0sq > d1sq) ? (d1sq / d0sq) : (d0sq / d1sq);
		if (quot < LENGTHRATIOCUTOFF) {
			return;
		}
		if (mFlexStrict) {
			q = getSubpathNext(n);
			getEndPoint(q, cd3);
			if (prodLt0(cd3.x - cd2.x, x1 - cd2.x)) {
				return;				/* x1 and x3 not on same side of x2 */
			}
			p = getSubpathPrev(e);
			getEndPoint(p.prev, cd4);
			if (prodLt0(cd4.x - x0, x1 - x0)) {
				return;				/* x1 and x4 not on same side of x0 */
			}
			if (Y_GOES_UP) {
				lft = (y0 > cd2.y) ? true: false;
			} else {
				lft = (y0 < cd2.y) ? true: false;
			}
			if ((lft && x0 > x1) || (!lft && x0 < x1)) {
				return;				/* concave */
			}
		}
		if (n != e.next) {
			/* something in the way */
			return;
		}
		if (x0 != cd2.x) {
			return;
		}
		addAutoFlexProp(e, false);
	}

	private void autoAddFlex()
	{
		PathElt e, n;
		Cd cd0 = new Cd();
		Cd cd1 = new Cd();
		for (e = mPathStart; e != null; e = e.next) {
			if (e.type != CURVETO || e.isFlex) {
				continue;
			}
			n = getSubpathNext(e);
			if (n.type != CURVETO) {
				continue;
			}
			getEndPoints(e, cd0, cd1);
			if (Math.abs(cd0.y - cd1.y) <= psDist(20)) {
				tryYFlex(e, n, cd0.x, cd0.y, cd1.x, cd1.y);
			}
			if (Math.abs(cd0.x - cd1.x) <= psDist(20)) {
				tryXFlex(e, n, cd0.x, cd0.y, cd1.x, cd1.y);
			}
		}
	}

	/* Module ACcharprop.c */

	private short VColorList[] = {
		'm', 	/* m */
		'M', 	/* M */
		'T', 	/* T */
		0x2026, 	/* ellipsis */
		0
	};

	private short HColorList[] = {
		0x2208, 	/* element */
		0x2261, 	/* equivalence */
		0x2209, 	/* notelement */
		0x00F7, 	/* divide */
		0
	};

	private short UpperSpecialChars[] = {
		0x00BF, 	/* questiondown */
		0x00A1, 	/* exclamdown */
		';', 		/* semicolon */
		0
	};

	private short LowerSpecialChars[] = {
		'?', 		/* question */
		'!', 		/* exclam */
		':', 		/* colon */
		0
	};

	private short NoBlueList[] = {
		'@', 		/* at */
		0x2022, 	/* bullet */
		0x00A9, 	/* copyright */
		0x00A4, 	/* currency */
		0x00AE, 	/* registered */
		0
	};

	private boolean findCodeInList(int unicode, short[] lst)
	{
		int u;
		int idx = 0;
		while (true) {
			u = lst[idx++];
			if (u == 0) {
				return false;
			}
			if (unicode == u) {
				return true;
			}
		}
	}

	private int specialCharType(int unicode)
	{
		/* 1 = upper; -1 = lower; 0 = neither */
		if (findCodeInList(unicode, UpperSpecialChars)) {
			return 1;
		}
		if (findCodeInList(unicode, LowerSpecialChars)) {
			return -1;
		}
		return 0;
	}

	private boolean hColorChar(int unicode)
	{
		return findCodeInList(unicode, HColorList);
	}

	private boolean vColorChar(int unicode)
	{
		return findCodeInList(unicode, VColorList);
	}

	private boolean noBlueChar(int unicode)
	{
		return findCodeInList(unicode, NoBlueList);
	}

	private boolean moveToNewClrs(int unicode)
	{
		return (unicode == 0x0025/*percent*/) || (unicode == 0x2030/*perthousand*/);
	}

	/* ACshuffle.c */
	private void initShuffleSubpaths()
	{
		int cnt = -1;
		PathElt e = mPathStart;
		while (e != null) {
			/* every element is marked with its subpath count */
			if (e.type == MOVETO) {
				cnt++;
			}
			e.count = (short)cnt;
			e = e.next;
		}
		cnt++;
		mRowCnt = cnt;
		mLinks = (cnt < 4 || cnt >= MAXCNT) ? null : new byte[cnt * cnt];
	}

	private void markLinks(ClrVal vL, boolean hFlg)
	{
		int i, j;
		ClrSeg seg;
		PathElt e;
		if (mLinks == null) {
			return;
		}
		for (; vL != null; vL = vL.vNxt) {
			if (vL == null) {
				continue;
			}
			seg = vL.vSeg1;
			if (seg == null) {
				continue;
			}
			e = seg.sElt;
			if (e == null) {
				continue;
			}
			i = e.count;
			seg = vL.vSeg2;
			if (seg == null) {
				continue;
			}
			e = seg.sElt;
			if (e == null) {
				continue;
			}
			j = e.count;
			if (i == j) {
				continue;
			}
			mLinks[mRowCnt * i + j] = 1;
			mLinks[mRowCnt * j + i] = 1;
		}
	}

	private void outPath(byte[] links, byte[] outlinks, byte[] output, int bst)
	{
		int srcIdx;
		int i = bst;
		PathElt e = mPathStart;
		while (e != null) {
			if (e.count == i) {
				break;
			}
			e = e.next;
		}
		if (e != null) {
			moveSubpathToEnd(e);
		}
		output[bst] = 1;
		srcIdx = bst * mRowCnt;
		for (i = 0; i < mRowCnt; i++) {
			outlinks[i] += links[srcIdx + 1];
		}
	}

	/* The intent of this code is to order the subpaths so that
	   the hints will not need to change constantly because it
	   is jumping from one subpath to another. Kanji characters
	   had the most problems with this which caused huge files
	   to be created. */
	private void doShuffleSubpaths()
	{
		byte[] sumlinks = new byte[MAXCNT];
		byte[] output = new byte[MAXCNT];
		byte[] outlinks = new byte[MAXCNT];
		int idx;
		int i, j, bst, bstsum, bstlnks;
		if (mLinks == null) {
			return;
		}
		for (i = 0; i < mRowCnt; i++) {
			output[i] = sumlinks[i] = outlinks[i] = 0;
		}
		for (i = 0, idx = 0; i < mRowCnt; i++) {
			for (j = 0; j < mRowCnt; j++) {
				if (mLinks[idx++] != 0) {
					sumlinks[i]++;
				}
			}
		}
		while (true) {
			bst = -1;
			bstsum = 0;
			for (i = 0; i < mRowCnt; i++) {
				if (output[i] == 0 && (bst == -1 || sumlinks[i] > bstsum)) {
					bstsum = sumlinks[i];
					bst = i;
				}
			}
			if (bst == -1) {
				break;
			}
			outPath(mLinks, outlinks, output, bst);
			while (true) {
				bst = -1;
				bstsum = 0;
				bstlnks = 0;
				for (i = 0; i < mRowCnt; i++) {
					if (output[i] == 0 && outlinks[i] >= bstlnks) {
						if (outlinks[i] > 0 &&
						    (bst == -1 || outlinks[i] > bstlnks ||
						     (outlinks[i] == bstlnks && sumlinks[i] > bstsum))) {
							bstlnks = outlinks[i];
							bst = i;
							bstsum = sumlinks[i];
						}
					}
				}
				if (bst == -1) {
					break;
				}
				outPath(mLinks, outlinks, output, bst);
			}
		}
	}

	/* Module ACstub.c */

	/* Procedures defined in stems/stemreport.c - StemHist */
	private void addVStem(int/*ACFixed*/ right, int/*ACFixed*/ left, boolean curved)
	{
		if (!curved || mAllStems) {
			mZoneData.add(ZN_VSTEM);
			mZoneData.add(fTrunc(fRnd(right)));
			mZoneData.add(fTrunc(fRnd(left)));
		}
	}

	private void addHStem(int/*ACFixed*/ top, int/*ACFixed*/ bottom, boolean curved)
	{
		if (!curved || mAllStems) {
			mZoneData.add(ZN_HSTEM);
			mZoneData.add(fTrunc(fRnd(top)));
			mZoneData.add(fTrunc(fRnd(bottom)));
		}
	}

	private void addCharExtreme(int/*ACFixed*/ bot, int/*ACFixed*/ top)
	{
		mZoneData.add(ZN_CHAREXTREME);
		mZoneData.add(fTrunc(fRnd(top)));
		mZoneData.add(fTrunc(fRnd(bot)));
	}

	private void addCharZone(int/*ACFixed*/ bot, int/*ACFixed*/ top)
	{
		mZoneData.add(ZN_CHARZONE);
		mZoneData.add(fTrunc(fRnd(top)));
		mZoneData.add(fTrunc(fRnd(bot)));
	}

	private void addZone(int/*ACFixed*/ bot, int/*ACFixed*/ top)
	{
		mZoneData.add(ZN_ZONE);
		mZoneData.add(fTrunc(fRnd(top)));
		mZoneData.add(fTrunc(fRnd(bot)));
	}

	/* Module ACfixed.c */

	private double fixToFlt(int/*ACFixed*/ x)
	{
		return (double)x / (double)FIX_ONE;
	}

	private int/*ACFixed*/ fltToFix(double f)
	{
		if (f >= FIXED_POS_INF / (double)FIX_ONE) {
			return FIXED_POS_INF;
		}
		if (f <= FIXED_NEG_INF / (double)FIX_ONE) {
			return FIXED_NEG_INF;
		}
		return (int/*ACFixed*/)(f * FIX_ONE);
	}

	/* Module ACread.c */

	/* append a new path element to inputPath (not current path) */
	private PathElt appendElement(int etype)
	{
		PathElt e;
		e = new PathElt();
		e.eltSN = ++mEltSN;
		if (e.eltSN == 0) e.eltSN++;
		e.type = (short)etype;
		if (mInputPathEnd != null) {
			mInputPathEnd.next = e;
			e.prev = mInputPathEnd;
		} else {
			mInputPathStart = e;
		}
		mInputPathEnd = e;
		return e;
	}

	private void closeSubPath()
	{
		if (mSubPathOpen) {
			mSubPathOpen = false;
			appendElement(CLOSEPATH);
		}
	}

	public void moveto(double x0, double y0)
	{
		PathElt newElt;
		closeSubPath();
		newElt = appendElement(MOVETO);
		newElt.x = tfmx(scaleAbs(fltToFix(x0)));
		newElt.y = tfmy(scaleAbs(fltToFix(y0)));
		mCurX = x0;
		mCurY = y0;
		mSubPathOpen = true;
	}

	public void lineto(double x1, double y1)
	{
		PathElt newElt;
		newElt = appendElement(LINETO);
		newElt.x = tfmx(scaleAbs(fltToFix(x1)));
		newElt.y = tfmy(scaleAbs(fltToFix(y1)));
		mCurX = x1;
		mCurY = y1;
	}

	public void curveto(double x1, double y1, double x2, double y2)
	{
		curveto(Math.round((mCurX + 2*x1) / 3.0), Math.round((mCurY + 2*y1) / 3.0),
			Math.round((2*x1 + x2) / 3.0), Math.round((2*y1 + y2) / 3.0),
			Math.round(x2), Math.round(y2));
	}

	public void curveto(double x1, double y1, double x2, double y2, double x3, double y3)
	{
		PathElt newElt;
		newElt = appendElement(CURVETO);
		newElt.x1 = tfmx(scaleAbs(fltToFix(x1)));
		newElt.y1 = tfmy(scaleAbs(fltToFix(y1)));
		newElt.x2 = tfmx(scaleAbs(fltToFix(x2)));
		newElt.y2 = tfmy(scaleAbs(fltToFix(y2)));
		newElt.x3 = tfmx(scaleAbs(fltToFix(x3)));
		newElt.y3 = tfmy(scaleAbs(fltToFix(y3)));
		mCurX = x3;
		mCurY = y3;
	}

	public void endchar()
	{
		closeSubPath();
		try {
			doGlyph();
		} catch (ACException e) {
			emitRawPath();
		}
	}

	public void setMatrix(Matrix matrix) {
		if (mOutlineConsumer != null) {
			mOutlineConsumer.setMatrix(matrix);
		}
	}

	/* Called on glyph failure. Simply emit the input path to the output consumer. */
	private void emitRawPath()
	{
		if (mOutlineConsumer != null) {
			PathElt e = mPathStart;
			while (e != null) {
				switch (e.type) {
				case MOVETO: {
					double x = fixToFlt(unScaleAbs(itfmx(e.x)));
					double y = fixToFlt(unScaleAbs(itfmx(e.y)));
					mOutlineConsumer.moveto(x,y);
					break;
				}
				case LINETO: {
					double x = fixToFlt(unScaleAbs(itfmx(e.x)));
					double y = fixToFlt(unScaleAbs(itfmx(e.y)));
					mOutlineConsumer.lineto(x,y);
					break;
				}
				case CURVETO: {
					double c1x = fixToFlt(unScaleAbs(itfmx(e.x1)));
					double c1y = fixToFlt(unScaleAbs(itfmy(e.y1)));
					double c2x = fixToFlt(unScaleAbs(itfmx(e.x2)));
					double c2y = fixToFlt(unScaleAbs(itfmy(e.y2)));
					double c3x = fixToFlt(unScaleAbs(itfmx(e.x3)));
					double c3y = fixToFlt(unScaleAbs(itfmy(e.y3)));
					mOutlineConsumer.curveto(c1x, c1y, c2x, c2y, c3x, c3y);
					break;
				}
				case CLOSEPATH: {
					break;
				}
				}
				e = e.next;
			}
			mOutlineConsumer.endchar();
		}
	}

	/* copy the inputGlyph to the current path */
	private void resetGlyph()
	{
		PathElt ie, ce;
		mPathStart = mPathEnd = null;
		mCurX = mCurY = 0;
		/* walk through the inputPath and copy each element to the current path */
		ie = mInputPathStart;
		while (ie != null) {
			ce = (PathElt)ie.clone();
			if (mPathEnd != null) {
				mPathEnd.next = ce;
				ce.prev = mPathEnd;
			} else {
				mPathStart = ce;
			}
			mPathEnd = ce;
			/* next input path element */
			ie = ie.next;
		}
	}

	/* Module utility stuff */

	private void copyBands()
	{
		int i;
		mLenTopBands = Math.min(mTopZones.length, MAXBLUES);
		for (i = 0; i < mLenTopBands; i++) {
			mTopBands[i] = scaleAbs(fixInt(mTopZones[i]));
		}
		mLenBotBands = Math.min(mBottomZones.length, MAXBLUES);
		for (i = 0; i < mLenBotBands; i++) {
			mBotBands[i] = scaleAbs(fixInt(mBottomZones[i]));
		}
	}

	private int/*ACFixed*/ fixInt(int i)
	{
		return i << FIX_SHIFT;
	}

	private int/*ACFixed*/ fRnd(int/*ACFixed*/ x)
	{
		return (x + FIX_HALF) & (-1 << FIX_SHIFT);
	}

	private int/*ACFixed*/ fHalfRnd(int/*ACFixed*/ x)
	{
		return (x + FIX_QUARTER) & (-1 << (FIX_SHIFT - 1));
	}

	private int fTrunc(int/*ACFixed*/ x)
	{
		return x >> FIX_SHIFT;
	}

	private int/*ACFixed*/ fixHalfMul(int/*ACFixed*/ f)
	{
		return f >> 1;
	}

	private int/*ACFixed*/ fixTwoMul(int/*ACFixed*/ f)
	{
		return f << 1;
	}

	private int/*ACFixed*/ tfmx(int/*ACFixed*/ x)
	{
		return fixHalfMul(x) + X0;
	}

	private int/*ACFixed*/ tfmy(int/*ACFixed*/ y)
	{
		return fixHalfMul(y) + Y0;
	}

	private int/*ACFixed*/ itfmx(int/*ACFixed*/ x)
	{
		return fixTwoMul(x) - X0;
	}

	private int/*ACFixed*/ itfmy(int/*ACFixed*/ y)
	{
		return fixTwoMul(y) - Y0;
	}

	private int/*ACFixed*/ dtfmx(int/*ACFixed*/ x)
	{
		return fixHalfMul(x);
	}

	private int/*ACFixed*/ dtfmy(int/*ACFixed*/ y)
	{
		return fixHalfMul(y);
	}

	private int/*ACFixed*/ idtfmx(int/*ACFixed*/ x)
	{
		return fixTwoMul(x);
	}

	private int/*ACFixed*/ idtfmy(int/*ACFixed*/ y)
	{
		return fixTwoMul(y);
	}

	private int/*ACFixed*/ psDist(int d)
	{
		return dtfmx(fixInt(d));
	}

	private boolean isVertical(int/*ACFixed*/ x1, int/*ACFixed*/ y1, int/*ACFixed*/ x2, int/*ACFixed*/ y2)
	{
		return vertQuo(x1, y1, x2, y2) > 0;
	}

	private boolean isHorizontal(int/*ACFixed*/ x1, int/*ACFixed*/ y1, int/*ACFixed*/ x2, int/*ACFixed*/ y2)
	{
		return horzQuo(x1, y1, x2, y2) > 0;
	}

	private boolean prodLt0(int/*ACFixed*/ f0, int/*ACFixed*/ f1)
	{
		return (f0 < 0 && f1 > 0) || (f0 > 0 && f1 < 0);
	}

	private boolean prodGe0(int/*ACFixed*/ f0, int/*ACFixed*/ f1)
	{
		return !prodLt0(f0, f1);
	}

	private ClrSeg leftList()
	{
		return mSegLists[0];
	}

	private ClrSeg rightList()
	{
		return mSegLists[1];
	}

	private ClrSeg topList()
	{
		return mSegLists[2];
	}

	private ClrSeg botList()
	{
		return mSegLists[3];
	}
}
