/*
*
*	File: Fond.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2010 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import java.io.UnsupportedEncodingException;
import java.util.Map;

import com.adobe.agl.util.ULocale;
import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.Subset;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.font.opentype.OTByteArray.OTByteArrayBuilder;

/**
 * Gives access to the Apple 'fond' table.
 */
final public class Fond extends Table
{
	// Offsets
	static final int fVersion = 0;		// uint16_t
	static final int fCount_fond = 2;	// uint16_t
	static final int fCount_nfnt = 4;	// uint16_t
	static final int fUnused = 6;		// uint16_t
	static final int fResourceID = 8;	// int16_t
	static final int fFamilyID = 10;	// int16_t
	static final int fFlags = 12;		// uint16_t
	static final int fStyle = 14;		// uint16_t
	static final int fScript = 16;		// int16_t
	static final int fLanguage = 18;	// int16_t
	static final int fOffset = 20;		// uint32_t
	static final int fLength = 24;		// uint32_t
	static final int fNameLength = 28;	// uint8_t
	static final int fName = 29;		// uint8_t[]

	// Style bits
	static final int STYLE_BOLD = 1;
	static final int STYLE_ITALIC = 2;

	// Locale strings
	static final String[] SCRIPTID_TO_ULOCALE =
	{
		"en",			//smRoman		0	Roman		-> English
		"ja",			//smJapanese		1	Japanese	-> Japanese
		"zh-Hant",		//smTradChinese		2	Traditional Chinese
		"ko",			//smKorean		3	Korean		-> Korean
		"ar",			//smArabic		4	Arabic		-> Arabic
		"he",			//smHebrew		5	Hebrew		-> Hebrew
		"el",			//smGreek		6	Greek		-> Greek
		"ru",			//smCyrillic		7	Cyrillic	-> Russian
		"MacSymbol",		//smRSymbol		8	Right-to-left symbols
		"hi",			//smDevanagari		9	Devanagari	-> Hindi
		"pa",			//smGurmukhi		10	Gurmukhi	-> Punjabi
		"gu",			//smGujarati		11	Gujarati	-> Gujarati
		null,			//smOriya		12	Oriya
		null,			//smBengali		13	Bengali
		null,			//smTamil		14	Tamil
		null,			//smTelugu		15	Telugu
		null,			//smKannada		16	Kannada/Kanarese
		null,			//smMalayalam		17	Malayalam
		null,			//smSinhalese		18	Sinhalese
		null,			//smBurmese		19	Burmese
		null,			//smKhmer		20	Khmer
		"th",			//smThai		21	Thai		-> Thai
		null,			//smLaotian		22	Laotian
		null,			//smGeorgian		23	Georgian
		null,			//smArmenian		24	Armenian
		"zh-Hans",		//smSimpChinese		25	Simplified Chinese
		null,			//smTibetan		26	Tibetan
		null,			//smMongolian		27	Mongolian
		null,			//smEthiopic		28	Geez/Ethiopic
		"MacCentralEuropean",	//smEastEurRoman	29	Extended Roman for Slavic and Baltic languages
		null,			//smVietnamese		30	Extended Roman for Vietnamese
		null,			//smExtArabic		31	Extended Arabic for Sindhi
		null,			//smUninterp		32	Uninterpreted symbols
	};

	protected Fond(FontByteArray buffer)
		throws java.io.IOException, InvalidFontException, UnsupportedFontException
	{
		super(buffer);
	}

	public void subsetAndStream(Subset subset, Map tables)
	{
		OTByteArrayBuilder newData = OTByteArray.getOTByteArrayBuilderInstance(data);
		tables.put(Tag.table_fond, newData);
	}

	public void stream(Map tables)
	{
		OTByteArrayBuilder newData = getDataAsByteArray();
		tables.put(Tag.table_fond, newData);
	}

	public OTByteArray getData()
	{
		return data;
	}

	public String getName()
		throws InvalidFontException
	{
		int script = data.getint16(fScript);
		int nameLen = data.getint8(fNameLength);
		byte[] nameBytes = data.getBytes(fName, nameLen);
		String charsetName = Name.getCharsetName(Name.PlatformId.MACINTOSH, script);
		if (charsetName == null) {
			charsetName = Name.getCharsetName(Name.PlatformId.MACINTOSH, 0);
		}
		try {
			return new String(nameBytes, charsetName);
		} catch (UnsupportedEncodingException e) {
			return null;
		}
	}

	public boolean isBold()
		throws InvalidFontException
	{
		int style = data.getint16(fStyle);
		return ((style & STYLE_BOLD) != 0);
	}

	public boolean isItalic()
		throws InvalidFontException
	{
		int style = data.getint16(fStyle);
		return ((style & STYLE_ITALIC) != 0);
	}

	public ULocale getLocale()
		throws InvalidFontException
	{
		int script = data.getint16(fScript);
		script = Math.min(script, SCRIPTID_TO_ULOCALE.length - 1);
		String locale = SCRIPTID_TO_ULOCALE[script];
		if (locale == null) {
			locale = SCRIPTID_TO_ULOCALE[0];
		}
		return new ULocale(locale);
	}
}
