/**
 * 
 */
package com.adobe.fontengine.fontmanagement;

import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.lang.ref.SoftReference;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.adobe.fontengine.font.FontData;
import com.adobe.fontengine.font.FontImpl;
import com.adobe.fontengine.font.FontLoadingException;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;

/**
 * A font that comes from a Mac resource file whether that be a resource fork or a data fork resource file.
 *
 */
public class ResourceFont extends FontImpl 
{
	static final long serialVersionUID = 1;

	protected final URL resourceURL;
	protected final int type;
	protected final int fontID;
	protected final int fondID;
	protected String canonicalPath;
	protected long length;
	protected long lastModified;
	protected ConcurrentHashMap descCache = new ConcurrentHashMap(16, 0.75f, 1);
	transient protected SoftReference fontRef;


	/**
	 * Constructor.
	 */
	public ResourceFont(URL resource, int type, int fontID, int fondID) 
	{
		this.resourceURL = resource;
		this.type = type;
		this.fontID = fontID;
		this.fondID = fondID;
		initCacheTags();
	}

	/**
	 * Constructor.
	 */
	public ResourceFont(URL resource, int type, int fontID, int fondID, FontData fontData) 
	{
		this(resource, type, fontID, fondID);
		this.fontRef = new SoftReference(fontData);
	}

	protected synchronized FontData retrieveFontData() 
	throws InvalidFontException, UnsupportedFontException, FontLoadingException
	{
		FontData font = null;
		if ((font = (FontData) fontRef.get()) == null)
		{
			font = FontLoader.fromResourceURL(this.resourceURL, this.type, this.fontID, this.fondID);
			fontRef = new SoftReference(font);
		}

		return font;
	}

	public int hashCode() 
	{
		final int prime = 31;
		int result = 1;
		result = prime * result + this.fontID;
		result = prime * result + this.fondID;
		result = prime * result	+ ((this.resourceURL == null) ? 0 : this.resourceURL.hashCode());
		return result;
	}

	public boolean equals(Object obj) 
	{
		if (this == obj)
		{
			return true;
		}
		if (!(obj instanceof ResourceFont))
		{
			return false;
		}
		final ResourceFont other = (ResourceFont) obj;
		if (this.fontID != other.fontID)
		{
			return false;
		}
		if (this.fondID != other.fondID)
		{
			return false;
		}
		if (resourceURL == null) {
			if (other.resourceURL != null)
			{
				return false;
			}
		} else if (!resourceURL.equals(other.resourceURL))
		{
			return false;
		}
		return true;
	}

	public String toString()
	{
		return new String(this.resourceURL.toString() + " - resource id = " + this.fontID + ", fond id = " + this.fondID);
	}

	private void readObject(ObjectInputStream s) 
	throws IOException, ClassNotFoundException
	{
		s.defaultReadObject();
		this.fontRef = new SoftReference(null);        
	}

	/* These are the cache support methods. ResourceFonts ARE supported for the
	 * cache so they should provide the necessary support to connect a ResourceFont
	 * to its underlying file and cached FontDescription objects.
	 */
	private void initCacheTags()
	{
		try {
			try {
				canonicalPath = resourceURL.toURI().getPath();
				if (canonicalPath == null) {
					throw new IOException("Cannot resolve path to file");
				}
				canonicalPath = canonicalPath.replace("/..namedfork/rsrc", "");
			} catch (URISyntaxException e) {
				throw new IOException(e.toString());
			}
			File file = new File(canonicalPath);
			canonicalPath = file.getCanonicalPath();
			length = file.length();
			lastModified = file.lastModified();
		} catch (IOException e) {
			canonicalPath = null;
			length = 0;
			lastModified = 0;
		}
	}

	public String getCanonicalPath()
	{
		return canonicalPath;
	}

	public long getLength()
	{
		return length;
	}

	public long getLastModified()
	{
		return lastModified;
	}

	public Object getCachedFontDescription(String key)
	{
		return descCache.get(key);
	}

	public Map<String, Object> getCachedFontDescriptionMap()
	{
		return descCache;
	}

	public void setCachedFontDescription(String key, Object value)
	{
		descCache.put(key, value);
	}
}
