/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.search.eval;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.UnsupportedRepositoryOperationException;
import javax.jcr.query.Row;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.Privilege;

import org.apache.felix.scr.annotations.Component;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.day.cq.search.Predicate;

/**
 * Restricts the result to items where the current session has the specified
 * <a href="https://docs.adobe.com/content/docs/en/spec/jcr/2.0/16_Access_Control_Management.html#16.2.3%20Standard%20Privileges">
 * JCR privileges</a>.
 *
 * <p>
 * This is a filtering-only predicate and cannot leverage a search index.
 * Does not support facet extraction.
 *
 * <h3>Name:</h3>
 * hasPermission
 *
 * <p/>
 * <h3>Properties:</h3>
 * <dl>
 * <dt>hasPermission</dt>
 * <dd>comma-separated JCR privileges that the current user session must ALL have
 * for the node in question; for example <code>jcr:write,jcr:modifyAccessControl</code></dd>
 * </dl>
 * 
 * @since 6.1
 */
@Component(metatype = false, factory = "com.day.cq.search.eval.PredicateEvaluator/hasPermission")
public class PermissionPredicateEvaluator extends AbstractPredicateEvaluator {
    public static final String HAS_PERMISSION_PREDICATE = "hasPermission";

    private static final Logger log = LoggerFactory.getLogger(PermissionPredicateEvaluator.class);

    @Override
    public boolean includes(Predicate p, Row row, EvaluationContext context) {
        if (!p.hasNonEmptyValue(HAS_PERMISSION_PREDICATE)) {
            return true;
        }

        try {
            ResourceResolver resolver = context.getResourceResolver();
            Session session = resolver.adaptTo(Session.class);
            AccessControlManager acm = session.getAccessControlManager();
            final Resource resource = context.getResource(row);
            String permissions[] = p.get(HAS_PERMISSION_PREDICATE).split(",");
            Privilege privileges[] = new Privilege[permissions.length];
            for (int i = 0; i < permissions.length; i++) {
                privileges[i] = acm.privilegeFromName(permissions[i]);
            }
            return acm.hasPrivileges(resource.getPath(), privileges);

        } catch (UnsupportedRepositoryOperationException e) {
            log.error("Exception occured while evaluating "
                + HAS_PERMISSION_PREDICATE + " predicate", e);
        } catch (RepositoryException e) {
            log.error("Exception occured while evaluating "
                + HAS_PERMISSION_PREDICATE + " predicate", e);
        }
        return false;
    }

    @Override
    public boolean canXpath(Predicate predicate, EvaluationContext context) {
        return false;
    }

    @Override
    public boolean canFilter(Predicate predicate, EvaluationContext context) {
        return true;
    }

}
