/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.granite.rest.utils;

import java.util.AbstractMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.sling.api.resource.ValueMap;
import org.apache.sling.api.wrappers.ModifiableValueMapDecorator;

/**
 * A value map wrapper which implements mapping of certain property names.
 * <p>
 * By default the following properties are mapped.
 * <ul>
 *   <li>jcr:title to dc:title</li>
 *   <li>jcr:description to dc:description</li>
 *   <li>jcr:language to dc:language</li>
 * </ul>
 */
public class ModifiableMappedValueMapDecorator extends ModifiableValueMapDecorator {

    /**
     * Creates a new wrapper around a given map.
     * 
     * @param base Wrapped object
     */
    public ModifiableMappedValueMapDecorator(Map<String,Object> base) {
        super(base);
    }

    /**
     * Creates a new wrapper around a given map.
     * 
     * @param base Wrapped object
     */
    public ModifiableMappedValueMapDecorator(ValueMap base) {
        super(base);
    }

    // ---------- Map

    /**
     * {@inheritDoc}
     */
    public Object get(Object key) {
        return super.get(unmapProperty((String)key));
    }

    /**
     * {@inheritDoc}
     */
    public Object put(String key, Object value) {
        return super.put(unmapProperty(key), value);
    }

    /**
     * {@inheritDoc}
     */
    public void putAll(Map<? extends String, ?> t) {
        for (String key : t.keySet()) {
            put(key, t.get(key));
        }
    }

    /**
     * {@inheritDoc}
     */
    public boolean containsKey(Object key) {
        return super.containsKey(unmapProperty((String)key));
    }

    /**
     * {@inheritDoc}
     */
    public Set<String> keySet() {
        Set<String> mapped = new HashSet<String>();
        for (String key : super.keySet()) {
            mapped.add(mapProperty(key));
        }
        return mapped;
    }

    /**
     * {@inheritDoc}
     */
    public Set<Entry<String, Object>> entrySet() {
        Set<Entry<String,Object>> set = new HashSet<Entry<String,Object>>();
        for (Entry<String,Object> e : super.entrySet()) {
            set.add(new AbstractMap.SimpleEntry<String,Object>(mapProperty((String)e.getKey()), e.getValue()));
        }
        return set;
    }

    /**
     * Returns the unmapped property name for the provided {@code unmappedName}.
     * The implementation of this method maps the properties
     * {@code jcr:title,jcr:description,jcr:language} to
     * {@code dc:title,dc:description,dc:language}.
     * 
     * @param unmappedName Unmapped property name
     * @return Mapped property name
     */
    public static String mapProperty(String unmappedName) {
        if (unmappedName.matches("^jcr:(title|description|language)$")) {
            return unmappedName.replaceFirst("jcr:", "dc:");
        }
        return unmappedName;
    }

    /**
     * Returns the unmapped property name for the provided {@code mappedName}.
     * The implementation of this method maps the properties
     * {@code dc:title,dc:description,dc:language} to
     * {@code jcr:title,jcr:description,jcr:language}.
     * 
     * @param mappedName Mapped property name
     * @return Unmapped property name
     */
    public static String unmapProperty(String mappedName) {
        if (mappedName.matches("^dc:(title|description|language)$")) {
            return mappedName.replaceFirst("dc:", "jcr:");
        }
        return mappedName;
    }

}
