/* ****************************************************************************
 *
 *	File: ASDate.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2003-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.types;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.SimpleTimeZone;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.adobe.internal.io.stream.OutputByteStream;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFParseException;

/**
 * Represents a date which is a specific point in time.  It provides for input and output
 * of that date in ASN.1 format that is used by PDF.
 */
public class ASDate extends ASObject implements Comparable
{
	// Class Static
	private static final DecimalFormat yearFormat;
	private static final DecimalFormat dateFormat;
	// List of fallback dateparsers to accommodate dates which do not conform to PDF spec
	private static ArrayList<NonConformingDateParser> fallbackDateParsers;
	static {
		DecimalFormatSymbols dfs = new DecimalFormatSymbols(Locale.US);
		dfs.setZeroDigit('0');
		dfs.setDecimalSeparator('.');
		dfs.setMinusSign('-');
		yearFormat = new DecimalFormat("####", dfs);
		yearFormat.setMinimumIntegerDigits(4);
		yearFormat.setGroupingUsed(false);

		dateFormat = new DecimalFormat("#", dfs);
		dateFormat.setMinimumIntegerDigits(2);
		dateFormat.setGroupingUsed(false);
		
		fallbackDateParsers = new ArrayList<NonConformingDateParser>();
		fallbackDateParsers.add(DateParserNoTimeZone.getInstance());
	}
	// Date format string for parsing
	// optional "D:" + Year + optional Month + optional Day + optional Hour + optional Minute + optional Second
	// + optional time zone ( [+-Z] + optional hour + optional minute) 
	// + require that there is nothing beyond the end of the pattern
	private static final String datePatternString = 
		"^(D:)?(\\d{4})(\\d{2})?(\\d{2})?(\\d{2})?(\\d{2})?(\\d{2})?(?:([\\+-Z])(?:(\\d{2})')?(?:(\\d{2})')?)?\\z";
	private static final Pattern datePattern = Pattern.compile(datePatternString);

	// Instance Variables
	private GregorianCalendar calendar;
	private ASString dateString;
	private boolean dateStringValid = true;
	private boolean timeZoneSet = true;

	/**
	 * Create an ASDate that references this moment in time.
	 */
	public ASDate()
	{
		this.calendar = (GregorianCalendar) Calendar.getInstance(Locale.US);
		this.calendar.setGregorianChange(new Date(Long.MIN_VALUE));
		this.calendar.setTimeZone(TimeZone.getDefault());
		// can't represent milliseconds - get rid of them now to make comparisons work
		this.calendar.set(Calendar.MILLISECOND, 0);
	}

	/**
	 * Create an ASDate that points to the moment in time given by the <code>Date</code>.
	 */
	public ASDate(Date date)
	{
		this();
		this.calendar.setTime(date);
		// can't represent milliseconds - get rid of them now to make comparisons work
		this.calendar.set(Calendar.MILLISECOND, 0);
	}

	/**
	 * Create an ASDate that points to the moment in time given by the <code>Date</code>.
	 */
	public ASDate(Date date, TimeZone zone)
	{
		this();
		this.calendar.setTimeZone(zone);
		this.calendar.setTime(date);
		// can't represent milliseconds - get rid of them now to make comparisons work
		this.calendar.set(Calendar.MILLISECOND, 0);		
	}

	/**
	 * Create an ASDate that point to the moment in time given by the String representation
	 * using the format given in the PDF 1.6 spec on page 133.
	 */
	public ASDate(String dateString)
	throws PDFParseException
	{
		this(new ASString(dateString));
	}

	/**
	 * Create an ASDate that point to the moment in time given by the String representation
	 * using the format given in the PDF 1.6 spec on page 133.
	 */
	public ASDate(ASString dateString)
	throws PDFParseException
	{
		this();
		String s = dateString.toString();		
		try {
			parse(s);
		} catch (PDFParseException e) {
			this.dateStringValid = false;
			GregorianCalendar fallbackCalendar = null;
			for(NonConformingDateParser dateParser : fallbackDateParsers)
			{
				fallbackCalendar = dateParser.parse(calendar, s);
				if(fallbackCalendar != null)
				{
					this.calendar = fallbackCalendar;
					this.dateStringValid = true;
					this.timeZoneSet = dateParser.hasTimeZone(fallbackCalendar, s);
					break;
				}					
			}
			
		}
		this.dateString = dateString;
	}

	/**
	 * If the <code>ASDate</code> object was instantiated with a string then it may be invalid and not
	 * represent a correct date as per the PDF spec.  If the date is not valid then there is no way to
	 * convert the internal representation to a real date or timezone.
	 * @return <code>true</code> if the date is valid; <code>false</code> otherwise
	 */
	public boolean isDateValid()
	{
		return this.dateStringValid;
	}
	
	/**
	 * Compares two Dates for ordering.
	 *
	 * @param   obj   the <code>ASDate</code> to be compared.
	 * @return  the value <code>0</code> if the argument ASDate is equal to
	 *          this ASDate; a value less than <code>0</code> if this ASDate
	 *          is before the ASDate argument; and a value greater than
	 *      <code>0</code> if this ASDate is after the ASDate argument.
	 */
	public int compareTo(Object obj) 
	{
		return compareTo((ASDate) obj);
	}

	/**
	 * Compares two Dates for ordering.
	 *
	 * @param   otherASDate   the <code>ASDate</code> to be compared.
	 * @return  the value <code>0</code> if the argument ASDate is equal to
	 *          this ASDate; a value less than <code>0</code> if this ASDate
	 *          is before the ASDate argument; and a value greater than
	 *      <code>0</code> if this ASDate is after the ASDate argument.  Invalid
	 *      dates are considered to be before other dates.
	 */
	public int compareTo(ASDate otherASDate) 
	{
		Date date = this.toDate();
		Date otherDate = otherASDate.toDate();
		if (date == null)
		{
			if (otherDate == null)
			{
				// both null
				return 0;
			}
			// only this one null
			return -1;
		} else if (otherDate == null)
		{
			// only other one null
			return 1;
		}
		return date.compareTo(otherDate);
	}
	
	/**
	 * Tests if this date is before the specified date.
	 * @param otherDate an ASDate
	 * @return <code>true</code> if and only if the instant represented by this <code>ASDate</code>
	 * is strictly earlier than the instant represented by <code>otherDate</code>; 
	 * <code>false</code> otherwise
	 */
	public boolean before(ASDate otherDate) 
	{
		return this.compareTo(otherDate) < 0;
	}
	
	/**
	 * Tests if this date is after the specified date.
	 * @param otherDate an ASDate
	 * @return <code>true</code> if and only if the instant represented by this <code>ASDate</code>
	 * is strictly later than the instant represented by <code>otherDate</code>; 
	 * <code>false</code> otherwise
	 */
	public boolean after(ASDate otherDate) 
	{
		return this.compareTo(otherDate) > 0;
	}
	
	/* (non-Javadoc)
	 * @see java.lang.Object#equals(java.lang.Object)
	 */
	@Override
	public boolean equals(Object obj)
	{
		if (obj == null)
		{
			return false;
		}
		if (!(obj instanceof ASDate))
		{
			return false;
		}
		
		return this.compareTo(obj) == 0;
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#hashCode()
	 */
	@Override
	public int hashCode()
	{
		return this.toDate().hashCode();
	}

	/**
	 * Generates a String for output to the serialized PDF.
	 */
	public String asString()
	{
		return this.getDateString().asString();
	}

	/**
	 * Generates a String for debugging, messages, etc.
	 */
	@Override
	public String toString()
	{
		return this.getDateString().toString();
	}

	/**
	 * Generates a String for LiveCycle
	 */
	private static final String datePatternStringNoTZ = "^(D:)?(\\d{4})(\\d{2})?(\\d{2})?(\\d{2})?(\\d{2})?(\\d{2})?";
	private static final Pattern datePatternNoTZ = Pattern.compile(datePatternStringNoTZ);

	public String toStringTimezoneUnaware()
	{
		String orig = getDateString().toString();
		if (orig == null)
			return null;
		Matcher matcher = datePatternNoTZ.matcher(orig);
		if (!matcher.find())
			return null;
		int year = Integer.parseInt(matcher.group(2));
		int month = matcher.group(3) == null ? 1 : Integer.parseInt(matcher.group(3));
		int date = matcher.group(4) == null ? 1 : Integer.parseInt(matcher.group(4));
		int hour = matcher.group(5) == null ? 0 : Integer.parseInt(matcher.group(5));
		int minute = matcher.group(6) == null ? 0 : Integer.parseInt(matcher.group(6));
		int second = matcher.group(7) == null ? 0 : Integer.parseInt(matcher.group(7));
		GregorianCalendar calNoTZ = (GregorianCalendar)Calendar.getInstance(Locale.US);
		calNoTZ.set(year, month - 1, date, hour, minute, second);
		StringBuilder result = new StringBuilder("D:");
		result.append(yearFormat.format(calNoTZ.get(Calendar.YEAR)));
		result.append(dateFormat.format(calNoTZ.get(Calendar.MONTH) + 1));
		result.append(dateFormat.format(calNoTZ.get(Calendar.DAY_OF_MONTH)));
		result.append(dateFormat.format(calNoTZ.get(Calendar.HOUR_OF_DAY)));
		result.append(dateFormat.format(calNoTZ.get(Calendar.MINUTE)));
		result.append(dateFormat.format(calNoTZ.get(Calendar.SECOND)));
		return result.toString();
	}

	public ASDate(String dateString, TimeZone zone)
		throws PDFParseException
	{
		this(dateString);
		this.calendar.setTimeZone(zone);
	}

	/**
	 * Return the <code>Date</code> represented by this ASDate if the internal representation is valid.
	 * @return the date of the this object
	 */
	public Date toDate()
	{
		if (!dateStringValid)
		{
			return null;
		}
		return this.calendar.getTime();
	}

	/**
	 * Returns if timezone is set explicitly set for this date
	 * @return the timezone of this object
	 */
	public boolean hasTimeZone()
	{
		return timeZoneSet;
	}
	
	/**
	 * Return the <code>TimeZone</code> represented by this ASDate if the internal representation is valid.
	 * @return the timezone of this object
	 */
	public TimeZone getTimeZone()
	{
		if (!dateStringValid)
		{
			return null;
		}
		return this.calendar.getTimeZone();
	}		

	/**
	 * Parse the specified byte array representing a PDF Date to set our
	 * internal fields.
	 *
	 * @param s	a PDF Date string
	 * @throws PDFParseException
	 */
	private void parse(String s) 
	throws PDFParseException
	{
		Matcher matcher = datePattern.matcher(s);

		if (!matcher.find())
		{
			// regex wasn't matched so the date isn't valid
			throw new PDFParseException("Invalid date format - " + s);
		}
		int year = Integer.parseInt(matcher.group(2));
		int month = matcher.group(3) == null ? 1 : Integer.parseInt(matcher.group(3));
		int date = matcher.group(4) == null ? 1 : Integer.parseInt(matcher.group(4));
		int hour = matcher.group(5) == null ? 0 : Integer.parseInt(matcher.group(5));
		int minute = matcher.group(6) == null ? 0 : Integer.parseInt(matcher.group(6));
		int second = matcher.group(7) == null ? 0 : Integer.parseInt(matcher.group(7));
		boolean hasGMT = matcher.group(8) != null;
		int gmtSign = 0;
		int offsetGMTHour = 0;
		int offsetGMTMinute = 0;
		if (hasGMT)
		{
			gmtSign = matcher.group(8).equals("-") ? -1 : +1;
			offsetGMTHour = matcher.group(9) == null ? 0 : Integer.parseInt(matcher.group(9));
			offsetGMTMinute = matcher.group(10) == null ? 0 : Integer.parseInt(matcher.group(10));			
		}

		// convert all into Java Calendar
		this.calendar.clear();
		if (hasGMT)
		{
			TimeZone timeZone = TimeZone.getTimeZone("GMT");
			timeZone.setRawOffset( gmtSign * ((offsetGMTHour * 60) + offsetGMTMinute) * 60 * 1000);
			this.calendar.setTimeZone(timeZone);
		}
		this.timeZoneSet = hasGMT;
		this.calendar.set(year, month - 1, date, hour, minute, second);	
	}

	/**
	 * Returns the date string if it exists and if not creates it, caches it, and then
	 * returns it.
	 * @return the date string
	 */
	private ASString getDateString()
	{
		if (this.dateString == null)
		{
			this.dateString = new ASString(makeString());
		}
		return this.dateString;
	}

	private String makeString() 
	{
		StringBuilder date = new StringBuilder("D:");

		date.append(yearFormat.format(this.calendar.get(Calendar.YEAR)));
		date.append(dateFormat.format(this.calendar.get(Calendar.MONTH) + 1));
		date.append(dateFormat.format(this.calendar.get(Calendar.DAY_OF_MONTH)));
		date.append(dateFormat.format(this.calendar.get(Calendar.HOUR_OF_DAY)));
		date.append(dateFormat.format(this.calendar.get(Calendar.MINUTE)));
		date.append(dateFormat.format(this.calendar.get(Calendar.SECOND)));

		TimeZone zone = this.calendar.getTimeZone();
		int gmtOffsetMilli = zone.getOffset(this.calendar.getTime().getTime());
		int gmtOffsetHour = gmtOffsetMilli / 3600000;
		int gmtOffsetMinute = Math.abs((gmtOffsetMilli % 3600000) / 60000);

		if ((gmtOffsetHour == 0) && (gmtOffsetMinute == 0)) 
		{
			date.append('Z');
		} else {
			if (gmtOffsetHour < 0) 
			{
				date.append('-');
				date.append(dateFormat.format(-gmtOffsetHour));
			} else {
				date.append('+');
				date.append(dateFormat.format(gmtOffsetHour));
			}
			date.append('\'');
			date.append(dateFormat.format(gmtOffsetMinute));
			date.append('\'');
		}
		return date.toString();
	}

	/**
	 * Writes the ADate to the given OutputByteStream in the format expected by the PDF Spec.
	 * @see ASString
	 * @param outputByteStream OutputByteStream to write to.
	 * @throws PDFIOException
	 */
	@Override
	public void write(OutputByteStream outputByteStream)
	throws PDFIOException
	{
		this.getDateString().write(outputByteStream);
	}	
}

/**
 * Defines how dates not conforming to PDF spec are parsed
 * @author hraghav
 */
interface NonConformingDateParser
{
	/**
	 * Parses string date and returns updated calendar. 
	 * @param calendar
	 * @param date
	 * @return GregorianCalendar if date was parsed, else null.
	 */
	GregorianCalendar parse(GregorianCalendar calendar, String date);
	
	/**
	 * @param calendar
	 * @param date
	 * @return true if this date has time zone information else false
	 */
	boolean hasTimeZone(GregorianCalendar calendar, String date);
}

/**
 * Parses dates of format which do not have time zone offsets, but contain Z followed by
 * apostrophe ex: (D:20120510120515Z').
 * @author hraghav
 */
class DateParserNoTimeZone implements NonConformingDateParser 
{
	private static final String datePatternStringNoTimeZone = "^(D:)?(\\d{4})(\\d{2})?(\\d{2})?(\\d{2})?(\\d{2})?(\\d{2})?(?:([\\+-Z])')\\z";
	private static final Pattern datePatternCompiled = Pattern.compile(datePatternStringNoTimeZone);
	private static DateParserNoTimeZone _instance = new DateParserNoTimeZone();
	private static TimeZone tz = new SimpleTimeZone(0, "");
	
	/**
	 * 
	 */
	private DateParserNoTimeZone()
	{
		// private construction to prevent instantiation.
	}
	
	/**
	 * GetInstance method to return singleton instance of this class
	 * @return DateParserNoTimeZone
	 */
	static DateParserNoTimeZone getInstance()
	{
		return _instance;
	}
	
	
	public GregorianCalendar parse(GregorianCalendar calendar, String s)
	{
		Matcher matcher = datePatternCompiled.matcher(s);

		if (!matcher.find())
		{
			// regex wasn't matched so the date isn't valid
			return null;
		}
		int year = Integer.parseInt(matcher.group(2));
		int month = matcher.group(3) == null ? 1 : Integer.parseInt(matcher.group(3));
		int date = matcher.group(4) == null ? 1 : Integer.parseInt(matcher.group(4));
		int hour = matcher.group(5) == null ? 0 : Integer.parseInt(matcher.group(5));
		int minute = matcher.group(6) == null ? 0 : Integer.parseInt(matcher.group(6));
		int second = matcher.group(7) == null ? 0 : Integer.parseInt(matcher.group(7));
	
		// convert all into Java Calendar
		calendar.clear();
		
		calendar.set(year, month - 1, date, hour, minute, second);
		calendar.setTimeZone(tz);
		
		return calendar;	
	}

	public boolean hasTimeZone(GregorianCalendar calendar, String date) 
	{
		return false;
	}
	
}