/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2017 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ************************************************************************/

package com.day.cq.dam.scene7.api.model;

import java.util.LinkedList;
import java.util.List;

import org.apache.commons.lang.StringUtils;

public class Scene7AssetSet {

    private final List<Scene7AssetSetMember> members;

    public Scene7AssetSet(List<Scene7AssetSetMember> members) {
        this.members = members != null ? members : new LinkedList<Scene7AssetSetMember>();
    }

    /**
     * Return the list of asset set members
     * 
     * @return List of <code>Scene7AssetSetMember</code> objects
     */
    public List<Scene7AssetSetMember> getAssetSetMembers() {
        return members;
    }

    /**
     * Return the proper representation for the asset set
     *
     * Examples:
     *
     * 1-row SpinSet2d:
     *      ${getCatalogId([a|48358052])},${getCatalogId([a|48358054])},${getCatalogId([a|48358053])}
     *
     * 3-row SpinSet2d:
     *      {${getCatalogId([a|4540947])},${getCatalogId([a|4540948])},${getCatalogId([a|4540949])}},{${getCatalogId([a|4561615])},${getCatalogId([a|4540949])}},{${getCatalogId([a|4541014])},${getCatalogId([a|4540948])}}
     *
     * MediaSet:
     *      ${getCatalogId([a|36594182])};${getCatalogId([a|36594182])};advanced_image;,${getCatalogId([a|40524221])};${getCatalogId([a|40524221])};spin;,${getCatalogId([a|424372576])};${getCatalogId([a|424372576])};advanced_imageset;
     *
     * @return String
     */
    @Override
    public String toString() {
        if (members.isEmpty()) {
            return "";
        }
        StringBuilder sb = new StringBuilder(members.size() * 30);

        for (Scene7AssetSetMember member : members) {
            if (sb.length() > 0) {
                sb.append(',');
            }
            if (Scene7AssetSetMember.Type.SPIN_SET_ROW.equals(member.getType())) {
                if (members.size() > 1) {  // multirow SpinSet2d
                    sb.append('{');
                }
                boolean first = true;
                for (String h : member.getHandles()) {
                    if (!first) {
                        sb.append(',');
                    }
                    sb.append("${getCatalogId([").append(h).append("])}");
                    first = false;
                }
                if (members.size() > 1) {  // multirow SpinSet2d
                    sb.append('}');
                }
            } else {

                for (String h : member.getHandles()) {
                    sb.append("${getCatalogId([").append(h).append("])};");
                }

                if (member.getType() != null) {
                    String preset = member.getType().toString();
                    if (StringUtils.isNotEmpty(preset)) {
                        sb.append(preset).append(';');
                    }
                }
            }
        }

        return sb.toString();
    }
}
