/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/

package com.day.cq.replication;

import static com.day.cq.replication.ReplicationResultInfo.NONE;

/**
 * The <code>ReplicationResult</code> is returned by the {@link TransportHandler}
 * after sending a request to the recipient. The instance conveys information
 * on the success of request delivery and request processing. Optionally it also adds
 * additional information for the result through {@link ReplicationResultInfo}
 */
public class ReplicationResult {

    /**
     * OK result.
     */
    public static final ReplicationResult OK = new ReplicationResult(true, 200, "OK");

    /**
     * Flag indicating whether the replication was successful.
     */
    private final boolean success;

    /**
     * Status code.
     */
    private final int code;

    /**
     * Status message.
     */
    private final String message;
    
    private final ReplicationResultInfo resultInfo;

    /**
     * Creates an instance of this class.
     *
     * @param success See {@link #isSuccess()}
     * @param code See {@link #getCode()}
     * @param message See {@link #getMessage()}
     * @param resultInfo See {@link #getReplicationResultInfo()}
     */
    public ReplicationResult(boolean success, int code, String message, ReplicationResultInfo resultInfo) {
        this.success = success;
        this.code = code;
        this.message = message;
        if (resultInfo == null) {
            resultInfo = NONE;
        }
        this.resultInfo = resultInfo;
    }
    
    /**
     * Creates an instance of this class.
     *
     * @param success See {@link #isSuccess()}
     * @param code See {@link #getCode()}
     * @param message See {@link #getMessage()}
     */
    public ReplicationResult(boolean success, int code, String message) {
        this.success = success;
        this.code = code;
        this.message = message;
        this.resultInfo = NONE;
    }

    /**
     * Returns <code>true</code> If the replication action and optional content
     * could be sent to the recipient. <code>false</code> is returned if there
     * was a transfer problem.
     * <p>
     * Note that this flag indicates whether the recipient received the
     * request for further processing. It does <b>not</b> indicate whether the
     * recipient could do something useful with it.
     * 
     * @return {@code true} If the replication action and optional content
     * could be sent to the recipient. {@code false} is returned if there
     * was a transfer problem.
     */
    public boolean isSuccess() {
        return success;
    }

    /**
     * Returns some status code indication sent by the recipient about the
     * success of processing the replication action and optional content.
     * <p>
     * For example an HTTP based transport agent may set this field to the HTTP
     * response status code.
     * 
     * @return some status code indication sent by the recipient about the
     * success of processing the replication action and optional content.
     */
    public int getCode() {
        return code;
    }

    /**
     * Returns some status message providing clear text indication sent by the
     * recipient about the success of processing the replication action and
     * optional content.
     * <p>
     * For example an HTTP based transport agent may set this field to the HTTP
     * response status message.
     * 
     * @return some status message providing clear text indication sent by the
     * recipient about the success of processing the replication action and
     * optional content.
     */
    public String getMessage() {
        return message;
    }

    /**
     * Returns any additional information for replication action
     * 
     * @return additional info
     */
    public ReplicationResultInfo getReplicationResultInfo() {
        return resultInfo;
    }

    @Override
    public String toString() {
        return "ReplicationResult{" +
            "success=" + isSuccess() +
            ", code=" + getCode() +
            ", message=" + getMessage() +
            ", info={id=" + resultInfo.getArtifactId() + "}" +
            '}';
    }
}
