package com.adobe.xfa.text;

import com.adobe.xfa.gfx.GFXGlyphOrientation;
import com.adobe.xfa.ut.CoordPair;
import com.adobe.xfa.ut.UniCharIterator;

/**
 * The run is the basic building block of text layout.
 * <p>
 * A run is sequence of glyphs that share common attributes and don't
 * require any special inter-glyph positioning.  A run can be either
 * horizontal or vertical.	Horizontal runs are used for Latin,
 * Middle-Eastern and certain Asian scripts.  The implicit rendering
 * order of a horizontal run is always left-to-right (LTR), even it
 * contains glyphs based on right-to-left (RTL) characters.  Vertical
 * runs are used for CJK text in vertical format.
 * </p>
 * @exclude from published api.
 */
public class TextGlyphRun {
	private TextAttr mpoAttr;
	private CoordPair moShift;
	private CoordPair moPosition;
	private final TextIntArray moGlyphs = new TextIntArray();
	private boolean mbIsCharRun;
	private boolean mbIsRTL;
	private int meGlyphOrientation = GFXGlyphOrientation.HORIZONTAL;

/**
 * Default constructor.
 * <p>
 * The run is initially empty of glyphs and has a null attribute
 * pointer.  Its position and shift are both set to (0,0).
 */
	public TextGlyphRun () {
	}

/**
 * Copy constructor.
 * <p>
 * Create a new glyph run, copying the contents of the given source run.
 * Note that copying the run's attributes means adding a reference to
 * the shared attribute object.
 * @param oSource - Source run object to copy.
 */
	public TextGlyphRun (TextGlyphRun oSource) {
		copyFrom (oSource);
	}

/**
/**
 * Set the attributes to be used for this run.
 * <p>
 * All text attributes can be specified through a single TextAttr
 * object.	This object represents a possibly sparse collection of
 * attributes.	Please see the description of the TextAttr class for
 * more information.
 * @param oAttr - Attribute set to use for this run.  Any omitted
 * attributes will be inherited from the previous run.	If attributes
 * are missing on the first run, TextAttr defaults will be used.	The
 * run will cache a copy of the attribute object; the caller can delete
 * or modify the original after the method returns.
 */
	public void setAttr (TextAttr oAttr) {
		mpoAttr = new TextAttr (oAttr);
	}

/**
 * Set the attributes to be used for this run, via a reference-counted
 * attribute object.
 * <p>
 * Please see method SetAttr() for more information on attributes.
 * @param poAttr - Attribute set to use for this run.  Unlike method
 * SetAttr(), this method does not make a copy.  Instead, the given
 * attribute object is assumed to be reference counted.  After the call,
 * the run will have added its own reference to the object.  This allows
 * the caller to create a common attribute instance to be shared by
 * several runs.
 */
	public void setAttrRef (TextAttr poAttr) {
		mpoAttr = poAttr;
	}

/**
 * Return the text attributes that apply to this run.
 * @return Text attributes for the run; null if attributes have never
 * been set.  Note that the return value uses the text attribute
 * reference counting model.  If the client chooses to hang on to the
 * return pointer, it must add a reference.
 */
	public TextAttr getAttr () {
		return mpoAttr;
	}

/**
 * Specify a shift for the run.
 * <p>
 * A shift occurs if the run doesnt start at the end point of the
 * previous run.  This could be the result of either a baseline shift, a
 * horizontal shift such as kerning adjustment, or both.  There is no
 * need to set the shift for the first run of a line if that run starts
 * at the left endpoint of the line's baseline.
 * </p>
 * <p>
 * A horizontal shift implicitly cascades through subsequent runs.	In
 * other words, if a run is shifted left due to kerning, the next run
 * would normally start at the end point of the shifted run.  There
 * would be no need to explicitly declare a shift on subsequent runs.
 * Indeed, if the next run did declare a shift, the shift would
 * compound.
 * </p>
 * <p>
 * On the other hand, vertical shifts must be re-asserted on each run.
 * If a run is shifted away from its baseline, the next run will revert
 * to the baseline if it doesn't declare its own vertical shift.
 * </p>
 * @param oShift - Shift to apply to this run.
 */
	public void setShift (CoordPair oShift) {
		moShift = oShift;
	}

/**
 * Obtain the run's shift.
 * @return Shift that applies to this run.
 */
	public CoordPair getShift () {
		return (moShift == null) ? CoordPair.zeroZero() : moShift;
	}

/**
 * Convenience method to query whether the run has any shift.
 * @return True if the run has an X and/or Y shift; false if no shift.
 */
	public boolean isShifted () {
		return (moShift != null)
			&& ((moShift.x().value() != 0) || (moShift.y().value() != 0));
	}

/**
 * Obtain the runs position.
 * <p>
 * This call makes sense only after formatting.  It returns the
 * position--relative to the left endpoint of the lines baseline--of the
 * run.
 * @return Run's position.
 */
	public CoordPair getPosition () {
		return (moPosition == null) ? CoordPair.zeroZero() : moPosition;
	}

/**
 * Add a single glyph to the run.
 * @param nAdd - Glyph ID to add to the run.
 */
	public void addGlyph (int nAdd) {
		moGlyphs.add (nAdd);
	}

	public void addCharGlyphs (String sCharGlyphs) {
		UniCharIterator iter = new UniCharIterator (sCharGlyphs);
		for (int c = iter.next(); c != 0; c = iter.next()) {
			addGlyph (c);
		}
	}

/**
 * Return the number of glyphs currently in the run.
 * @return Number of glyphs currently in the run.
 */
	public int getGlyphCount () {
		return moGlyphs.getSize();
	}

/**
 * Extract one glyph from the run, by index.
 * @param nIndex - Index of the desired glyph.	Index numbers start at
 * zero.  Unpredictable results will occur if the index is out of range.
 * @return Glyph ID at the specified index.
 */
	public int getGlyph (int nIndex) {
		return moGlyphs.get (nIndex);
	}

/**
 * Tells the glyph run whether its 32-bit values represent Unicode
 * characters or glyph IDs.
 * @param bCharRun - True if this run represents Unicode characters;
 * false if it represents glyph IDs.  If the client never sets this
 * flag, it defaults to false.
 */
	public void setCharRun (boolean bCharRun) {
		mbIsCharRun = bCharRun;
	}

/**
 * Query whether this run represents Unicode characters or glyph IDs.
 * @return True if the run represents Unicode characters; false if it
 * represents glyph IDs.
 */
	public boolean isCharRun () {
		return mbIsCharRun;
	}

/**
 * Tells the glyph run whether it represents right-to-left (RTL) glyphs.
 * <p>
 * AXTE needs to know glyph direction when resolving ambiguous caret
 * positions in bidirectional text.  Unfortunately it cannot glean such
 * direction from the mappings alone; the application must provide glyph
 * direction independently of the mappings.
 * @param bRTL - True if this run represents RTL glyphs; false if it
 * represents left-to-right (LTR) glyphs.  If the client never sets this
 * flag, it defaults to false.
 */
	public void setRTL (boolean bRTL) {
		mbIsRTL = bRTL;
	}

/**
 * Query whether this run represents RTL glyphs.
 * @return True if the run represents RTL glyphs; false if it represents
 * LTR glyphs.
 */
	public boolean isRTL () {
		return mbIsRTL;
	}

/**
 * Set the glyph orientation for this run.
 * @param eGlyphOrientation - New glyph orientation to use.
 */
	public void setGlyphOrientation (int eGlyphOrientation) {
		meGlyphOrientation = eGlyphOrientation;
	}

/**
 * Query the glyph orientation for this run.
 * @return Glyph orientation currently in effect.
 */
	public int getGlyphOrientation () {
		return meGlyphOrientation;
	}

/**
 * Assignment operator.
 * <p>
 * Copies all contents of the given source run.  Note that copying the
 * run's attributes means adding a reference to the shared attribute
 * object.
 * @param oSource - Source run object to copy.
 */
	public void copyFrom (TextGlyphRun oSource) {
		if (this != oSource) {
			moShift = oSource.moShift;
			moPosition = oSource.moPosition;
			moGlyphs.copyFrom (oSource.moGlyphs);
			mbIsCharRun = oSource.mbIsCharRun;
			mbIsRTL = oSource.mbIsRTL;
			meGlyphOrientation = oSource.meGlyphOrientation;
			mpoAttr = oSource.mpoAttr;
		}
	}

	void setPosition (CoordPair oPosition) {
		moPosition = oPosition;
	}
}
