/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 * Copyright 2019 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.cq.dam.commons.util;


import org.apache.jackrabbit.api.security.JackrabbitAccessControlEntry;
import org.apache.jackrabbit.api.security.JackrabbitAccessControlList;
import org.apache.jackrabbit.commons.jackrabbit.authorization.AccessControlUtils;

import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.security.AccessControlEntry;
import javax.jcr.security.AccessControlList;
import javax.jcr.security.AccessControlManager;
import javax.jcr.security.AccessControlPolicy;
import javax.jcr.security.Privilege;

public final class PrivateFolderAndCollectionUtil {

    public static AccessControlEntry getDenyAllEveryoneACE(Session session, String path) throws RepositoryException {
        AccessControlManager acm = session.getAccessControlManager();
        for (AccessControlPolicy policy : acm.getPolicies(path)) {
            if (policy instanceof AccessControlList) {
                AccessControlList accessControlList = (AccessControlList) policy;
                AccessControlEntry[] entries = accessControlList.getAccessControlEntries();
                AccessControlEntry denyAllACE = getDenyAllEveryoneACEFromAccessControlEntries(session, entries);
                if (denyAllACE != null) {
                    return denyAllACE;
                }
            }
        }
        return null;
    }

    public static boolean isPrivate(Session session, String path) {
        try {
            return (getDenyAllEveryoneACE(session, path) != null);
        } catch (RepositoryException re) {
            return false;
        }
    }

    public static void addDenyAllEveryoneAsFirstACE(Session session, String path) throws RepositoryException {
        AccessControlUtils.denyAllToEveryone(session, path);
        moveDenyAllEveryoneToFirstRestriction(session, path);
    }

    public static void removeDenyAllEveryoneACE(Session session, String path) throws RepositoryException {
        AccessControlManager acm = session.getAccessControlManager();

        for (AccessControlPolicy policy : acm.getPolicies(path)) {
            if (policy instanceof AccessControlList) {
                AccessControlList accessControlList = (AccessControlList) policy;
                AccessControlEntry[] entries = accessControlList.getAccessControlEntries();
                AccessControlEntry denyAllACE = getDenyAllEveryoneACEFromAccessControlEntries(session, entries);
                if (denyAllACE != null) {
                    accessControlList.removeAccessControlEntry(denyAllACE);
                }
                acm.setPolicy(path, accessControlList);
            }
        }
    }

    private static void moveDenyAllEveryoneToFirstRestriction(Session session, String path) throws RepositoryException {
        AccessControlManager acm = session.getAccessControlManager();
        for (AccessControlPolicy policy : acm.getPolicies(path)) {
            if (policy instanceof AccessControlList) {
                AccessControlList accessControlList = (AccessControlList) policy;
                AccessControlEntry[] entries = accessControlList.getAccessControlEntries();
                AccessControlEntry denyAllACE = getDenyAllEveryoneACEFromAccessControlEntries(session, entries);
                if (denyAllACE != null) {
                    ((JackrabbitAccessControlList) accessControlList).orderBefore(denyAllACE, entries[0]);
                }
                acm.setPolicy(path, accessControlList);
            }
        }
    }

    private static AccessControlEntry getDenyAllEveryoneACEFromAccessControlEntries(Session session, AccessControlEntry[] entries) throws RepositoryException {
        for (AccessControlEntry ace : entries) {
            boolean isEveryone = ace.getPrincipal().equals(AccessControlUtils.getEveryonePrincipal(session));
            if (!isEveryone) {
                continue;
            }
            if (ace instanceof JackrabbitAccessControlEntry) {
                boolean isDenyACE = !((JackrabbitAccessControlEntry) ace).isAllow();
                if (!isDenyACE) {
                    continue;
                }
            }
            for (Privilege privilege : ace.getPrivileges()) {
                if (privilege.getName().equalsIgnoreCase("jcr:all")) {
                    return ace;
                }
            }
        }
        return null;
    }
}
