/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.parser.taghandlers;

import java.io.UnsupportedEncodingException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.day.cq.wcm.designimporter.ErrorCodes;
import org.apache.commons.lang.StringEscapeUtils;
import org.ccil.cowan.tagsoup.AttributesImpl;
import org.xml.sax.Attributes;

import com.day.cq.dam.indd.PageBuilder;
import com.day.cq.dam.indd.PageComponent;
import com.day.cq.wcm.designimporter.DesignImporterContext;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.UnsupportedTagContentException;
import com.day.cq.wcm.designimporter.api.ContainerComponentProvider;
import com.day.cq.wcm.designimporter.api.HTMLContentProvider;
import com.day.cq.wcm.designimporter.api.PageComponentProvider;
import com.day.cq.wcm.designimporter.api.TagHandler;
import com.day.cq.wcm.designimporter.api.TagHandlerProvider;
import com.day.cq.wcm.designimporter.parser.HTMLContentType;
import com.day.cq.wcm.designimporter.util.TagUtils;

/*
AdobePatentID="2613US01"
*/

/**
 * The abstract class that pulls up all the common tag handling functionality.
 *
 * <p>
 * The tag handlers usually attempt to delegate all the SAX events to nested handlers if
 * they exist. If there aren't any nested handlers, the incoming SAX events are used to
 * either populate the html markup/css/script buffers or generate CQ page
 * components(via component provider subclasses).
 * <p>
 *
 * <p>
 * Nested handlers are created when an element matching a tag handler registration rule is
 * encountered. Then on, all the SAX events are directed to the instantiated nested
 * handler until end of nested tag is encountered. Once that happens, the nested tag handler is
 * asked for whatever content/components it has culled out. After that, the nested tag handler
 * is popped out for GC.
 * </p>
 */
public abstract class AbstractTagHandler implements TagHandler, PageComponentProvider, HTMLContentProvider {

    /**
     * The buffer for maintaining the inline style content culled by this tag handler and all the child tag handlers that were created in
     * its stack
     */
    protected StringBuffer cssBuffer = new StringBuffer();

    /**
     * The current delegate for this tag handler.
     */
    protected TagHandler delegate;

    /**
     * The buffer for maintaining the HTML markup culled by this tag handler and all the child tag handlers that were created in its stack
     */
    protected StringBuffer htmlBuffer = new StringBuffer();

    protected DesignImporterContext designImporterContext;

    protected Map<String, String> metaData = new HashMap<String, String>();

    /**
     * The {@link PageBuilder} for builing CQ page components.
     */
    protected PageBuilder pageBuilder;

    /**
     * The list of page components culled by this tag handler and all the child tag handlers that were created in its stack
     */
    protected List<PageComponent> pageComponents = new ArrayList<PageComponent>();

    /**
     * The list of reference script paths found by this tag handler and all the child tag handlers that were created in its stack
     */
    protected List<String> referencedScripts = new ArrayList<String>();

    /**
     * The list of reference CSS paths found by this tag handler and all the child tag handlers that were created in its stack
     */
    protected List<String> referencedStyleSheets = new ArrayList<String>();

    /**
     * The buffer for maintaining the inline script culled by this tag handler and all the child tag handlers that were created in its stack
     */
    protected StringBuffer scriptBuffer = new StringBuffer();

    protected TagHandlerProvider tagHandlerProvider;

    private int counter;

    protected String componentDivStartTag = "";
    
    public static final String NAME_HINT_PROPERTY_KEY = "cq:importNameHint";

    public void beginHandling(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        if (atts.getIndex("data-cq-component") != -1 && localName.matches("div|span")) {
            AttributesImpl attributes = new AttributesImpl(atts);
            if ("span".equalsIgnoreCase(localName)) {
                attributes.addAttribute("", "style", "style", "CDATA", "display: inline-block");
            }
            componentDivStartTag = TagUtils.getStartTag(uri, localName, qName, attributes);
        }
        if (this instanceof ContainerComponentProvider) designImporterContext.componentSuffixGenerator.startComponentStack();
    }

    public void characters(char[] ch, int start, int length) throws DesignImportException {
        if (delegate != null) {
            delegate.characters(ch, start, length);
        } else {
            String chars = new String(ch).substring(start, start + length)/* .replace('\n', ' ').replace('\t', ' ').trim() */;
            if (chars.length() > 0) {
				htmlBuffer.append(StringEscapeUtils.escapeHtml(chars));
            }
        }
    }

    public void endElement(String uri, String localName, String qName) throws DesignImportException {
        if (delegate != null) {
            if (--counter == 0) { // Termination condition
                delegate.endHandling(uri, localName, qName);

                if (delegate instanceof HTMLContentProvider) {
                    for (HTMLContentType contentType : HTMLContentType.values()) {
                        if (((HTMLContentProvider) delegate).supportsContent(contentType)) {
                            Object content = ((HTMLContentProvider) delegate).getContent(contentType);
                            if (content != null) pushContent(content, contentType);
                        }
                    }
                }

                if (delegate instanceof PageComponentProvider) {
                    List<PageComponent> components = ((PageComponentProvider) delegate).getPageComponents();
                    if (components != null) getPageComponents().addAll(components);
                }

                delegate = null;
            } else {
                delegate.endElement(uri, localName, qName);
            }
        }
    }

    public void endHandling(String uri, String localName, String qName) throws DesignImportException {
        if (this instanceof ContainerComponentProvider) designImporterContext.componentSuffixGenerator.endComponentStack();
    }

    public Object getContent(HTMLContentType htmlContentType) {

        switch (htmlContentType) {

            case META:
                return metaData;

            case MARKUP:
                return htmlBuffer.toString();

            case SCRIPT_INCLUDE:
                return referencedScripts;

            case SCRIPT_INLINE:
                return scriptBuffer.toString();

            case STYLES_INLINE:
                return cssBuffer.toString();

            case STYLESHEET_INCLUDE:
                return referencedStyleSheets;
        }

        return null;
    }

    public List<PageComponent> getPageComponents() {
        return pageComponents;
    }

    public void setDesignImporterContext(DesignImporterContext designImporterContext) {
        this.designImporterContext = designImporterContext;
    }

    public void setPageBuilder(PageBuilder pageBuilder) {
        this.pageBuilder = pageBuilder;
    }

    public void setTagHandlerProvider(TagHandlerProvider tagHandlerProvider) {
        this.tagHandlerProvider = tagHandlerProvider;
    }

    public void startElement(String uri, String localName, String qName, Attributes atts) throws DesignImportException {
        if (delegate == null) {
            delegate = tagHandlerProvider.createTagHandler(localName, atts);
            if (delegate != null) {
                if (delegate instanceof CanvasComponentTagHandler) {
                    // Only one canvas is supported.
                    throw new UnsupportedTagContentException(
                            ErrorCodes.NESTED_CANVAS_COMPONENT);
                }
                delegate.setTagHandlerProvider(tagHandlerProvider);
                delegate.setDesignImporterContext(designImporterContext);

                if (delegate instanceof PageComponentProvider) {
                    ((PageComponentProvider) delegate).setPageBuilder(pageBuilder);
                }

                delegate.beginHandling(uri, localName, qName, atts);
                counter = 1;
            }
        } else {
            delegate.startElement(uri, localName, qName, atts);
            counter++;
        }
    }

    public boolean supportsContent(HTMLContentType htmlContentType) {
        // This base class puts no restiction on the type of content it supports.
        // The extending classes may override this method to restrict their support.
        return true;
    }

    /**
     * Pushes the HTML content, based on its type, to the respective buffers.
     * 
     * @param content The HTML content which could be some markup, script, includes etc.
     * @param contentType The {@link HTMLContentType}
     */
    @SuppressWarnings("unchecked")
    private void pushContent(Object content, HTMLContentType contentType) {

        switch (contentType) {

            case META:
                metaData.putAll((Map<? extends String, ? extends String>) content);
                break;

            case MARKUP:
                htmlBuffer.append(content);
                break;

            case SCRIPT_INCLUDE:
                referencedScripts.addAll((Collection<? extends String>) content);
                break;

            case SCRIPT_INLINE:
                scriptBuffer.append(content);
                break;

            case STYLES_INLINE:
                cssBuffer.append(content);
                break;

            case STYLESHEET_INCLUDE:
                referencedStyleSheets.addAll((Collection<? extends String>) content);
                break;
        }

    }
}
