/*************************************************************************
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 * Copyright 1997 Adobe
 * All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains
 * the property of Adobe and its suppliers, if any. The intellectual
 * and technical concepts contained herein are proprietary to Adobe
 * and its suppliers and are protected by all applicable intellectual
 * property laws, including trade secret and copyright laws.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 **************************************************************************/
package com.day.crx;

import javax.jcr.RepositoryException;

/**
 * The <code>CRXModule</code> interface defines the API which must be
 * implemented by modules extending the functionality of the repository. Modules
 * are installed into the repository either by configuration or by calling the
 * {@link CRXRepository#installModule(javax.jcr.Session, CRXModule)} method.
 * <p>
 * <b>Lifecycle</b>
 * <p>
 * The life cycle of a module encompasses six stages:
 * <dl>
 * <dt>Instantiation</dt>
 * <dd>The extension class instantiated either by the repository in the case of
 * a configured module or by the application when manually installing the
 * module.</dd>
 * <dt>Configuration</dt>
 * <dd>The extension class is configured. In the case of a configured module,
 * configuration is based on bean style properties being set. When manually
 * installing the module, configuration may happen in an application specific
 * way.</dd>
 * <dt>Start</dt>
 * <dd>The module is started when it is installed into the repository. Starting
 * the module comprises two steps: (1) {@link #start(CRXSession) Starting} and
 * (2) registering the module internally. These operations are implemented by
 * the repository.</dd>
 * <dt>Operation</dt>
 * <dd>After having been started until being {@link #stop() stopped}, the
 * module may operate as implemented. A module should not operate before being
 * started and should cease operating after having been stopped.</dd>
 * <dt>Stop</dt>
 * <dd>The module is stopped when it is uninstalled from the repository.
 * Stopping the module comprises two steps: (1) Unregistering the module
 * internally and (2) {@link #stop() stopping} the module. After having been
 * stopped, the module should not operate anymore.</dd>
 * <dt>Garbage Collection</dt>
 * <dd>After the module has been stopped, the repository releases all hard
 * references to the module instance. Provided no other hard references exist
 * any more, the object will be collected by the Java VM sooner or later.</dd>
 * </dl>
 * <p>
 * When the repository is shutting down, all modules which are still installed
 * are uninstalled, that is unregistered and stopped.
 * <p>
 * <b>Configuration</b>
 * <p>
 * A module configured with the the repository configuration file is configured
 * automatically when the repository is started. Modules are configured in
 * <code>Module</code> elements nested within the <code>Modules</code>
 * element. Configuration properties may be set by nested <code>param</code>
 * elements.
 * <p>
 * When the module is installed by the repository, the module's class as
 * configured in the <code>Module.class</code> attribute is instantiated and
 * the configured properties from the nested <code>param</code> elements are
 * set JavaBean style.
 * <p>
 * For the repository to be able to instantiate the module, the module must have
 * a public default constructor. Also the property setter methods must be public
 * and for each property a setter and a getter method must be provided for the
 * parameter configuration to be successfull.
 * <p>
 * <b>Installation</b>
 * <p>
 * Modules may be installed automatically when the repository is started or
 * manually by the application. To automatically start the modules, they must be
 * configured as described above. To manually install a module, the application
 * is responsible for the instantiation and configuration of the module.
 * <p>
 * To install, the application will call the
 * {@link CRXRepository#installModule(javax.jcr.Session, CRXModule)} which then
 * calls the {@link #start(CRXSession)} method to start the module before
 * registering it.
 *
 */
public interface CRXModule {

    /**
     * Returns the name of this module.
     * <p>
     * This name is expected to be unique amongst the registered modules and
     * is just used to identify the module. Other than to register the module
     * under this name, the repository does not use it for anything else.
     *
     * @return The non-<code>null</code>, non-empty module name.
     */
    String getName();

    /**
     * Starts this module and makes it ready for use.
     * <p>
     * This method is called by the repository before registering the module
     * internally. This means, that while this method is active, this module is
     * not yet visible by the {@link CRXRepository#getModule(String)} and
     * {@link CRXRepository#getModules()} method.
     * <p>
     * If this method fails by throwing a <code>RepositoryException</code>,
     * any resources already acquired must be released. A module failing to
     * start is also not registered and thus visible. Finally a module failing
     * to start will never be stopped, that is the {@link #stop()} method will
     * not be called.
     * <p>
     * This method should only throw a <code>RepositoryException</code> if
     * failing to start. Notably, the implementation should prevent throwing a
     * <code>RuntimeException</code>.
     * <p>
     * <b><i>NOTE:</i></b>
     * <ul>
     * <li>The module MUST NOT call the <code>logout()</code> method on this
     *  <code>session</code>. Otherwise the repository may behave unexpectedly.
     * <li>The <code>session</code> is a shared Session. This means, the Session
     *  is not thread safe and may at most be used for concurrent read
     *  operations but MUST NOT be used to write to the repository. If the
     *  module requires writing to the repository, a new Session MUST be
     *  acquired calling the {@link CRXSession#getSession(String)} method on
     *  the <code>session</code>. Such session SHOULD of course be logged out
     *  when the module is {@link #stop() stopped}}.
     * </ul>
     *
     * @param session The {@link CRXSession} to access the system workspace of
     *            the repository. This session is shared and MUST NOT be used
     *            to write to the repository. Use the
     *            {@link CRXSession#getSession(String)} method to get a session
     *            to write to the repository.
     *
     * @throws RepositoryException May be thrown if the module fails to start.
     */
    void start(CRXSession session) throws RepositoryException;

    /**
     * Stops this module.
     * <p>
     * This method is called after the module has been unregistered. If the
     * module failed to start, this method is never called.
     * <p>
     * This method is not excpected to throw any exception. Care must be taken
     * to prevent <code>RuntimeExceptions</code> from occurring.
     */
    void stop();

}
