/*
*
*	File: LookupTable.java
*
*
*	ADOBE CONFIDENTIAL
*	___________________
*
*	Copyright 2004-2005 Adobe Systems Incorporated
*	All Rights Reserved.
*
*	NOTICE: All information contained herein is, and remains the property of
*	Adobe Systems Incorporated and its suppliers, if any. The intellectual
*	and technical concepts contained herein are proprietary to Adobe Systems
*	Incorporated and its suppliers and may be covered by U.S. and Foreign
*	Patents, patents in process, and are protected by trade secret or
*	copyright law. Dissemination of this information or reproduction of this
*	material is strictly forbidden unless prior written permission is obtained
*	from Adobe Systems Incorporated.
*
*/

package com.adobe.fontengine.font.opentype;

import com.adobe.fontengine.font.FontByteArray;
import com.adobe.fontengine.font.InvalidFontException;
import com.adobe.fontengine.font.UnsupportedFontException;
import com.adobe.fontengine.inlineformatting.AttributedRun;

/**
 * A LookupTable is the base class for GSUB/GPOS tables.
 * It captures:
 * <ul>
 * <li>the resolution of feature tags to lookup indices
 * <li>the common lookup types (contextual, chaining contextual and extension)
 * </ul>
 */
public abstract class LookupTable extends LayoutTable {

  public LookupTable (FontByteArray buffer) 
  throws java.io.IOException, InvalidFontException, UnsupportedFontException {
    super (buffer); 
  }
  
  // While both GPOS and GSUB have the offsets to the script list, feature
  // list and lookup lists at the same positions, these could in principle
  // be different, so Gpos and Gsub provide getters.

  /** Returns the offset of the ScriptList in the table. */
  abstract protected int getScriptListOffset () throws InvalidFontException;
  
  /** Returns the offset of the FeatureList in the table. */
  abstract protected int getFeatureListOffset () throws InvalidFontException;
  
  /** Returns the offset of the LookupList in the table. */
  abstract protected int getLookupListOffset () throws InvalidFontException;
  
 
  /** Resolves a script/lang/feature list to a set of lookup offsets. 
   * The lookups of the required feature are not included, but
   * can be obtained by resolving <code>Tag.feature_REQUIRED</code>.
   * @param scriptTag the script under which to resolve the feature
   * @param langSysTag the langage under which to resolve the feature
   * @param featureTags the features to resolve
   * @return array of lookup offsets; may be empty but never null
   */
  public int[][] resolveFeatureTag (int scriptTag, int langSysTag, int[] featureTags)
  throws InvalidFontException {
    int[][] result = new int [featureTags.length][];
    
    // Resolve the scriptTag to a scriptIndex. This is, in order of preference:
    // - the index of the script with an exact match on the tag
    // - the index of the script with the tag Tag.script_DFLT
    // - the value -1, meaning there is no appropriate script
    
    int scriptListOffset = getScriptListOffset ();
    int scriptCount = this.data.getuint16 (scriptListOffset);
    int resolvedScriptIndex = -1;
    
    for (int script = 0; script < scriptCount; script++) {
      int thisScriptTag = this.data.getint32 (scriptListOffset + 2 + 6*script);
      if (scriptTag == thisScriptTag) {
        resolvedScriptIndex = script;
        break; }
      if (Tag.script_DFLT == thisScriptTag) {
        resolvedScriptIndex = script; }}
    
    if (resolvedScriptIndex == -1) {
      for (int i = 0; i < result.length; i++) {
        result [i] = new int [0]; }
      return result; }

    // Resolve the languageTag to an offset to a LangSys table. This is,
    // in order of preference:
    // - the offset of the langSys table with an exact match on the tag
    // - the offset of the default langSys table, if present
    // - the value 0, meaning there is no appropriate language
    
    int scriptOffset = this.data.getOffset (scriptListOffset, 2 + 6*resolvedScriptIndex + 4);
    int langSysCount = this.data.getuint16 (scriptOffset + 2);
    // start with the default language; this will be 0 if there is no 
    // default language
    int resolvedLangSysOffset = this.data.getOffset (scriptOffset, 0);
    
    for (int lang = 0; lang < langSysCount; lang++) {
      if (langSysTag == this.data.getuint32 (scriptOffset + 4 + 6*lang)) {
        resolvedLangSysOffset = this.data.getOffset (scriptOffset, 4 + 6*lang + 4);
        break; }}
             
    if (resolvedLangSysOffset == 0) {
      for (int i = 0; i < result.length; i++) {
        result [i] = new int [0]; }
      return result; }
    

    // Resolve the featureTag to a feature index. There is, in order of 
    // preference:
    // - the required feature, if there is one and featureTag is 
    //   Tag.feature_REQUIRED
    // - the index of the feature with the tag featureTag if featureTag is
    //   not Tag.feature_REQUIRED
    // - the value -1, meaning there is no appropriate features
    
    int featureListOffset = getFeatureListOffset ();

    for (int feature = 0; feature < featureTags.length; feature++) {
      int resolvedFeatureIndex = -1;
      
      if (Tag.feature_REQUIRED == featureTags [feature]) {
        int requiredFeatureIndex = this.data.getuint16 (resolvedLangSysOffset + 2); 
        if (requiredFeatureIndex == 0xffff) {
          resolvedFeatureIndex = -1; }
        else {
          resolvedFeatureIndex = requiredFeatureIndex; }}
      
      else {
        int featureCount = this.data.getuint16 (resolvedLangSysOffset + 4);
        for (int f = 0; f < featureCount; f++) {
          int thisFeatureIndex = this.data.getuint16 (resolvedLangSysOffset + 6 + 2*f);
          if (featureTags [feature] == this.data.getuint32 (featureListOffset + 2 + 6*thisFeatureIndex)) {
            resolvedFeatureIndex = thisFeatureIndex; }}}
      
      if (resolvedFeatureIndex == -1) {
        result [feature] = new int [0]; }
      
      else {
        // Assemble the sorted list of lookup indices
        int featureOffset = featureListOffset 
               + this.data.getuint16 (featureListOffset + 2 + resolvedFeatureIndex*6 + 4);
        int lookupCount = this.data.getuint16 (featureOffset + 2);
        result [feature] = new int [lookupCount];
        for (int l = 0; l < lookupCount; l++) {
          result [feature][l] = this.data.getuint16 (featureOffset + 4 + l*2); }
        java.util.Arrays.sort (result [feature]);
        
        // Convert to lookup offsets
        int lookupListOffset = getLookupListOffset ();
        for (int l = 0; l < lookupCount; l++) {
          result[feature] [l] = this.data.getOffset (lookupListOffset, 2 + 2*result[feature][l]); }}}
    
    return result;
  }
  
  /* Determine the presence of a feature.
   * A feature is deemed to be present in a font if it appears in
   * in the feature list; it could very well be that the feature is 
   * not referenced by any script/language combination, or only some.
   * 
   * @param featureTag the feature to look up
   * @return true iff the feature appears in the feature list
   */
  public boolean featureIsPresent (int featureTag) throws InvalidFontException {
    int featureListOffset = getFeatureListOffset ();
    int featureCount = this.data.getuint16 (featureListOffset);
    
    for (int f = 0; f < featureCount; f++) {
      if (featureTag == this.data.getuint32 (featureListOffset + 2 + 6*f)) {
        return true; }}
    
    return false;
  }

  
  public int applyLookups (int[] lookupOffsets,
                           AttributedRun run, int start, int limit,
                           OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    for (int l = 0; l < lookupOffsets.length; l++) {
      int lookupType = this.data.getuint16 (lookupOffsets [l]);
      int lookupFlag = this.data.getuint16 (lookupOffsets [l] + 2);
      int subtableCount = this.data.getuint16 (lookupOffsets [l]+ 4);
      
      int curGlyph = start;
      while (curGlyph < limit) {
        if (lookupFlagCovers (lookupFlag, gdef, run.elementAt (curGlyph))) {
          curGlyph++;
          continue; }
        int st = 0;
        boolean doneAtThisPos = false;
        while (st < subtableCount && ! doneAtThisPos) {
          int stOffset = this.data.getOffset (lookupOffsets [l], 6 + 2*st);
          
          LookupResult result 
            = applyLookupSubtable (lookupType, lookupFlag, stOffset, 
                                   run, start, limit, curGlyph, selector, gdef);
          if (result.applied) {
            curGlyph = result.nextToProcess;
            limit += result.countAdjust; 
            doneAtThisPos = true; }
          st++; }
        
        if (!doneAtThisPos) {
          // no subtable worked, just move to the next glyph
          curGlyph++; }}}
    
    return limit;
  }
  
  
  /** Report the result of applying a lookup at a position. */
  final static protected class LookupResult {
    /** The lookup was applied. */
    public boolean applied;
    /** The next glyph to process.*/
    public int nextToProcess;
    public int countAdjust;
    
    public LookupResult (boolean applied, int nextToProcess, int countAdjust) {
      this.applied = applied;
      this.nextToProcess = nextToProcess;
      this.countAdjust = countAdjust;
    }
  }
  
  static final protected LookupResult lookupNotApplied = new LookupResult (false, 0, 0);
  
  
  abstract protected LookupResult applyLookupSubtable 
          (int lookupType, int lookupFlag, int stOffset,
           AttributedRun run, int start, int limit, int curGlyph, 
           OTSelector selector, Gdef gdef)
  throws InvalidFontException;
  


  protected LookupResult applyContextualSubtable
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph, 
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    
    switch (format) {
      case 1: { 
        return applyContextualSubtableFormat1 (lookupFlag, stOffset,
            run, start, limit, curGlyph, selector, gdef); }
      case 2: { 
        return applyContextualSubtableFormat2 (lookupFlag, stOffset,
            run, start, limit, curGlyph, selector, gdef); }
      case 3: { 
        return applyContextualSubtableFormat3 (lookupFlag, stOffset,
            run, start, limit, curGlyph, selector, gdef); }
      default: {
        throw new InvalidFontException ("Invalid contextual lookup subtable format (" + format + ")"); }}
  }
   
      
  protected LookupResult applyContextualSubtableFormat1
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci == -1) {
      return lookupNotApplied; }
    
    int ruleSetOffset = stOffset + this.data.getuint16 (stOffset + 6 + ci*2);
    int ruleCount = this.data.getuint16 (ruleSetOffset);
    
    for (int s = 0; s < ruleCount; s++) {
      int ruleOffset = this.data.getOffset (ruleSetOffset, 2 + 2*s);
      
      int[] matchedPositions = matchOneRule (ruleOffset, lookupFlag,
           run, start, limit, inPos, selector, gdef);
      
      if (matchedPositions.length != 0 && selector.isApplied (run, matchedPositions)) {
        
        int glyphCount = this.data.getuint16 (ruleOffset);
        int applyCount = this.data.getuint16 (ruleOffset + 2);
        int applyOffset = ruleOffset + 4 + 2*(glyphCount-1);
        return applySubLookups (applyCount, applyOffset, run, start, limit, matchedPositions, 
            selector, gdef); }}
    
    return lookupNotApplied;
  }
  
      
  protected LookupResult applyContextualSubtableFormat2
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int classDefOffset = this.data.getOffset (stOffset, 4);
    int inPos = curGlyph;
    
    if (-1 == getCoverageIndex (run.elementAt (inPos), coverageOffset)) {
      return lookupNotApplied; }
    
    int cl = getClassIndex (run.elementAt (inPos), classDefOffset);
    if (cl > this.data.getuint16 (stOffset + 6) - 1) {
      return lookupNotApplied; }
    
    int classSetOffset = this.data.getOffset (stOffset, 8 + 2*cl);
    if (classSetOffset == 0) {
      return lookupNotApplied; }
    
    int classRuleCnt = this.data.getuint16 (classSetOffset);
    
    for (int i = 0; i < classRuleCnt; i++) {
      int classRuleOffset = this.data.getOffset (classSetOffset, 2 + 2*i);
      int glyphCount = this.data.getuint16 (classRuleOffset);
      
      int[] matchedPositions = matchOneClassRule (classRuleOffset, classDefOffset, 
          lookupFlag, run, start, limit, inPos, selector, gdef);
      
      if (matchedPositions.length != 0 && selector.isApplied (run, matchedPositions)) {
        int applyCount = this.data.getuint16 (classRuleOffset + 2);
        int applyOffset = classRuleOffset + 4 + 2*(glyphCount-1);
        return applySubLookups (applyCount, applyOffset,
            run, start, limit, matchedPositions, 
            selector, gdef); }}
    return lookupNotApplied; 
  }
  
  protected LookupResult applyContextualSubtableFormat3
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int glyphCount = this.data.getuint16 (stOffset + 2);
    int inPos = curGlyph;
    int [] matchedPositions = new int [glyphCount];
    
    for (int i = 0; i < glyphCount; i++) {
      if (i != 0) {
        while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
          inPos++; }}
      
      if (inPos >= limit || -1 == getCoverageIndex (run.elementAt (inPos), this.data.getOffset (stOffset, 6 + 2*i))) {
        return lookupNotApplied; }
      
      matchedPositions [i] = inPos;
      inPos++; }
    
    if (! selector.isApplied (run, matchedPositions)) {
      return lookupNotApplied; }
    
    int applyCount = this.data.getuint16 (stOffset + 4);
    int applyOffset = stOffset + 6 + 2*glyphCount;
    
    return applySubLookups (applyCount, applyOffset, run, start, limit, matchedPositions, selector, gdef); 
  }
  

  static final int[] noMatch = {};
  
  int[] matchOneRule (int ruleOffset, int lookupFlag,
                      AttributedRun run, int start, int limit, int inPos, 
                      OTSelector selector, Gdef gdef) 
  throws InvalidFontException {
    
    int glyphCount = this.data.getuint16 (ruleOffset);
    int [] matchedPositions = new int [glyphCount];
    matchedPositions [0] = inPos;
    inPos++;
    
    for (int i = 1; i < glyphCount; i++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (inPos >= limit || run.elementAt (inPos) != this.data.getuint16 (ruleOffset + 4 + 2*(i-1))) {
        return noMatch; }
      
      matchedPositions [i] = inPos;
      inPos++; }
    
    return matchedPositions;
  }
  
  
  int[] matchOneClassRule (int classRuleOffset, int classOffset, int lookupFlag,
                           AttributedRun run, int start, int limit, int inPos,
                           OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int glyphCount = this.data.getuint16 (classRuleOffset);
    int[] matchedPositions = new int [glyphCount];
    matchedPositions [0] = inPos;
    inPos++;
    
    for (int g = 1; g < glyphCount; g++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (inPos >= limit
          || getClassIndex (run.elementAt (inPos), classOffset) 
               != this.data.getuint16 (classRuleOffset + 4 + 2*(g-1))) {
        return noMatch; }
      
      matchedPositions [g] = inPos;
      inPos++; }
    
    return matchedPositions;
  }
  
  protected LookupResult applyChainingContextualSubtable
         (int lookupFlag, int stOffset,
          AttributedRun run, int start, int limit, int curGlyph,
          OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int format = this.data.getuint16 (stOffset);
    
    switch (format) {
      case 1: { 
        return applyChainingContextualSubtableFormat1 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 2: { 
        return applyChainingContextualSubtableFormat2 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      case 3: { 
        return applyChainingContextualSubtableFormat3 (lookupFlag, stOffset, run, start, limit, curGlyph, selector, gdef); }
      
      default: {
        throw new InvalidFontException ("Invalid ChainingContextualLookupSutable format (" + format + ")"); }}
  }


  protected LookupResult applyChainingContextualSubtableFormat1
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    int ci = getCoverageIndex (run.elementAt (inPos), coverageOffset);
    if (ci == -1) {
      return lookupNotApplied; }
    
    int ruleSetOffset = this.data.getOffset (stOffset, 6 + 2*ci);
    int ruleCount = this.data.getuint16 (ruleSetOffset);
    
    for (int s = 0; s < ruleCount; s++) {
      int ruleOffset = this.data.getOffset (ruleSetOffset, 2 + 2*s);
      
      int [] matchedPositions 
      = matchOneChainRule (ruleOffset, lookupFlag, run, start, limit, inPos,
          selector, gdef);
      
      if (   matchedPositions.length != 0
          && selector.isApplied (run, matchedPositions)) {
        int backtrackGlyphCount = this.data.getuint16 (ruleOffset);
        int inputGlyphCount = this.data.getuint16 (ruleOffset + 2 
            + 2*backtrackGlyphCount);
        int lookaheadGlyphCount = this.data.getuint16 (ruleOffset + 4
            + 2*backtrackGlyphCount
            + 2*(inputGlyphCount -1));
        int applyCountOffset = ruleOffset + 6
                                 + 2*backtrackGlyphCount
                                 + 2*(inputGlyphCount - 1)
                                 + 2*lookaheadGlyphCount;
        int applyCount = this.data.getuint16 (applyCountOffset);
        int applyOffset = applyCountOffset + 2;
        return applySubLookups (applyCount, applyOffset, run, start, limit, matchedPositions, selector, gdef); }}
    
    return lookupNotApplied;
  }
      

  protected LookupResult applyChainingContextualSubtableFormat2
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int coverageOffset = this.data.getOffset (stOffset, 2);
    int inPos = curGlyph;
    
    if (-1 == getCoverageIndex (run.elementAt (inPos), coverageOffset)) {
      return lookupNotApplied; }
    
    int backtrackClassDefOffset = this.data.getOffset (stOffset, 4);
    int inputClassDefOffset = this.data.getOffset (stOffset, 6);
    int lookaheadClassDefOffset = this.data.getOffset (stOffset, 8);
    
    int cl = getClassIndex (run.elementAt (inPos), inputClassDefOffset);
    if (cl > this.data.getuint16 (stOffset + 10) - 1) {
      return lookupNotApplied; }
    int ruleSetOffset = this.data.getOffset (stOffset, 12 + 2*cl);
    if (ruleSetOffset == 0) {
      return lookupNotApplied; }
    int ruleCount = this.data.getuint16 (ruleSetOffset);
    
    for (int s = 0; s < ruleCount; s++) {
      int ruleOffset = this.data.getOffset (ruleSetOffset, 2 + 2*s);
      
      int [] matchedPositions 
      = matchOneChainClassRule (ruleOffset,
          backtrackClassDefOffset,
          inputClassDefOffset,
          lookaheadClassDefOffset,
          lookupFlag,
          run, start, limit, inPos, selector, gdef);
      
      if (   matchedPositions.length != 0 && selector.isApplied (run, matchedPositions)) {
        int backtrackGlyphCount = this.data.getuint16 (ruleOffset);
        int inputGlyphCount = this.data.getuint16 (ruleOffset + 2 
                                + 2*backtrackGlyphCount);
        int lookaheadGlyphCount = this.data.getuint16 (ruleOffset + 4
                                    + 2*backtrackGlyphCount
                                    + 2*(inputGlyphCount -1));
        int applyCountOffset = ruleOffset + 6
                                 + 2*backtrackGlyphCount
                                 + 2*(inputGlyphCount - 1)
                                 + 2*lookaheadGlyphCount;
        int applyCount = this.data.getuint16 (applyCountOffset);
        int applyOffset = applyCountOffset + 2;
        return applySubLookups (applyCount, applyOffset, run, start, limit, matchedPositions, selector, gdef); }}
    
    return lookupNotApplied;
  }
      

  protected LookupResult applyChainingContextualSubtableFormat3
        (int lookupFlag, int stOffset,
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int backtrackGlyphCount = this.data.getuint16 (stOffset + 2);
    int inputGlyphCount = this.data.getuint16 (stOffset + 4 + 2*backtrackGlyphCount);
    int lookaheadGlyphCount = this.data.getuint16 (stOffset + 6 + 2*backtrackGlyphCount
        + 2*inputGlyphCount);
    
    int backPos = curGlyph - 1;
    
    for (int b = 0 ; b < backtrackGlyphCount; b++) {
      while (start <= backPos && lookupFlagCovers (lookupFlag, gdef, run.elementAt (backPos))) {
        backPos--; }
      
      if (backPos < start || -1 == getCoverageIndex (run.elementAt (backPos), this.data.getOffset (stOffset, 4 + 2*b))) {
        return lookupNotApplied; }
      
      backPos--; }
    
    int [] matchedPositions = new int [inputGlyphCount];
    int inPos = curGlyph;
    
    for (int i = 0; i < inputGlyphCount; i++) {
      if (i != 0) {
        while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
          inPos++; }}
      
      if (   limit <= inPos 
          || -1 == getCoverageIndex (run.elementAt (inPos), this.data.getOffset (  stOffset, 6 
                               + 2*backtrackGlyphCount
                               + 2*i))) {
        return lookupNotApplied; }
      
      matchedPositions [i] = inPos;
      inPos++; }
    
    if (! selector.isApplied (run, matchedPositions)) {
      return lookupNotApplied; }
    
    for (int l = 0; l < lookaheadGlyphCount; l++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (   limit <= inPos 
          || -1 == getCoverageIndex (run.elementAt (inPos),
                        this.data.getOffset (  stOffset, 8
                                + 2*backtrackGlyphCount
                                + 2*inputGlyphCount
                                + 2*l))) {
        return lookupNotApplied; }
      
      inPos++; } 
    
    int applyCountOffset = stOffset + 8
                             + 2*backtrackGlyphCount
                             + 2*inputGlyphCount
                             + 2*lookaheadGlyphCount;
    int applyCount = this.data.getuint16 (applyCountOffset);
    int applyOffset = applyCountOffset + 2;
    return applySubLookups (applyCount, applyOffset, run, start, limit, matchedPositions,
        selector, gdef);
  }
  
  int[] matchOneChainRule (int ruleOffset, int lookupFlag, 
                           AttributedRun run, int start, int limit, int inPos, 
                           OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int backtrackGlyphCount = this.data.getuint16 (ruleOffset);
    int inputGlyphCount = this.data.getuint16 (ruleOffset + 2 
                            + 2*backtrackGlyphCount);
    int lookaheadGlyphCount = this.data.getuint16 (ruleOffset + 4
                                + 2*backtrackGlyphCount
                                + 2*(inputGlyphCount - 1));
    
    int backPos = inPos - 1;
    for (int b = 0;  b < backtrackGlyphCount; b++) {
      
      while (start <= backPos && lookupFlagCovers (lookupFlag, gdef, run.elementAt (backPos))) {
        backPos--; }
      
      if (backPos < start || run.elementAt (backPos) != this.data.getuint16 (ruleOffset + 2 + 2*b)) {
        return noMatch; }
      
      backPos--; }
    
    
    int [] matchedPositions = new int [inputGlyphCount];
    matchedPositions [0] = inPos;
    inPos++;
    
    for (int i = 1; i < inputGlyphCount; i++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (   limit <= inPos 
          || run.elementAt (inPos) != this.data.getuint16 (ruleOffset + 4
                        + 2*backtrackGlyphCount + 2*(i-1))) {
        return noMatch; }
      
      matchedPositions [i] = inPos;
      inPos++; }
    
    for (int l = 0; l < lookaheadGlyphCount; l++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (    limit <= inPos 
          ||  run.elementAt (inPos) != this.data.getuint16 (ruleOffset + 6
                   + 2*backtrackGlyphCount
                   + 2*(inputGlyphCount-1) + 2*l)) {
        return noMatch; }
      
      inPos++; }
    
    return matchedPositions;
  }
  
  
  int[] matchOneChainClassRule (
              int ruleOffset,
              int backtrackClassDefOffset,
              int inputClassDefOffset,
              int lookaheadClassDefOffset,
              int lookupFlag,
              AttributedRun run, int start, int limit, int inPos,
              OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int backtrackGlyphCount = this.data.getuint16 (ruleOffset);
    int inputGlyphCount = this.data.getuint16 (ruleOffset + 2 
        + 2*backtrackGlyphCount);
    int lookaheadGlyphCount = this.data.getuint16 (ruleOffset + 4
        + 2*backtrackGlyphCount
        + 2*(inputGlyphCount - 1));
    
    int backPos = inPos - 1;
    for (int b = 0; b < backtrackGlyphCount; b++) {
      
      while (start <= backPos && lookupFlagCovers (lookupFlag, gdef, run.elementAt (backPos))) {
        backPos--; }
      
      if (backPos < start
          || getClassIndex (run.elementAt (backPos), backtrackClassDefOffset)
                  != this.data.getuint16 (ruleOffset + 2 + 2*b)) {
        return noMatch; }
      
      backPos--; }
    
    int [] matchedPositions = new int [inputGlyphCount];
    matchedPositions [0] = inPos;
    inPos++;
    
    for (int i = 1; i < inputGlyphCount; i++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (limit <= inPos
          || getClassIndex (run.elementAt (inPos), inputClassDefOffset)
                 != this.data.getuint16 (ruleOffset + 4
                               + 2*backtrackGlyphCount + 2*(i-1))) {
        return noMatch; }
      
      matchedPositions [i] = inPos;
      inPos++; }
    
    for (int l = 0; l < lookaheadGlyphCount; l++) {
      
      while (inPos < limit && lookupFlagCovers (lookupFlag, gdef, run.elementAt (inPos))) {
        inPos++; }
      
      if (limit <= inPos
          || getClassIndex (run.elementAt (inPos), lookaheadClassDefOffset)
               != this.data.getuint16 (ruleOffset + 6
                              + 2*backtrackGlyphCount
                              + 2*(inputGlyphCount-1) + 2*l)) {
        return noMatch; }
      
      inPos++; }
    
    return matchedPositions;
  }
  
  
  protected LookupResult applyExtensionSubtable
        (int lookupFlag, int stOffset, 
         AttributedRun run, int start, int limit, int curGlyph,
         OTSelector selector, Gdef gdef)
  throws InvalidFontException {
    
    int extensionLookupType = this.data.getuint16 (stOffset + 2);
    int extensionOffset = stOffset + (int)this.data.getuint32 (stOffset + 4);
    
    return applyLookupSubtable (extensionLookupType, 
        lookupFlag, extensionOffset, run, start, limit, curGlyph, selector, gdef);
  }
  
  
  protected LookupResult applySubLookups
        (int count, int offset,
         AttributedRun run, int start, int limit, int[] matchedPositions, 
         OTSelector selector, Gdef gdef) 
  throws InvalidFontException {

    // We need to track how the matchedPositions are affected by applying 
    // sublookups. While the spec is not clear about this, Microsoft has
    // confirmed the following behaviour:
    //   input pattern = (10 11 12 13)
    //   glyphs ignored by lookup flags: 50 and 51
    //   first sublookup: applied at 1, replaces 11 by 20, 21, 22, 23
    //   second sublookup: applied at 6, replaced 13 by 24
    // For the input 10 50 11 51 12 13, we have:
    //   matchedPositions = {0, 2, 4, 5}
    // after the first subst lookup
    //   glyphs = 10 50 20 21 22 23 51 12 13
    //   matchedPositions = {0, 2, 3, 4, 5, 7, 8}
    // now position 6 is defined (it's glyph 8) and after the second sublookup:
    //   glyphs = 10 50 20 21 22 23 51 12 24
    //   matchedPositions = {0, 2, 3, 4, 5, 7, 8}
    // 
    // The modification of matchedPositions is actually a lot trickier than
    // it looks. Consider the case where the first sublookup is actually
    // a contextual lookup. My guess is that the right way to do this is
    // to use an proxy shaper during the sublookups; the proxy shaper forwards
    // the transformations to the client-provided shaper, and also ajusts 
    // matched positions.
    
    LookupResult result 
       = new LookupResult (true,
                            matchedPositions [matchedPositions.length -1] + 1,
                            0);
    
    int lookupListOffset = getLookupListOffset ();

    for (int su = 0; su < count; su++) {
      int sequenceIndex = this.data.getuint16 (offset);
      int lookupIndex = this.data.getuint16 (offset + 2);
      offset += 4; 
      int curPos = matchedPositions [sequenceIndex];
      
      int lookupOffset = this.data.getOffset (lookupListOffset, 2 + 2*lookupIndex);
      int lookupType = this.data.getuint16 (lookupOffset);
      int lookupFlag = this.data.getuint16 (lookupOffset + 2);
      int subtableCount = this.data.getuint16 (lookupOffset + 4);
      
      if (lookupFlagCovers (lookupFlag, gdef, run.elementAt (curPos))) {
        continue; }
      
      for (int st = 0; st < subtableCount; st++) {
        int stOffset = this.data.getOffset (lookupOffset, 6 + 2*st);
        
        LookupResult r = applyLookupSubtable (lookupType, lookupFlag, stOffset, 
                                              run, start, limit, curPos, selector, gdef);
        if (r.applied) {
          result.nextToProcess += r.countAdjust;
          result.countAdjust += r.countAdjust;
          break; }}}
    
    return result;
  }
  
  //------------------------------------------------------ required features ---

  // The code in this section is used to determine which fonts have a 
  // required feature. It is not used in production versions of AFE.
  
//  public static class ScriptLangRequiredFeature {
//  	public String table;
//  	public int scriptTag;
//  	public int langTag;
//  	public int requiredFeatureIndex;
//  	public int featureTag;
//  	
//  	public ScriptLangRequiredFeature (String table, int scriptTag, int langTag, int requiredFeatureIndex, int featureTag) {
//  		this.table = table;
//  		this.scriptTag = scriptTag;
//  		this.langTag = langTag;
//  		this.requiredFeatureIndex = requiredFeatureIndex;
//  		this.featureTag = featureTag;
//  	}
//  }
//  
//  public void inspectLangSysTable (String table, int langSysOffset, 
//                                   Set s, int scriptTag, int langTag)
//  throws InvalidFontException, UnsupportedFontException  {
//  	
//  	int requiredFeatureIndex = getuint16 (langSysOffset + 2);
//  	
//  	if (requiredFeatureIndex != 0xffff) { 
//  		int featureListOffset = getFeatureListOffset ();
//  		int featureTag = getint32 (featureListOffset + 2 + 6*requiredFeatureIndex);
//  		s.add (new ScriptLangRequiredFeature (table, scriptTag, langTag, 
//                                            requiredFeatureIndex, featureTag)); }
//  	}
//  
//  public Set/*<ScriptLang>*/ enumerateScriptLangWithRequiredFeature (String table)
//  throws InvalidFontException, UnsupportedFontException {
//  	
//  	Set/*<ScriptLang>*/ s = new HashSet/*<ScriptLang>*/ ();
//  	
//    int scriptListOffset = getScriptListOffset ();
//    int scriptCount = getuint16 (scriptListOffset);
//    
//    for (int script = 0; script < scriptCount; script++) {
//      int scriptTag = getint32 (scriptListOffset + 2 + 6*script);
//      int scriptOffset = getOffset (scriptListOffset, 2 + 6*script + 4);
//      
//      int langSysCount = getuint16 (scriptOffset + 2);
//      int defaultLangSysOffset = getOffset (scriptOffset, 0);
//      if (defaultLangSysOffset != 0) {
//      	inspectLangSysTable (table, defaultLangSysOffset, s, scriptTag, 0); }
//    
//      for (int lang = 0; lang < langSysCount; lang++) {
//      	int langSysTag = getint32 (scriptOffset + 4 + 6*lang);
//      	int langSysOffset = getOffset (scriptOffset, 4 + 6*lang + 4);
//      	
//      	inspectLangSysTable (table, langSysOffset, s, scriptTag, langSysTag); }}
//    
//    return s;
//  }  
}
