/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2005 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa;

import com.adobe.xfa.ut.ExFull;
import com.adobe.xfa.ut.FindBugsSuppress;
import com.adobe.xfa.ut.ResId;


/**
 * This class implements the NodeList interface based on a set of nodes
 * represented as the children of an element.
 * <p>
 * The NodeList interface is best suited to array storage, whereas an Element
 * stores children as a linked list. To compensate, we'll track our position in
 * the linked list so that normal forward iterations should be efficient.
 *
 * @exclude from published api -- Mike Tardif, May 2006.
 */

public final class ElementNodeList extends NodeList {
	private final Element mElement;

	private Node mLastNode;

	private int mLastPosn;

	private int mLength;

	ElementNodeList(Element e) {
		mElement = e;
		reset();
	}

	/**
	 * @see ListBase#append(Obj)
	 */
	public void append(Obj newNode) {
		if (isReadOnly() || newNode == null)
			return;

		if (!(newNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);
		
		mElement.appendChild((Node) newNode, true);
		reset();
	}

	/**
	 * Create a copy of this list.
	 * <br><br>This method does not make copies of the underlying object
	 *          implementations, it just place them in a new storage.
	 * 
	 * @return a new list
	 */
	public Object clone() {
		return new ArrayNodeList(mElement);
	}

	/**
	 * @see Obj#getClassAtom()
	 */
	public String getClassAtom() {
		return STRS.NODELIST;
	}

	/**
	 * @see Obj#getClassName()
	 */
	public String getClassName() {
		return STRS.NODELIST;
	}

	public Node getNamedItem(String name) {
		if (name == null)
			return null;

		String aName = name.intern();

		Node child = mElement.getFirstXFAChild();
		while (child != null) {
			if (aName == child.getName()) {
				return child;
			}
			child = child.getNextXFASibling();
		}
		return null;
	}
	/**
	 * @see NodeList#getNamedItem(String, String, int)
	 */
	@FindBugsSuppress(code="ES")
	public Node getNamedItem(String aName, String aClassName,
			int nTargetOccurrence/* 0 */) {
		if (aName == null && aClassName == null)
			return null;

		int nOccurrence = 0;

		Node tmp = mElement.getFirstXFAChild();
		while (tmp != null) {
			boolean bMatched = false;

			if (aName != null && aClassName != null) {
				if (aClassName == tmp.getClassAtom() && aName == tmp.getPrivateName())
					bMatched = true;
			} else if (aClassName != null && aClassName == tmp.getClassAtom())
				bMatched = true;
			else if (aName != null && aName == tmp.getPrivateName())
				bMatched = true;

			if (bMatched) {
				if (nOccurrence == nTargetOccurrence)
					return tmp;
				else
					nOccurrence++;
			}
			tmp = tmp.getNextXFASibling();
		}
		return null;
	}
	
	/**
	 * @see NodeList#getOccurrence(Node)
	 */
	@FindBugsSuppress(code="ES")
	Integer getOccurrence(Node oNode) {
	    int nOccurrence = 0;
	    
	    String aName = oNode.getName();
	    String aClassName = oNode.getClassAtom();

		Node child = mElement.getFirstXFAChild();
		while (child != null) {

	        if (aClassName == child.getClassAtom() && aName == child.getName()) {
	            if (oNode == child)
	                return Integer.valueOf(nOccurrence);
	            nOccurrence++;
	        }
	        child = child.getNextXFASibling();
	    }
		return null;
	}

	/**
	 * @see ListBase#insert(Obj, Obj)
	 */
	public void insert(Obj newNode, Obj refNode) {
		if (isReadOnly() || newNode == null)
			return;

		if (refNode == null)
			throw new ExFull(ResId.InsertFailedException);

		if (!(newNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);

		if (!(refNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);
		
		mElement.insertChild((Node)newNode, (Node)refNode, true);
		reset();
	}

	/**
	 * @see ListBase#item(int)
	 */
	public Obj item(int index) {
		if (index < mLastPosn) {
			mLastPosn = 0;
			mLastNode = mElement.getFirstXFAChild();
		}
		while (index >= mLastPosn) {
			if (index == mLastPosn)
				return mLastNode;
			mLastNode = mLastNode.getNextXFASibling();
			mLastPosn++;
		}
		return null;
	}

	public int length() {
		if (mLength == -1) {
			Node child = mElement.getFirstXFAChild();
			mLength = 0;
			while (child != null) {
    			mLength++;
    		    child = child.getNextXFASibling();
			}
		}
		return mLength;
	}

	/**
	 * @see ListBase#remove(Obj)
	 */
	public void remove(Obj removeNode) {
		if (isReadOnly() || removeNode == null)
			return;

		if (!(removeNode instanceof Node))
			throw new ExFull(ResId.ArgumentMismatchException);
		
		Node node = (Node)removeNode;
		
		if (node.getXFAParent() != mElement)
			throw new ExFull(ResId.RemoveFailedException); // child not found!
		
		node.remove();
		reset();
	}

	private void reset() {
		mLastNode = mElement.getFirstXFAChild();
		mLastPosn = 0;
		mLength = -1;
	}
	
	/* @exclude from published api. */
	boolean appendPermsCheck() {
		// Check permissions on the parent node and all its ancestors.
		
		if (mElement != null) {
			if (!mElement.checkAncestorPerms())
				return false;
		}

		return true;
	}
	
	/* @exclude from published api. */
	boolean removePermsCheck(Obj obj) {
		// Check permissions on the parent node and all its ancestors.
		// Check permissions on the child node and all its decendents.

		if (mElement != null) {
			if (!mElement.checkAncestorPerms())
				return false;
		}

		if (obj instanceof Node) {
			Node node = (Node)obj;

			if (!node.checkDescendentPerms())
				return false;
		}

		return true;
	}
	
	/* @exclude from published api. */
	boolean insertPermsCheck() {
		// Check permissions on the parent node and all its ancestors.

		if (mElement != null) {
			if (!mElement.checkAncestorPerms())
				return false;
		}

		return true;
	}
}