/*
 * Copyright 1997-2011 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.commons;

import org.apache.commons.lang.StringUtils;
import org.apache.sling.api.request.RequestPathInfo;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;

/**
 * Utility class for creating a sling {@link RequestPathInfo} by parsing a URL
 * path. Note that this does not check for the existence of the resource and
 * finding the longest-matching path as the default Sling resource resolution
 * does, it only works by parsing the raw string.
 *
 * <p>
 * The parsing structure looks like this:
 * <code>/resource/path.selector.ext/suffix</code>. The rules are (following the
 * Sling convention):
 *
 * <ul>
 * <li><b>extension</b> is everything between the last dot in the URL and the
 * next slash (or the end of the string)</li>
 * <li><b>suffix</b> is everything after the extension (including the slash)</li>
 * <li><b>resource path</b> is everything from the start up to the first dot
 * after the last slash (excluding extension and suffix)</li>
 * <li><b>selectors</b> are the dot-separated elements between the path and the
 * extension</li>
 * </ul>
 *
 */
public class PathInfo implements RequestPathInfo {

    private final String path;
    private final String selectorString;
    private final String[] selectors;
    private final String extension;
    private final String suffix;
    private Resource suffixResource;

    /**
     * Creates a {@link RequestPathInfo} by parsing the path part of a URL. This method does not support url path
     * suffixes containing dots (".").
     *
     * @param urlPath the path part of a URL
     */
    public PathInfo(String urlPath) {
        if (urlPath == null || !urlPath.startsWith("/")) {
            throw new IllegalArgumentException("Not an absolute path: " + urlPath);
        }
        int lastDot = urlPath.lastIndexOf(".");
        if (lastDot >= 0) {
            // pathToParse: /foo/bar.selector.two.ext/suf/fix
            // => extension = ext/suf/fix
            // => p         = /foo/bar.selector.two
            String extPlusSuffix = urlPath.substring(lastDot + 1);
            urlPath = urlPath.substring(0, lastDot);

            // => extension = ext/suf/fix
            // => suffix    = /suf/fix
            int firstSlash = extPlusSuffix.indexOf("/");
            if (firstSlash >= 0) {
                extension = extPlusSuffix.substring(0, firstSlash);
                suffix = extPlusSuffix.substring(firstSlash);
            } else {
                extension = extPlusSuffix;
                suffix = null;
            }

            String selectorStr;
            // 1) p = /foo/bar.selector.two
            // 2) p = dummypath
            int lastSlash = urlPath.lastIndexOf("/");
            if (lastSlash >= 0) {
                // (1)
                // => bar.selector.two (temporary, "bar" will be added to path below)
                selectorStr = urlPath.substring(lastSlash + 1);
                // => /foo/ (temporary)
                urlPath = urlPath.substring(0, lastSlash + 1);
            } else {
                // (2)
                // => dummypath
                selectorStr = urlPath;
            }

            // 1) selectorStr = bar.selector.two (temporary)
            int firstDot = selectorStr.indexOf(".");
            if (firstDot >= 0) {
                // => /foo/bar (final)
                urlPath = urlPath + selectorStr.substring(0, firstDot);
                // => selector.two (final)
                selectorStr = selectorStr.substring(firstDot + 1);
            } else {
                // /foo/bar.ext

                // /foo/bar
                // ext

                // /foo/
                // bar

                // => p = /foo/bar (final)
                urlPath = urlPath + selectorStr;
                // => selectorStr == null (final)
                selectorStr = null;
            }

            path = urlPath;

            if (selectorStr == null) {
                selectorString = null;
                selectors = new String[] {};
            } else {
                selectorString = selectorStr;
                selectors = selectorStr.split("\\.");
            }

        } else {
            path = urlPath;
            selectorString = null;
            selectors = new String[] {};
            extension = null;
            suffix = null;
        }
    }

    /**
     * Create a {@link RequestPathInfo} object using the given {@link ResourceResolver} and <code>urlPath</code>. The
     * resolver is used to map the underlying resource and reliably disassemble the urlPath into its segments.
     *
     * @param resolver The resource resolver.
     * @param urlPath  The url path.
     */
    public PathInfo(final ResourceResolver resolver, final String urlPath) {
        final Resource resource = resolver.resolve(urlPath);
        if (null != resource) {

            String info = resource.getResourceMetadata().getResolutionPathInfo();

            final int slashIndex = StringUtils.indexOf(info, "/");
            suffix = (slashIndex >= 0) ? StringUtils.substring(info, slashIndex + 1) : null;
            info = StringUtils.chomp(info, "/" + suffix);

            final int extIndex = StringUtils.lastIndexOf(info, ".");
            extension = (extIndex >= 0) ? StringUtils.substring(info, extIndex + 1) : null;

            info = StringUtils.chomp(info, "." + extension);

            selectorString = info;
            selectors = StringUtils.split(info, ".");

            path = resolver.map(resource.getPath());

            if (suffix != null) {
                suffixResource = resolver.getResource(suffix);
            }
        } else {
            path = urlPath;
            selectorString = null;
            selectors = new String[] {};
            extension = null;
            suffix = null;
            suffixResource = null;
        }
    }

    public String getResourcePath() {
        return path;
    }

    public String getSelectorString() {
        return selectorString;
    }

    public String[] getSelectors() {
        return selectors;
    }

    public String getExtension() {
        return extension;
    }

    public String getSuffix() {
        return suffix;
    }

    public Resource getSuffixResource() { return suffixResource; }

}
