/*******************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2014 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ******************************************************************************/
package com.day.cq.replication;

import javax.jcr.Session;

/**
 * Transforms the replication path for the replication request.
 * It provides a hook, while building and importing replication content package, that can map a given replication path
 * (i.e. /content/dam/skiing/images/Banner.JPG) to  new target replication path (i.e. /content/dam/campaigns/skiing/images/Banner.JPG)
 *
 * This can be used when you need to replicate between system where path hierarchies on one system do not exactly match to path
 * hierarchies on other system.
 *
 * Please note that it only allows for one-to-one mapping between source and target for a replication request.
 *
 * Implementations can choose using {@link #accepts(javax.jcr.Session, ReplicationAction, Agent)}, whether to map/transform given replication path
 * in the context of the current replication request.
 *
 *
 */
public interface ReplicationPathTransformer {
    /**
     * Transforms the replication path. Should return the same <code>replicationPath</code> in case of error
     * or if it can not be transform. i.e. an implementation map a given replication path
     * (i.e. /content/dam/skiing/images/Banner.JPG) to  new target replication path (i.e. /content/dam/campaigns/skiing/images/Banner.JPG)
     * or vice versa, depending on the current replication context i.e. replication vs reverse replication
     *
     * Implementations can choose using {@link #accepts(javax.jcr.Session, ReplicationAction, Agent)}, whether to map/transform given replication path
     * in the context of the current replication request.
     *
     * Replication Agent <code>agent</code> and Replication Action <code>action</code> is also available, based on the context in which
     * the transform is called.
     * i.e. in case of reverse replication, agent will not be null. So based on agent, transform method implementation may choose to do
     * reverse mapping by identifying that it is a reverse replication.
     *
     * @param session jcr session
     * @param replicationPath replication path that needs to be transformed
     * @param action  replication action if available
     * @param agent   replication agent if available in the current context. i.e. Replication agent is not available while importing
     *                a replication content package.
     * @return the transformed path
     */
    String transform(Session session, String replicationPath, ReplicationAction action, Agent agent);

    /**
     * Returns <code>true</code> if Transformer can transform replication path for given replication request.
     *
     * This allows implementation to choose which requests to intercept i.e. replicate request in case of reverse replication, would have the
     * {@link com.day.cq.replication.Agent}, so the implementation can check if it is the agent that implementation is interested in.
     * Or using {@link Session}, implementation can look at the current user context and decide whether a transformation is needed or not.
     *
     *
     * @param session jcr session
     * @param action  replication action if available
     * @param agent   replication agent if available in the current context. i.e. Replication agent is not available while importing
     *                a replication content package.
     * @return <code>true</code> if it can transform
     * <code>false</code> if it can not transform
     */
    boolean accepts(Session session, ReplicationAction action, Agent agent);
}
