/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.api;

import org.osgi.annotation.versioning.ProviderType;

import java.util.Calendar;
import java.util.List;

import javax.jcr.Node;
import javax.jcr.Session;
import javax.jcr.version.Version;

/**
 * <code>VersionManager</code> provides methods to manage page versions.
 */
@ProviderType
public interface VersionManager {

    /**
     * Returns a list of purge infos of the versions that are purged using the configured conditions.
     *
     * @param node of which the versions are to be purged
     * @param dryRun if <code>true</code> the versions are not removed
     * @return list of versions to purge
     */
    List<PurgeInfo> purgeVersions(Node node, boolean dryRun);

    /**
     * Returns a list of purge infos of the versions that are purged using the specified conditions.
     *
     * @param node of which the versions are to be purged
     * @param dryRun if <code>true</code> the versions are not removed
     * @param maxVersions max number of versions to retain
     * @param maxAge max age (in days) of versions to retain
     * @return list of versions to purge
     */
    List<PurgeInfo> purgeVersions(Node node, boolean dryRun, int maxVersions, int maxAge);

    /**
     * Returns a list of purge infos of the versions that are purged using the configured conditions.
     *
     * @param path of which the versions are to be purged
     * @param session to use for repository operations
     * @param dryRun if <code>true</code> the versions are not removed
     * @param recursive if versions for paths below given path should be purged
     * @return list of versions to purge
     */
    List<PurgeInfo> purgeVersions(Session session, String path, boolean dryRun, boolean recursive);

    /**
     * Returns a list of purge infos of the versions that are purged using the specified conditions.
     *
     * @param path of which the versions are to be purged
     * @param session to use for repository operations
     * @param dryRun if <code>true</code> the versions are not removed
     * @param recursive if versions for paths below given path should be purged
     * @param maxVersions max number of versions to retain
     * @param maxAge max age (in days) of versions to retain
     * @return list of versions to purge
     */
    List<PurgeInfo> purgeVersions(Session session, String path, boolean dryRun, boolean recursive, int maxVersions, int maxAge);

    /**
     * Returns a list of purge infos of the versions that are purged using the specified conditions.
     *
     * @param path of which the versions are to be purged
     * @param session to use for repository operations
     * @param dryRun if <code>true</code> the versions are not removed
     * @param recursive if versions for paths below given path should be purged
     * @param maxVersions max number of versions to retain
     * @param maxAge max age (in days) of versions to retain
     * @param minVersions min number of versions to retain if possible
     * @return list of versions to purge
     */
    List<PurgeInfo> purgeVersions(Session session, String path, boolean dryRun, boolean recursive, int maxVersions, int maxAge, int minVersions);

    /**
     * General info about version purging.
     */
    interface PurgeInfo {

        /**
         * Returns the name of the version
         * @return the name of the version
         */
        String getVersionName();

        /**
         * Returns the path of the version
         * @return the path of the version
         */
        String getVersionPath();

        /**
         * Returns the labels of the version
         * @return the labels of the version
         */
        String[] getVersionLabels();

        /**
         * Returns the creation date of the version
         * @return the creation date of the version
         */
        Calendar getVersionCreated();

        /**
         * Returns <code>true</code> if the version was not purged
         * @return <code>true</code> if the version was not purged
         */
        boolean isRetained();

        /**
         * Returns <code>true</code> if this is the current base version
         * @return <code>true</code> if this is the current base version
         */
        boolean isBaseVersion();

        /**
         * Returns an informative string of an error that occurred during purging, or <code>null</code>.
         * @return the error string or null.
         */
        String getError();

        /**
         * Returns the title of the versioned node if it is a page.
         * Otherwise it returns the node name.
         * @return The page's title or node's name
         */
        String getTitle();

        /**
         * Returns the node's path that this version belongs to.
         * @return The versioned node's path
         */
        String getVersionedNodePath();
    }
}