/*
 * Decompiled with CFR 0.152.
 */
package org.apache.jackrabbit.oak.plugins.index.lucene;

import com.google.common.base.Preconditions;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import org.apache.jackrabbit.oak.commons.PathUtils;
import org.apache.jackrabbit.oak.commons.PerfLogger;
import org.apache.jackrabbit.oak.plugins.index.AsyncIndexInfoService;
import org.apache.jackrabbit.oak.plugins.index.lucene.IndexCopier;
import org.apache.jackrabbit.oak.plugins.index.lucene.LuceneIndexDefinition;
import org.apache.jackrabbit.oak.plugins.index.lucene.LuceneIndexNode;
import org.apache.jackrabbit.oak.plugins.index.lucene.LuceneIndexNodeManager;
import org.apache.jackrabbit.oak.plugins.index.lucene.hybrid.NRTIndexFactory;
import org.apache.jackrabbit.oak.plugins.index.lucene.reader.DefaultIndexReaderFactory;
import org.apache.jackrabbit.oak.plugins.index.lucene.reader.LuceneIndexReaderFactory;
import org.apache.jackrabbit.oak.plugins.index.lucene.util.LuceneIndexHelper;
import org.apache.jackrabbit.oak.plugins.index.search.BadIndexTracker;
import org.apache.jackrabbit.oak.plugins.memory.EmptyNodeState;
import org.apache.jackrabbit.oak.spi.commit.CompositeEditor;
import org.apache.jackrabbit.oak.spi.commit.DefaultEditor;
import org.apache.jackrabbit.oak.spi.commit.Editor;
import org.apache.jackrabbit.oak.spi.commit.EditorDiff;
import org.apache.jackrabbit.oak.spi.commit.SubtreeEditor;
import org.apache.jackrabbit.oak.spi.mount.MountInfoProvider;
import org.apache.jackrabbit.oak.spi.mount.Mounts;
import org.apache.jackrabbit.oak.spi.state.EqualsDiff;
import org.apache.jackrabbit.oak.spi.state.NodeState;
import org.apache.jackrabbit.oak.spi.state.NodeStateUtils;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexTracker {
    private static final long AUTO_REFRESH_MILLIS = Long.getLong("oak.indexTracker.autoRefresh", 3153600000000L);
    private static final Logger log = LoggerFactory.getLogger(IndexTracker.class);
    private static final PerfLogger PERF_LOGGER = new PerfLogger(LoggerFactory.getLogger(IndexTracker.class.getName() + ".perf"));
    private final LuceneIndexReaderFactory readerFactory;
    private final NRTIndexFactory nrtFactory;
    private final BadIndexTracker badIndexTracker = new BadIndexTracker();
    private NodeState root = EmptyNodeState.EMPTY_NODE;
    private AsyncIndexInfoService asyncIndexInfoService;
    private volatile Map<String, LuceneIndexNodeManager> indices = Collections.emptyMap();
    private volatile boolean refresh;
    private volatile long nextAutoRefresh = System.currentTimeMillis() + AUTO_REFRESH_MILLIS;

    public IndexTracker() {
        this((IndexCopier)null);
    }

    public IndexTracker(IndexCopier cloner) {
        this((LuceneIndexReaderFactory)new DefaultIndexReaderFactory(Mounts.defaultMountInfoProvider(), cloner));
    }

    public IndexTracker(LuceneIndexReaderFactory readerFactory) {
        this(readerFactory, null);
    }

    public IndexTracker(LuceneIndexReaderFactory readerFactory, @Nullable NRTIndexFactory nrtFactory) {
        this.readerFactory = readerFactory;
        this.nrtFactory = nrtFactory;
    }

    public MountInfoProvider getMountInfoProvider() {
        return this.readerFactory.getMountInfoProvider();
    }

    public synchronized void close() {
        Map<String, LuceneIndexNodeManager> indices = this.indices;
        this.indices = Collections.emptyMap();
        for (Map.Entry<String, LuceneIndexNodeManager> entry : indices.entrySet()) {
            try {
                entry.getValue().close();
            }
            catch (IOException e) {
                log.error("Failed to close the Lucene index at " + entry.getKey(), e);
            }
        }
    }

    public synchronized void update(NodeState root) {
        if (this.refresh) {
            this.root = root;
            this.close();
            this.refresh = false;
            log.info("Refreshed the opened indexes");
        } else {
            this.diffAndUpdate(root);
            long now = System.currentTimeMillis();
            if (!this.refresh && now > this.nextAutoRefresh) {
                this.refresh = true;
                this.nextAutoRefresh = now + AUTO_REFRESH_MILLIS;
            }
        }
    }

    public void setAsyncIndexInfoService(AsyncIndexInfoService asyncIndexInfoService) {
        this.asyncIndexInfoService = asyncIndexInfoService;
    }

    public AsyncIndexInfoService getAsyncIndexInfoService() {
        return this.asyncIndexInfoService;
    }

    private synchronized void diffAndUpdate(final NodeState root) {
        if (this.asyncIndexInfoService != null && !this.asyncIndexInfoService.hasIndexerUpdatedForAnyLane(this.root, root)) {
            log.trace("No changed detected in async indexer state. Skipping further diff");
            this.root = root;
            return;
        }
        Map<String, LuceneIndexNodeManager> original = this.indices;
        final HashMap updates = Maps.newHashMap();
        HashSet<String> indexPaths = Sets.newHashSet();
        indexPaths.addAll(original.keySet());
        indexPaths.addAll(this.badIndexTracker.getIndexPaths());
        ArrayList<SubtreeEditor> editors = Lists.newArrayListWithCapacity(indexPaths.size());
        for (final String path : indexPaths) {
            editors.add(new SubtreeEditor((Editor)new DefaultEditor(){

                @Override
                public void leave(NodeState before, NodeState after) {
                    try {
                        if (IndexTracker.isStatusChanged(before, after) || IndexTracker.isIndexDefinitionChanged(before, after)) {
                            long start = PERF_LOGGER.start();
                            LuceneIndexNodeManager index = LuceneIndexNodeManager.open(path, root, after, IndexTracker.this.readerFactory, IndexTracker.this.nrtFactory);
                            PERF_LOGGER.end(start, -1L, "[{}] Index found to be updated. Reopening the LuceneIndexNode", (Object)path);
                            updates.put(path, index);
                        }
                    }
                    catch (IOException e) {
                        IndexTracker.this.badIndexTracker.markBadPersistedIndex(path, (Throwable)e);
                    }
                }
            }, Iterables.toArray(PathUtils.elements(path), String.class)));
        }
        EditorDiff.process(CompositeEditor.compose(editors), this.root, root);
        this.root = root;
        if (!updates.isEmpty()) {
            this.indices = ImmutableMap.builder().putAll(Maps.filterKeys(original, Predicates.not(Predicates.in(updates.keySet())))).putAll(Maps.filterValues(updates, Predicates.notNull())).build();
            this.badIndexTracker.markGoodIndexes(updates.keySet());
            for (final String path : updates.keySet()) {
                LuceneIndexNodeManager index = original.get(path);
                try {
                    if (index == null) continue;
                    index.close();
                }
                catch (IOException e) {
                    log.error("Failed to close Lucene index at " + path, e);
                }
            }
        }
    }

    public void refresh() {
        log.info("Marked tracker to refresh upon next cycle");
        this.refresh = true;
    }

    @Nullable
    public LuceneIndexNode acquireIndexNode(String path) {
        LuceneIndexNode indexNode;
        LuceneIndexNodeManager index = this.indices.get(path);
        LuceneIndexNode luceneIndexNode = indexNode = index != null ? index.acquire() : null;
        if (indexNode != null) {
            return indexNode;
        }
        return this.findIndexNode(path);
    }

    @Nullable
    private synchronized LuceneIndexNode findIndexNode(String path) {
        LuceneIndexNodeManager index = this.indices.get(path);
        if (index != null) {
            LuceneIndexNode indexNode = index.acquire();
            return Preconditions.checkNotNull(indexNode);
        }
        if (this.badIndexTracker.isIgnoredBadIndex(path)) {
            return null;
        }
        NodeState node = this.root;
        for (String name : PathUtils.elements(path)) {
            node = node.getChildNode(name);
        }
        try {
            if (LuceneIndexHelper.isLuceneIndexNode(node)) {
                index = LuceneIndexNodeManager.open(path, this.root, node, this.readerFactory, this.nrtFactory);
                if (index != null) {
                    LuceneIndexNode indexNode = index.acquire();
                    Preconditions.checkNotNull(indexNode);
                    this.indices = ImmutableMap.builder().putAll(this.indices).put(path, index).build();
                    this.badIndexTracker.markGoodIndex(path);
                    return indexNode;
                }
            } else if (node.exists()) {
                log.warn("Cannot open Lucene Index at path {} as the index is not of type {}", (Object)path, (Object)"lucene");
            }
        }
        catch (Throwable e) {
            this.badIndexTracker.markBadIndexForRead(path, e);
        }
        return null;
    }

    @Nullable
    public LuceneIndexDefinition getIndexDefinition(String indexPath) {
        LuceneIndexNodeManager indexNodeManager = this.indices.get(indexPath);
        if (indexNodeManager != null) {
            return indexNodeManager.getDefinition();
        }
        NodeState node = NodeStateUtils.getNode(this.root, indexPath);
        if (!node.exists()) {
            return null;
        }
        if (!node.hasChildNode(":index-definition")) {
            return null;
        }
        if (!LuceneIndexHelper.isLuceneIndexNode(node)) {
            return null;
        }
        return new LuceneIndexDefinition(this.root, node, indexPath);
    }

    public Set<String> getIndexNodePaths() {
        return this.indices.keySet();
    }

    public BadIndexTracker getBadIndexTracker() {
        return this.badIndexTracker;
    }

    public NodeState getRoot() {
        return this.root;
    }

    private static boolean isStatusChanged(NodeState before, NodeState after) {
        return !EqualsDiff.equals(before.getChildNode(":status"), after.getChildNode(":status"));
    }

    private static boolean isIndexDefinitionChanged(NodeState before, NodeState after) {
        return !EqualsDiff.equals(before.getChildNode(":index-definition"), after.getChildNode(":index-definition"));
    }
}

