/*******************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2013 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 ******************************************************************************/
package com.adobe.granite.keystore;

import java.security.Key;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.cert.Certificate;

import javax.net.ssl.KeyManager;
import javax.net.ssl.TrustManager;

import org.apache.sling.api.SlingIOException;
import org.apache.sling.api.resource.ResourceResolver;
import org.osgi.annotation.versioning.ProviderType;

/**
 * Provides secure access to {@code KeyManager} instances for accessing private keys and the system-wide {@code TrustManager} for validating
 * signatures and certificates.
 */
@ProviderType
public interface KeyStoreService {

    /**
     *  Name of the KeyStoreService service
     */
    String NAME = "com.adobe.granite.keystore.KeyStoreService";

    /**
     * Returns the private {@code KeyManager} of the user identified by the given {@link org.apache.sling.api.resource.ResourceResolver}.
     *
     * @param resolver a resolver identifying the user whose {@link KeyStore} is used for building the {@link KeyManager}
     * @return the private key manager of the user identified by the given resolver; this method does not return {@code null}
     * @throws SlingIOException                if an error occurs instantiating the {@link KeyManager} with data from persistence
     * @throws SecurityException               if the underlying key store cannot be instantiated due to security errors
     * @throws IllegalArgumentException        if any of the parameters are null
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    KeyManager getKeyManager(ResourceResolver resolver) throws SlingIOException, SecurityException, IllegalArgumentException,
            KeyStoreNotInitialisedException;

    /**
     * <p>
     * Returns the raw {@link KeyStore} owned by the user identified by the {@link ResourceResolver}. This key store allows managing the
     * keys in the store.
     * </p>
     * <p>
     * The {@code load} and {@code store} methods of the returned {@link KeyStore} instance always throw a {@link
     * java.security.KeyStoreException} because the actual storage is managed by the {@link KeyStoreService} service behind the scenes.
     * </p>
     * <p>
     * The {@link ResourceResolver} has to be open as long as write operations are performed with the {@link KeyStore}.
     * </p>
     * @param resolver a resolver identifying the user for which the {@link KeyStore} will be obtained
     * @return the key store owned by the user identified by the given resolver; this method does not return {@code null}
     * @throws SlingIOException                if an error occurs loading the {@link KeyStore} data from persistence
     * @throws SecurityException               if the key store cannot be instantiated due to security errors
     * @throws IllegalArgumentException        if any of the parameters are null
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    KeyStore getKeyStore(ResourceResolver resolver) throws SlingIOException, SecurityException, IllegalArgumentException,
            KeyStoreNotInitialisedException;

    /**
     * <p>
     * Returns the raw {@link KeyStore} owned by the user whose user id is {@code userId}. The given resource resolver must have the proper
     * access rights in order to access the given user's key store. This key store allows managing the keys in the store.
     * </p>
     * <p>
     * The {@code load} and {@code store} methods of the returned {@link KeyStore} instance always throw a {@link
     * java.security.KeyStoreException} because the actual storage is managed by the {@link KeyStoreService} service behind the scenes.
     * </p>
     * <p>
     * The {@link ResourceResolver} has to be open as long as write operations are performed with the {@link KeyStore}.
     * </p>
     * @param resolver the resource resolver to use for accessing the key store owned by the user given as parameter
     * @param userId   the ID of the user who owns the returned key store
     * @return the key store owned by the user identified by the given user ID. {@code null} is never returned.
     * @throws SlingIOException                if an error occurs loading the {@link KeyStore} data from persistence
     * @throws SecurityException               if the key store cannot be instantiated due to security errors
     * @throws IllegalArgumentException        if any of the parameters are null
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    KeyStore getKeyStore(ResourceResolver resolver, String userId) throws SlingIOException, SecurityException, IllegalArgumentException,
            KeyStoreNotInitialisedException;

    /**
     * Returns the system wide {@link TrustManager}.
     *
     * @param resolver a resolver identifying a user allowed to access the {@link TrustManager}
     * @return the global trust manager; this method does not return {@code null}
     * @throws SlingIOException                if an error occurs instantiating the {@link TrustManager} with data from persistence
     * @throws SecurityException               if the {@link ResourceResolver} is not allowed to access the global trust store
     * @throws IllegalArgumentException        if the {@code resolver} is {@code null}
     * @throws KeyStoreNotInitialisedException if the trust store has not been created
     * (see {@link #createTrustStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    TrustManager getTrustManager(ResourceResolver resolver) throws SlingIOException, SecurityException, KeyStoreNotInitialisedException;

    /**
     * <p>
     * Returns the global trust store if it's accessible to the user identified by the {@link ResourceResolver}. This key store allows
     * managing the keys in the store.
     * </p>
     * <p>
     * The {@code load} and {@code store} methods of the returned {@link KeyStore} instance always throw a {@link
     * java.security.KeyStoreException} because the actual storage is managed by the {@link KeyStoreService} service behind the scenes.
     * </p>
     * <p>
     * The {@link ResourceResolver} has to be open as long as write operations are performed with the {@link KeyStore}.
     * </p>
     * @param resolver a resolver identifying a user allowed to access the global trust store
     * @throws SlingIOException                if an error occurs loading the {@link KeyStore} data from persistence
     * @throws SecurityException               if the {@link ResourceResolver} is not allowed access to the global trust store or if the
     *                                         store cannot be instantiated due to security errors
     * @throws IllegalArgumentException        if any of the parameters are null
     * @throws KeyStoreNotInitialisedException if the trust store has not been created
     * (see {@link #createTrustStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    KeyStore getTrustStore(ResourceResolver resolver) throws SlingIOException, SecurityException, IllegalArgumentException,
            KeyStoreNotInitialisedException;

    /**
     * Changes the password used to protect a user's key store. For security reasons the password char arrays parameters should be cleaned
     * after successfully calling this method (see {@link java.util.Arrays#fill(char[], char)}).
     *
     * @param resolver        a resource resolver to use for accessing the key store owned by the user given as parameter
     * @param userId          the ID of the user who owns the key store
     * @param currentPassword the key store's current password; please make sure to clean this array after calling this method for security
     *                        purposes
     * @param newPassword     the key store's new password; please make sure to clean this array after calling this method for security
     *                        purposes
     * @throws SlingIOException                if an error occurs loading the {@link KeyStore} data from persistence
     * @throws SecurityException               if the {@code currentPassword} doesn't match the key store's password or if the key store
     *                                         cannot be instantiated due to security errors
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    void changeKeyStorePassword(ResourceResolver resolver, String userId, char[] currentPassword, char[] newPassword) throws
            SlingIOException, SecurityException, KeyStoreNotInitialisedException;

    /**
     * Changes the password used to protect the global trust store. For security reasons the password char arrays parameters should be
     * cleaned after successfully calling this method (see {@link java.util.Arrays#fill(char[], char)}).
     *
     * @param resolver        a resource resolver with write access to the trust store
     * @param currentPassword the trust store's current password; please make sure to clean this array after calling this method for
     *                        security purposes
     * @param newPassword     the trust store's new password; please make sure to clean this array after calling this method for security
     *                        purposes
     * @throws SlingIOException                if an error occurs loading the trust store data from persistence
     * @throws SecurityException               if the {@code currentPassword} doesn't match the key store's password or if the passed
     *                                         resolver doesn't have enough rights to modify the trust store
     * @throws KeyStoreNotInitialisedException if the trust store has not been created
     * (see {@link #createTrustStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    void changeTrustStorePassword(ResourceResolver resolver, char[] currentPassword, char[] newPassword) throws SlingIOException,
            SecurityException, KeyStoreNotInitialisedException;

    /**
     * Saves a {@code KeyPair} in the {@code KeyStore} owned by the user whose user id is {@code userId}, under the specified alias. The
     * given resource resolver must have the proper access rights in order to access the given user's key store. If an entry already exists
     * for the specified alias, it is overridden.
     *
     * @param resolver the resource resolver to use for accessing the key store owned by the user given as parameter
     * @param userId   the id of the user who owns the key store
     * @param keyPair  the {@code KeyPair} to save
     * @param alias    save the {@code KeyPair} under this alias
     * @throws SlingIOException                if an error occurs loading the {@link KeyStore} data from persistence
     * @throws SecurityException               if the key store cannot be instantiated due to security errors
     * @throws IllegalArgumentException        if any of the parameters are null
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    void addKeyStoreKeyPair(ResourceResolver resolver, String userId, KeyPair keyPair, String alias) throws SlingIOException,
            SecurityException, KeyStoreNotInitialisedException;

    /**
     * Retrieve a {@code KeyPair} from the {@code KeyStore} owned by the user whose user id is {@code userId}, stored under the specified
     * alias. The given resource resolver must have the proper access rights in order to access the given user's key store.
     *
     * @param resolver the resource resolver to use for accessing the key store owned by the user given as parameter
     * @param userId   the id of the user who owns the key store
     * @param alias    retrieve the {@code KeyPair} stored under this alias
     * @return the {@code KeyPair} or {@code null} if no {@code KeyPair} was stored with the given alias.
     * @throws SlingIOException                if an error occurs loading the {@link KeyStore} data from persistence
     * @throws SecurityException               if the key store cannot be instantiated due to security errors
     * @throws IllegalArgumentException        if any of the parameters are null
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    KeyPair getKeyStoreKeyPair(ResourceResolver resolver, String userId, String alias) throws SlingIOException, SecurityException,
            KeyStoreNotInitialisedException;

    /**
     * <p>
     * Creates and returns a raw {@code KeyStore} for the user whose user id is {@code userId}. For security reasons the password char
     * array parameter should be cleaned after successfully calling this method (see {@link java.util.Arrays#fill(char[], char)}).
     * </p>
     * <p>
     * The {@code load} and {@code store} methods of the returned {@link KeyStore} instance always throw a {@link
     * java.security.KeyStoreException} because the actual storage is managed by the {@link KeyStoreService} service behind the scenes.
     * </p>
     * <p>
     * The {@link ResourceResolver} has to be open as long as write operations are performed with the {@link KeyStore}. Also, it must have
     * the proper access rights in order to create the given user's key store.
     * </p>
     * Calling this method several times with the same parameters has no effect on an already created key store. However, calling this
     * method with a different password will throw a {@link org.apache.sling.api.SlingIOException}.
     *
     * @param resolver a resolver that will be used to create the {@link KeyStore}
     * @param userId the user for which the {@link KeyStore} will be created
     * @param password the password used for protecting the returned {@link java.security.KeyStore}; please make sure to clean this array
     *                 after calling this method for security purposes
     * @return the key store owned by the user whose user id is {@code userId}; this method does not return {@code null}
     * @throws SlingIOException         if an error occurs storing the {@link KeyStore} data to the persistence layer
     * @throws SecurityException        if the key store cannot be instantiated due to security errors
     * @throws IllegalArgumentException if any of the parameters are null
     */
    KeyStore createKeyStore(ResourceResolver resolver, String userId, char[] password) throws SlingIOException, SecurityException,
            IllegalArgumentException;

    /**
     * <p>
     * Creates and returns a raw {@code KeyStore} for the user identified by the {@code resolver}. For security reasons the password char
     * array parameter should be cleaned after successfully calling this method (see {@link java.util.Arrays#fill(char[], char)}).
     * </p>
     * <p>
     * The {@code load} and {@code store} methods of the returned {@link KeyStore} instance always throw a {@link
     * java.security.KeyStoreException} because the actual storage is managed by the {@link KeyStoreService} service behind the scenes.
     * </p>
     * <p>
     * The {@link ResourceResolver} has to be open as long as write operations are performed with the {@link KeyStore}.
     * </p>
     * <p>
     * Calling this method several times with the same parameters has no effect on an already created key store. However, calling this
     * method with a different password will throw a {@link org.apache.sling.api.SlingIOException}.
     * </p>
     * @param resolver a resolver identifying the user for which the {@link KeyStore} will be created
     * @param password the password used for protecting the returned {@link java.security.KeyStore}; please make sure to clean this array
     *                 after calling this method for security purposes
     * @return key store owned by the user identified by the given resolver; this method does not return {@code null}
     * @throws SlingIOException         if an error occurs storing the {@link KeyStore} data to the persistence layer
     * @throws SecurityException        if the key store cannot be instantiated due to security errors
     * @throws IllegalArgumentException if any of the parameters are null
     */
    KeyStore createKeyStore(ResourceResolver resolver, char[] password) throws SlingIOException, SecurityException,
            IllegalArgumentException;

    /**
     * <p>
     * Creates and returns the global trust store. The {@code resolver} provided should have enough access rights in order to successfully
     * create the trust store. For security reasons the password char array parameter should be cleaned after successfully calling this
     * method (see {@link java.util.Arrays#fill(char[], char)}).
     * </p>
     * <p>
     * The {@code load} and {@code store} methods of the returned {@link KeyStore} instance always throw a {@link
     * java.security.KeyStoreException} because the actual storage is managed by the {@link KeyStoreService} service behind the scenes.
     * </p>
     * <p>
     * The {@link ResourceResolver} has to be open as long as write operations are performed with the {@link KeyStore}.
     * </p>
     * <p>
     * Calling this method several times with the same parameters has no effect on an already created trust store. However, calling this
     * method with a different password will throw a {@link org.apache.sling.api.SlingIOException}.
     * </p>
     * @param resolver a resolver identifying a user allowed to create the global trust store
     * @param password the password used for protecting the returned {@link java.security.KeyStore}; please make sure to clean this array
     *                 after calling this method for security purposes
     * @return the global trust store; this method does not return {@code null}
     * @throws SlingIOException         if an error occurs storing the {@link KeyStore} data to the persistence layer
     * @throws SecurityException        if the {@link ResourceResolver} is not allowed access to the global trust store location or if the
     *                                  store cannot be instantiated due to security errors
     * @throws IllegalArgumentException if any of the parameters are null
     */
    KeyStore createTrustStore(ResourceResolver resolver, char[] password) throws SlingIOException, SecurityException,
            IllegalArgumentException;

    /**
     * Checks if a key store was created for the user whose user id is {@code userId}. The given resource resolver must have the proper
     * access rights in order to check the existence of the given user's key store.
     *
     * @param resolver the resource resolver to use for checking the existence of a key store for the user whose user id is {@code userId}
     * @param userId   the id of the user who owns the checked key store
     * @return {@code true} if the key store exists, {@code false} otherwise
     */
    boolean keyStoreExists(ResourceResolver resolver, String userId);

    /**
     * Checks if the global trust store was created.
     *
     * @param resolver the resolver used for checking the trust store's existence
     * @return {@code true} if the trust store exists, {@code false} otherwise
     */
    boolean trustStoreExists(ResourceResolver resolver);

    /**
     * Adds a private key entry with a certificate chain to the key store owned by the user with the user id equal to {@code userId}. This
     * method makes sure the entries are protected with the store's password for ease of use instead of requiring a password for each key.
     *
     * @param resolver a resource resolver to use for accessing the key store owned by the user given as parameter
     * @param userId   the id of the user who owns the key store
     * @param alias    the alias under which the key entry will be saved in the key store
     * @param key      the private key that will be added to the key store
     * @param chain    the private key's certificate chain
     * @throws SecurityException if the underlying key store cannot be instantiated due to security errors
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    void addKeyStoreKeyEntry(ResourceResolver resolver, String userId, String alias, Key key,
                             Certificate[] chain) throws SecurityException, KeyStoreNotInitialisedException;

    /**
     * Retrieves a protected entry from the key store owned by the user with the user id equal to {@code userId} for entries stored using
     * the key store's password
     * (see {@link com.adobe.granite.keystore.KeyStoreService#addKeyStoreKeyEntry(org.apache.sling.api.resource.ResourceResolver,
     * String, String, java.security.Key, java.security.cert.Certificate[])} .
     *
     * @param resolver a resource resolver to use for accessing the key store owned by the user given as parameter
     * @param userId   the id of the user who owns the key store
     * @param alias    the alias under which the key entry will be saved in the key store
     * @return the entry stored in the key store under {@code alias}
     * @throws SecurityException               if the underlying key store cannot be instantiated due to security errors
     * @throws KeyStoreNotInitialisedException if the key store has not been created
     * (see {@link #createKeyStore(org.apache.sling.api.resource.ResourceResolver, char[])})
     */
    KeyStore.Entry getKeyStoreEntry(ResourceResolver resolver, String userId, String alias) throws SecurityException,
            KeyStoreNotInitialisedException;
}
