/*************************************************************************
 *
 *	File: DataMatrixX12Compactor.java
 *
 **************************************************************************
 * 
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2011 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.adobe.xfa.pmp.datamatrixpmp;

import java.util.ArrayList;
import java.util.List;

/**
 * Ported from DataMatrixX12Compactor.cpp
 */
class DataMatrixX12Compactor extends DataMatrixBaseCompactor {
	static class C40Encoding {
		int messageValue;
		List<Integer> c40Value = new ArrayList<Integer>();
	}

	static class CharacterSet {
		int messageValue;
		int codeValue;

		CharacterSet(int messageValue, int codeValue) {
			super();
			this.messageValue = messageValue;
			this.codeValue = codeValue;
		}
	}

	static final CharacterSet[] basicSet = { new CharacterSet(13, 0), // X12
																		// segment
																		// terminator
																		// <CR>
			new CharacterSet(42, 1), // X12 segment separator *
			new CharacterSet(62, 2), // X12 subelement separator >
			new CharacterSet(32, 3), // space
			new CharacterSet(48, 4), // 0
			new CharacterSet(49, 5), // 1
			new CharacterSet(50, 6), // 2
			new CharacterSet(51, 7), // 3
			new CharacterSet(52, 8), // 4
			new CharacterSet(53, 9), // 5
			new CharacterSet(54, 10), // 6
			new CharacterSet(55, 11), // 7
			new CharacterSet(56, 12), // 8
			new CharacterSet(57, 13), // 9
			new CharacterSet(65, 14), // A
			new CharacterSet(66, 15), // B
			new CharacterSet(67, 16), // C
			new CharacterSet(68, 17), // D
			new CharacterSet(69, 18), // E
			new CharacterSet(70, 19), // F
			new CharacterSet(71, 20), // G
			new CharacterSet(72, 21), // H
			new CharacterSet(73, 22), // I
			new CharacterSet(74, 23), // J
			new CharacterSet(75, 24), // K
			new CharacterSet(76, 25), // L
			new CharacterSet(77, 26), // M
			new CharacterSet(78, 27), // N
			new CharacterSet(79, 28), // O
			new CharacterSet(80, 29), // P
			new CharacterSet(81, 30), // Q
			new CharacterSet(82, 31), // R
			new CharacterSet(83, 32), // S
			new CharacterSet(84, 33), // T
			new CharacterSet(85, 34), // U
			new CharacterSet(86, 35), // V
			new CharacterSet(87, 36), // W
			new CharacterSet(88, 37), // X
			new CharacterSet(89, 38), // Y
			new CharacterSet(90, 39) // Z
	};

	// ////////////////////////////////////////////////////////////////////
	/**
	 * Do the byte compaction.
	 * 
	 * @param message
	 *            - The input binary message.
	 * @returns A vector of the of the binary encoded message in PDF417 code
	 *          words.
	 */
	// ////////////////////////////////////////////////////////////////////
	void compact(char[] message) {
		m_codeWords.clear();
		m_valid = false;

		C40Encoding[] c40Encoding = new C40Encoding[message.length];

		int messageSize = message.length;
		int idx;
		for (idx = 0; idx < messageSize; idx++) {
			int v = message[idx];

			int jdx;
			boolean found = false;
			for (jdx = 0; (jdx < basicSet.length) && !found; jdx++) {
				if (v == basicSet[jdx].messageValue) {
					c40Encoding[idx] = new C40Encoding();
					c40Encoding[idx].messageValue = v;
					c40Encoding[idx].c40Value.add(basicSet[jdx].codeValue);
					found = true;
					continue;
				}
			}

			// Can't use X12 encoding for this message.
			if (!found)
				return;
		}

		// Concatenate the c40 codewords
		List<Integer> textCodeWords = new ArrayList<Integer>();
		for (idx = 0; idx < messageSize; idx++) {
			int jdx;
			for (jdx = 0; jdx < c40Encoding[idx].c40Value.size(); jdx++)
				textCodeWords.add(c40Encoding[idx].c40Value.get(jdx));
		}

		int c40Size = textCodeWords.size();

		if (c40Size < 3) {
			// For small messages, just use ASCII encoding
			return;
		}

		// The C40 encoding must end on a mod 3 boundary
		int c40End = messageSize - 1;
		while ((c40Size % 3 != 0) && (c40End > 0)) {
			c40Size -= c40Encoding[c40End].c40Value.size();
			c40End--;
		}

		c40End++;

		if (c40End > 0)
			m_codeWords.add(238); // X12 latch

		// Encode the real code words
		for (idx = 0; idx < c40Size; idx += 3) {
			int value16 = textCodeWords.get(idx) * 1600
					+ textCodeWords.get(idx + 1) * 40
					+ textCodeWords.get(idx + 2) + 1;
			m_codeWords.add((value16 / 256));
			m_codeWords.add((value16 % 256));
		}

		// To properly handle the remaining characters, we need to
		// know what symbol size might contain the data.
		int tempSymbolSize = findSymbolSize(m_codeWords.size());
		if (tempSymbolSize < 0)
			return; // Can't fit the data in a symbol

		// How much room is left in the symbol?
		int remainSymbolCodeWords = DataMatrixInfo.DATA_SYMBOLS[tempSymbolSize]
				- m_codeWords.size();

		// How much data remains?
		int remainTextCodeWords = 0;
		for (idx = c40End; idx < messageSize; idx++)
			remainTextCodeWords += c40Encoding[idx].c40Value.size();

		// Rule from standard:
		// If two symbol characters remain and three C40 values remain
		// to be encoded (which may include both data and shift chracters)
		// encode the three C40 values in the last symbol characters.
		// A final Unlatch codeword is not required.
		if (remainTextCodeWords == 0) {
			if (remainSymbolCodeWords > 0) {
				// Unlatch to ASCII.
				m_codeWords.add(254);

				// Add padding.
				DataMatrixPadder.addPadding(m_codeWords, tempSymbolSize);

				m_symbolSize = tempSymbolSize;
				m_valid = true;
			} else {
				// The message just fits!
				m_symbolSize = tempSymbolSize;
				m_valid = true;
			}
		} else
		// Rule from standard:
		// If one symbol character remains and one C40 value
		// (data character) remains to be encoded, the last symbol
		// character is encoded with the data character using
		// ASCII encodation scheme. The Unlatch character is not
		// encoded, but is assumed, before the last symbol character
		// whenever an odd number of symbol characters remain in
		// the symbol in the C40 encodation scheme.
		if ((remainSymbolCodeWords == 1) && (remainTextCodeWords == 1)) {
			// Encode the last character using ASCII, no need to unlatch, or
			// padding.
			if (c40Encoding[c40End].messageValue < 128) {
				m_codeWords.add(c40Encoding[c40End].messageValue + 1);

				m_symbolSize = tempSymbolSize;
				m_valid = true;
			} else {
				// We need to shift the ASCII encoding.
				m_codeWords.add(254); // Unlatch
				m_codeWords.add(235); // Upper shift
				m_codeWords.add(c40Encoding[c40End].messageValue - 127);

				// Find the symbol size that will fit the data.
				m_symbolSize = findSymbolSize(m_codeWords.size());
				if (m_symbolSize == -1)
					return; // Can't fit the data

				// Pad the data. No need to unlatch.
				DataMatrixPadder.addPadding(m_codeWords, m_symbolSize);

				m_valid = true;
			}
		} else
		// Use ASCII encoding for the remaining codewords.
		{
			// Encode the last characters as ASCII
			m_codeWords.add(254); // Unlatch
			for (idx = c40End; idx < messageSize; idx++) {
				int v = c40Encoding[idx].messageValue;
				if (v > 127) {
					v = v - 128;
					m_codeWords.add(235); // Upper shift
				}
				m_codeWords.add(v + 1);
			}

			// Find the symbol size that will fit the data.
			m_symbolSize = findSymbolSize(m_codeWords.size());
			if (m_symbolSize == -1)
				return; // Can't fit the data

			// Pad the data. No need to unlatch.
			DataMatrixPadder.addPadding(m_codeWords, m_symbolSize);

			m_valid = true;
		}
	}
}
