/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2015 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.personalization.ui.items;

import java.util.Calendar;
import java.util.Collection;
import java.util.Set;

import org.osgi.annotation.versioning.ProviderType;

import com.day.cq.personalization.Location;

/**
 * Represents an interface for datasource items related to personalization such
 * as activities and offers and is intended to be used by UI.
 */
@ProviderType
public interface PersonalizationItem {

    /**
     * Low priority constant
     */
    static int PRIORITY_LOW = 0;

    /**
     * Normal priority constant
     */
    static int PRIORITY_NORMAL = 1;

    /**
     * High priority constant
     */
    static int PRIORITY_HIGH = 2;

    /**
     * Returns the path.
     * 
     * @return Path of the resource.
     */
    String getPath();

    /**
     * Returns the resource type.
     * 
     * @return Resource type of the resource. If the resource is of type cq:Page
     *         the resource type of cq:PageContent will be returned.
     */
    String getResourceType();

    /**
     * Returns the title
     * 
     * @return Title of the resource or the resource name if title is not
     *         available.
     */
    String getTitle();

    /**
     * Returns the composed URL for navigating to the item. The URL is composed
     * as follows <code>{request.url}.html{resource.path}</code>.
     * 
     * @return Composed URL for navigating to the item.
     */
    String getNavigatorHref();

    /**
     * Returns the thumbnail path.
     * 
     * @return Path to the resource thumbnail.
     */
    String getThumbnailPath();

    /**
     * Returns the activity type.
     * 
     * @return One of the activity types such as 'ab', 'aem', 'xt' or
     *         {@code null} if resource is not an activity.
     */
    String getActivityType();

    /**
     * Returns the activity sync status.
     * 
     * @return One of the activity sync states such as'synced',
     *         'synced-with-errors', 'not-synced' or {@code null} if resource is
     *         not an activity.
     */
    String getSyncStatus();

    /**
     * Returns the last synchronization error in an untranslated form as
     * returned by the remote service.
     * 
     * @return The synchronization error or {@code null}
     */
    String getSyncError();

    /**
     * Returns the activity active status.
     * 
     * @return One of the activity active states such as 'live' or 'inactive' or
     *         {@code null} if the resource is not an activity.
     */
    String getActiveStatus();

    /**
     * Returns the activity priority.
     * 
     * <dl>
     *  <dt>{@link PersonalizationItem#PRIORITY_LOW}</dt><dd>Low priority.</dd>
     *  <dt>{@link PersonalizationItem#PRIORITY_NORMAL}</dt><dd>Normal priority. Default value.</dd>
     *  <dt>{@link PersonalizationItem#PRIORITY_HIGH}</dt><dd>High priority.</dd>
     * </dl>
     * 
     * @return Priority
     */
    int getPriority();

    /**
     * Returns a collection of {@link Location}'s related to this activity.
     * 
     * @return A collection of location's
     */
    Collection<Location> getLocations();

    /**
     * Returns the path to the target configuration.
     * 
     * @return Path to target configuration or {@code null} if target is not
     *         configured for this activity.
     */
    String getTargetConfigurationPath();

    /**
     * Returns the modification date or {@code null} if this information is not
     * available.
     * 
     * @return Modification date or {@code null}
     */
    Calendar getModifiedDate();

    /**
     * Returns the formatted user name by whom the item has been modified.
     * 
     * @return Formatted user name or identifier if an error occurred.
     */
    String getModifiedBy();

    /**
     * Returns the published date or {@code null} if this information is not
     * available.
     * 
     * @return Publication date or {@code null}
     */
    Calendar getPublishedDate();

    /**
     * Returns the formatted user name by whom the item has been published.
     * 
     * @return Formatted user name or identifier if an error occurred.
     */
    String getPublishedBy();

    /**
     * Returns the date the activity is set to active.
     * 
     * @return On date or {@code null}
     */
    Calendar getOnTime();

    /**
     * Returns the date the activity is set to inactive.
     * 
     * @return Off date or {@code null}
     */
    Calendar getOffTime();

    /**
     * Returns a set of relation identifiers for the quick actions.
     * 
     * @return Set of relation identifiers
     */
    Set<String> getQuickactionsRel();

    /**
     * Returns if the item is an activity.
     * 
     * @return {@code true} if the item is an activity, {@code false} otherwise
     */
    boolean isActivity();

    /**
     * Returns if the item is active.
     * 
     * @return {@code true} if the item is active, {@code false} otherwise
     */
    boolean isActive();

    /**
     * Returns if the item is a {@link com.day.cq.wcm.api.Page}.
     * 
     * @return {@code true} if the item is a {@code cq:Page}, {@code false} otherwise
     */
    boolean isPage();

    /**Returns if the item is a {@link com.day.cq.wcm.msm.api.LiveCopy}.
     * 
     * @return {@code true} if the item is a {@code LiveCopy}, {@code false} otherwise.
     */
    boolean isLiveCopy();

}
