/*
 * File: CosOptimizer.java
 *
 * ****************************************************************************
 *
 *	ADOBE CONFIDENTIAL
 *	___________________
 *
 *	Copyright 2005-2006 Adobe Systems Incorporated
 *	All Rights Reserved.
 *
 *	NOTICE: All information contained herein is, and remains the property of
 *	Adobe Systems Incorporated and its suppliers, if any. The intellectual
 *	and technical concepts contained herein are proprietary to Adobe Systems
 *	Incorporated and its suppliers and may be covered by U.S. and Foreign
 *	Patents, patents in process, and are protected by trade secret or
 *	copyright law. Dissemination of this information or reproduction of this
 *	material is strictly forbidden unless prior written permission is obtained
 *	from Adobe Systems Incorporated.
 *
 * ***************************************************************************/
package com.adobe.internal.pdftoolkit.core.cos;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import com.adobe.internal.pdftoolkit.core.exceptions.PDFCosParseException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFIOException;
import com.adobe.internal.pdftoolkit.core.exceptions.PDFSecurityException;
import com.adobe.internal.pdftoolkit.core.types.ASName;

/**
 * CosOptimizer provides routines optimizing PDF objects prior to saving.
 * For instance, finding and marking free objects that are no longer
 * referenced within a PDF.
 *
 * @author dpond
 */
final class CosOptimizer
{

	/**
	 * Searches a COS document for all referenced objects rooted in 'roots'.
	 * All objects which are not refernced in 'roots' are then marked
	 * free so that they will not be written out during the next save.
	 * Returns true if at least one object is marked as free.
	 *
	 * @param cosDoc - The COS document whose objects are to be checked.
	 * @param roots - An array of the CosContainers that are the roots
	 * 				  of the trees for which we want inclusion testing
	 * 				  performed.
	 * @return boolean , returns true if at least one object is freed.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 * @throws IOException 
	 */
	static boolean freeUnreferencedObjects(CosDocument cosDoc, CosContainer[] roots, boolean fixStreamLengths)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		// Find all of the referenced objects from "special" root containers.
		CosList referencedInfos = CosOptimizer.findReferencedIndirectObjects(roots, fixStreamLengths);
		CosList objStmInfos = cosDoc.getObjStmInfos();
		Iterator iter = objStmInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
			referencedInfos.add(stmInfo.getObjNum(), stmInfo);
		}
		CosList allInfos = cosDoc.buildObjectList(false);
		iter = allInfos.iterator();
		boolean objectsFreed = false;
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if (referencedInfos.get(info.getObjNum()) == null) {
				CosObject obj = info.getObject(false);
				if (obj != null)
					obj.close();
				info.markFree();
				objectsFreed = true;
			}
		}
		return objectsFreed;
	}

	/**
	 * Searches a COS document for all referenced objects rooted in 'roots'.
	 * All objects which are not refernced in 'roots' are then placed in
	 * an array and returned to the caller.
	 */
	static CosObject[] getUnreferencedObjects(CosDocument cosDoc, CosContainer[] roots)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		// Find all of the referenced objects from "special" root containers.
		CosList referencedInfos = CosOptimizer.findReferencedIndirectObjects(roots, false);
		CosList objStmInfos = cosDoc.getObjStmInfos();
		Iterator iter = objStmInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo stmInfo = (CosObjectInfo)iter.next();
			referencedInfos.add(stmInfo.getObjNum(), stmInfo);
		}
		CosList allInfos = cosDoc.buildObjectList(false);
		CosList unrefedObjs = new CosList();
		iter = allInfos.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if (referencedInfos.get(info.getObjNum()) == null) {
				CosObject obj = info.getObject();
				if (obj == null)
					obj = cosDoc.getXRef().getIndirectObject(info);
				if (obj != null)
					unrefedObjs.add(info.getObjNum(), obj);
			}
		}
		if (unrefedObjs.isEmpty())
			return null;
		CosObject[] list = new CosObject[unrefedObjs.count()];
		iter = unrefedObjs.iterator();
		int idx = 0;
		while (iter.hasNext())
			list[idx++] = (CosObject)iter.next();
		return list;
	}

	/**
	 * Finds all of the indirect COS objects that are rooted in one
	 * of the root containers specified by 'roots'. If an object
	 * cannot be found by traversing one of the trees stemming from
	 * roots, then it is not considered "referenced" for this
	 * particular invocation.
	 *
	 * @param roots - An array of the CosContainers that are the roots
	 * 				  of the trees for which we want inclusion testing
	 * 				  performed.
	 *
	 * @return a set containing all refernced CosObjects that are rooted
	 * in one of the roots containers.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	private static CosList findReferencedIndirectObjects(CosContainer[] roots, boolean fixStreamLengths)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (roots == null) return null;
		CosList foundObjects = new CosList();
		for (int i = 0; i < roots.length; i++) {
			CosContainer root = roots[i];
			if (root != null) {
				if (root.isIndirect())
					foundObjects.add(root.getInfo().getObjNum(), root.getInfo());
				findReferencedIndirectObjects(root, foundObjects, fixStreamLengths);
			}
		}
		return foundObjects;
	}

	/**
	 * Finds all of the indirect COS objects that are rooted in root and
	 * adds them to the set of found objects.
	 *
	 * @param container - The root container whose tree of objects is to be
	 * 				 added to the set foundObjects.
	 *
	 * @param foundObjects - The set of found objects to which newly
	 * 						 discovered refernced objects are to be
	 * 						 added. foundObjects is a parameter so that
	 * 						 a history can be maintained of previously
	 * 						 found objects in order to prevent infinite
	 * 						 recursion.
	 * @throws PDFCosParseException
	 * @throws PDFIOException
	 * @throws PDFSecurityException
	 */
	private static void findReferencedIndirectObjects(CosContainer container, CosList foundObjects, boolean fixStreamLengths)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		ArrayList stateStack = new ArrayList();
		Iterator containerIter = null;
		CosObject containerItem = null;
		boolean pushState = true;
		while (true) {
			if (pushState) {
				if (container instanceof CosArray) {
					containerIter = ((CosArray)container).iterator();
				} else {
					if (fixStreamLengths && container instanceof CosStream) {
						CosObject cosLength = ((CosStream)container).get(ASName.k_Length);
						if (cosLength instanceof CosNumeric && cosLength.isIndirect()) {
							cosLength = cosLength.getDocument().createCosNumeric(cosLength.numberValue());
							((CosStream)container).put(ASName.k_Length, cosLength);
						}
					}
					List<ASName> keys = ((CosDictionary)container).getKeys();
					containerIter = keys.iterator();
				}
				pushState = false;
			}
			while (containerIter.hasNext()) {
				if (container instanceof CosArray) {
					containerItem = (CosObject)containerIter.next();
				} else {
					ASName key = (ASName)containerIter.next();
					containerItem = ((CosDictionary)container).get(key);
				}
				if (containerItem != null) {
					CosObjectInfo itemInfo = containerItem.getInfo();
					if (itemInfo != null) {
						if (foundObjects.get(itemInfo.getObjNum()) != null)
							continue;
						foundObjects.add(itemInfo.getObjNum(), itemInfo);
					}
					if (containerItem instanceof CosContainer) {
						pushState = true;
						break;
					}
				}
			}
			if (pushState) {
				stateStack.add(container);
				stateStack.add(containerIter);
				container = (CosContainer)containerItem;
			} else {
				if (stateStack.isEmpty())
					return;
				containerIter = (Iterator)stateStack.get(stateStack.size() - 1);
				stateStack.remove(stateStack.size() - 1);
				container = (CosContainer)stateStack.get(stateStack.size() - 1);
				stateStack.remove(stateStack.size() - 1);
			}
		}
	}

	/**
	 * Removes new unreferenced objects in the incremental save case. Only
	 * checks objects that have been created since the last open/save. Dirty
	 * objects that refer to previously saved instances must be emitted by
	 * definition. These dirty objects form a basis of reference for other
	 * objects that have been newly created. Any newly created objects not
	 * referenced from that base must not be referenced at all, so they are
	 * marked free and not emitted.
	 * @throws IOException 
	 */
	static void freeUnreferencedObjectsIncremental(CosDocument cosDoc, int objNumLimit)
		throws PDFCosParseException, PDFIOException, PDFSecurityException, IOException
	{
		CosList oldDirtyObjects = new CosList();
		CosList newDirtyObjects = cosDoc.buildObjectList(true);
		Iterator iter = newDirtyObjects.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if (info.getObjNum() >= objNumLimit)
				break;
			oldDirtyObjects.add(info.getObjNum(), info);
			iter.remove();
		}
		CosList referencedDirtyObjects = new CosList();
		iter = oldDirtyObjects.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			findReferencedIndirectObjectsIncremental(info.getObject(), referencedDirtyObjects);
		}
		findReferencedIndirectObjectsIncremental(cosDoc.getTrailer(), referencedDirtyObjects);
		iter = newDirtyObjects.iterator();
		while (iter.hasNext()) {
			CosObjectInfo info = (CosObjectInfo)iter.next();
			if (referencedDirtyObjects.get(info.getObjNum()) == null) {
				CosObject obj = info.getObject(false);
				if (obj != null)
					obj.close();
				info.markFree();
			}
		}
	}

	/**
	 * Here we check the references in an object tree. We always start with
	 * an OLD dirty object from the list, so we we don't need to record the
	 * reference for that one. Only objects that are new this time can be
	 * candidates for removal. We go down the object tree and stop when we
	 * find a reference to an object that's not dirty or that we already saw.
	 * Direct objects are never recorded in the reference list, but direct
	 * containers must be processed for their contents.
	 */
	private static void findReferencedIndirectObjectsIncremental(CosObject obj, CosList list)
		throws PDFCosParseException, PDFIOException, PDFSecurityException
	{
		if (!(obj instanceof CosContainer))
			return;
		CosContainer container = (CosContainer)obj;
		ArrayList stateStack = new ArrayList();
		Iterator containerIter = null;
		CosObject containerItem = null;
		boolean pushState = true;
		while (true) {
			if (pushState) {
				if (container instanceof CosArray) {
					containerIter = ((CosArray)container).iterator();
				} else {
					List<ASName> keys = ((CosDictionary)container).getKeys();
					containerIter = keys.iterator();
				}
				pushState = false;
			}
			while (containerIter.hasNext()) {
				if (container instanceof CosArray) {
					containerItem = (CosObject)containerIter.next();
				} else {
					ASName key = (ASName)containerIter.next();
					containerItem = ((CosDictionary)container).get(key);
				}
				if (containerItem != null) {
					CosObjectInfo itemInfo = containerItem.getInfo();
					if (itemInfo != null) {
						if (!itemInfo.isDirty())
							continue;
						if (list.get(itemInfo.getObjNum()) != null)
							continue;
						list.add(containerItem.getInfo().getObjNum(), containerItem.getInfo());
					}
					if (containerItem instanceof CosContainer) {
						pushState = true;
						break;
					}
				}
			}
			if (pushState) {
				stateStack.add(container);
				stateStack.add(containerIter);
				container = (CosContainer)containerItem;
			} else {
				if (stateStack.isEmpty())
					return;
				containerIter = (Iterator)stateStack.get(stateStack.size() - 1);
				stateStack.remove(stateStack.size() - 1);
				container = (CosContainer)stateStack.get(stateStack.size() - 1);
				stateStack.remove(stateStack.size() - 1);
			}
		}
	}
	
}
