/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright 2007 Adobe Systems Incorporated All Rights Reserved.
 *
 * NOTICE: All information contained herein is, and remains the property of
 * Adobe Systems Incorporated and its suppliers, if any. The intellectual and
 * technical concepts contained herein are proprietary to Adobe Systems
 * Incorporated and its suppliers and may be covered by U.S. and Foreign
 * Patents, patents in process, and are protected by trade secret or copyright
 * law. Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained from
 * Adobe Systems Incorporated.
 */
package com.adobe.xfa.formcalc;


import com.adobe.xfa.ut.ExFull;


/**
 * This class defines static methods to implement
 * the FormCalc arithmetic calculations.
 *
 *  U R L   F U N C T I O N S
 *      get, put, ost.
 *
 * @author Mike P. Tardif
 *
 * @exclude from published api.
 */
final class BuiltinUrl {

	/*
	 *  Disallow instances of this class.
	 */
	private BuiltinUrl() {
	}

	/*
	 *  Get   
	 *      This function download the contents of the given url.
	 *
	 */
	static void Get(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null;
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 1);
			Builtins.maxArgs(nArgs, 1);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			Builtins.limitNullArgs(oParser, nArgs, oArgSym);
			//
			// get the url arg.
			//
			String sUrl = oParser.getString(oArgSym[0]);
			String sRes = null;
			//
			// if a protocolhost exists then download the url.
			//
			ProtocolHost oProtocolHost = oParser.moProtocolHost;
			if (oProtocolHost != null) {
				CalcSymbol oSym = null;
				try {
					oSym = oProtocolHost.getUrl(sUrl);
					sRes = oParser.getString(oSym);
					CalcSymbol.delete(oSym, oParser);
				} catch (ExFull oException) {
					String sErr = oException.toString();
					if (sErr != null) {
						throw new CalcException(sErr);
					}
					else {
						throw new CalcException();
					}
				} catch (CalcException e) {
					if (oSym != null)
						CalcSymbol.delete(oSym, oParser);
					throw e;
				}
			}
			oRetSym = new CalcSymbol(sRes);
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}


	/*
	 *  Put   
	 *      This function upload the contents of the given url.
	 *
	 */
	static void Put(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null;
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 2);
			Builtins.maxArgs(nArgs, 3);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			Builtins.limitNullArgs(oParser, nArgs, oArgSym);
			//
			// get the url arg.
			//
			String sUrl = oParser.getString(oArgSym[0]);
			String sData = oParser.getString(oArgSym[1]);
			String sEnc = (nArgs > 2) ? oParser.getString(oArgSym[2]) : "UTF-8";
			String sRes = null;
			//
			// if a protocolhost exists then upload the data to the url.
			//
			ProtocolHost oProtocolHost = oParser.moProtocolHost;
			if (oProtocolHost != null) {
				CalcSymbol oSym = null;
				try {
					oSym = oProtocolHost.putUrl(sUrl, sData, sEnc);
					sRes = oParser.getString(oSym);
					CalcSymbol.delete(oSym, oParser);
				} catch (ExFull oException) {
					String sErr = oException.toString();
					if (sErr != null) {
						throw new CalcException(sErr);
					}
					else {
						throw new CalcException();
					}
				} catch (CalcException e) {
					if (oSym != null)
						CalcSymbol.delete(oSym, oParser);
					throw e;
				}
			}
			oRetSym = new CalcSymbol(sRes);
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}


	/*
	 *  Post   
	 *      This function osts the given data to the given url.
	 *
	 */
	static void Post(CalcParser oParser, CalcSymbol[] oArgSym) {
		final int nArgs = oArgSym.length;
		CalcSymbol oRetSym = null;
		try {
			//
			// check the number of args vs the number required.
			//
			Builtins.minArgs(nArgs, 2);
			Builtins.maxArgs(nArgs, 5);
			//
			// check for error-valued, return-valued and null-valued args.
			//
			Builtins.limitExceptionArgs(oArgSym);
			Builtins.limitNullArgs(oParser, nArgs, oArgSym);
			//
			// get the url arg.
			//
			String sUrl = oParser.getString(oArgSym[0]);
			String sData = oParser.getString(oArgSym[1]);
			String sContent = (nArgs > 2) ? oParser.getString(oArgSym[2])
										: "application/octet-stream";
			String sEnc = (nArgs > 3) ? oParser.getString(oArgSym[3]) : "UTF-8";
			String sHead = (nArgs > 4) ? oParser.getString(oArgSym[4]) : "";
			String sRes = null;
			//
			// if a protocolhost exists then upload the data to the url.
			//
			ProtocolHost oProtocolHost = oParser.moProtocolHost;
			if (oProtocolHost != null) {
				CalcSymbol oSym = null;
				try {
					oSym = oProtocolHost.postUrl(sUrl, sHead, sData,
													sContent, sEnc);
					sRes = oParser.getString(oSym);
					CalcSymbol.delete(oSym, oParser);
				} catch (ExFull oException) {
					String sErr = oException.toString();
					if (sErr != null) {
						throw new CalcException(sErr);
					}
					else {
						throw new CalcException();
					}
				} catch (CalcException e) {
					if (oSym != null)
						CalcSymbol.delete(oSym, oParser);
					throw e;
				}
			}
			oRetSym = new CalcSymbol(sRes);
		} catch (CalcException e) {
			oRetSym = e.getSymbol();
			if (oRetSym.getType() != CalcSymbol.TypeNull)
				oParser.mbInThrow = true;
		}
		//
		// push the result on the stack.
		//
		oParser.mStack.push(oRetSym);
	}
}