/*************************************************************************
 *
 * ADOBE CONFIDENTIAL
 * ___________________
 *
 *  Copyright 2012 Adobe Systems Incorporated
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 **************************************************************************/
package com.day.cq.wcm.designimporter.api;

import com.day.cq.wcm.designimporter.CanvasBuildOptions;
import com.day.cq.wcm.designimporter.DesignImportException;
import com.day.cq.wcm.designimporter.DesignImporterContext;

import javax.jcr.RepositoryException;
import java.io.InputStream;

/**
 * Defines contract for building a canvas from a raw HTML document stream.
 *
 * <p>
 * The CanvasBuilder is responsible for building a canvas component tree,
 * along with the top level canvas component script as well as the design and clientlibs.
 * A CanvasBuilder builds canvas artifacts for one html document. If a design package contains
 * multiple importable html documents, multiple CanvasBuilders hook in to accomplish import.
 * </p>
 *
 * <p>
 * The CanvasBuilder should dispatch appropriate events during major events of its lifetime. These
 * events, listed as they appear in order, are tabulated as follows:
 * <table border="1">
 *     <tr><th>Event</th><th>Description</th></tr>
 *     <tr><td>EVENT_STARTING</td><td>Dispatched before the CanvasBuilder begins building. May be used for any pre-processing</td></tr>
 *     <tr><td>EVENT_PARSE_COMPLETE</td><td>Dispatched when the html document stream has been successfully parsed.</td></tr>
 *     <tr><td>EVENT_COMPONENT_TREE_BUILT</td><td>Dispatched after the canvas component tree has been persisted to the repository</td></tr>
 *     <tr><td>EVENT_CANVAS_SCRIPT_BUILT</td><td>Dispatched after the canvas component script has been generated and persisted successfully</td></tr>
 *     <tr><td>EVENT_COMPLETE</td><td>Marks the completion of canvas building.</td></tr>
 * </table>
 * </p>
 *
 * <p>
 * The bundled implementation of the CanvasBuilder also attaches contextual information to the above events. This information comes handy
 * to find various state parameters and process accordingly.</p>
 *
 * <p>
 * The contextual information is sent in form of event payload, accessible via {@link org.osgi.service.event.Event#getProperty(String)}.
 * The below table enlists various event properties along with their type, description and the events the payload is available with.
 * <table border="1">
 *     <tr>
 *         <th>Property</th>
 *         <th>Type</th>
 *         <th>Description</th>
 *         <th>Available</th>
 *     </tr>
 *     <tr>
 *         <td>designImporterContext</td>
 *         <td>{@link DesignImporterContext}</td>
 *         <td>The design importer context associated with the canvas build operation</td>
 *         <td>ALL</td>
 *     </tr>
 *     <tr>
 *         <td>parseResult</td>
 *         <td>{@link com.day.cq.wcm.designimporter.parser.ParseResult}</td>
 *         <td>Encapsulates the result of html document parsing. Can be used to access the markup formed and/or cq components culled</td>
 *         <td>EVENT_PARSE_COMPLETE and all events onward</td>
 *     </tr>
 *     <tr>
 *         <td>canvas</td>
 *         <td>{@link org.apache.sling.api.resource.Resource}</td>
 *         <td>The resource representing the root of the canvas component tree built</td>
 *         <td>EVENT_COMPONENT_TREE_BUILT and all events onward</td>
 *     </tr>
 * </table>
 * </p>
 */
public interface CanvasBuilder {

    String EVENT_STARTING = "com/day/cq/wcm/designimporter/CanvasPageBuilder/STARTING";

    String EVENT_PARSE_COMPLETE = "com/day/cq/wcm/designimporter/CanvasPageBuilder/PARSE_COMPLETE";

    String EVENT_COMPONENT_TREE_BUILT = "com/day/cq/wcm/designimporter/CanvasPageBuilder/COMPONENT_TREE_BUILT";

    String EVENT_CANVAS_SCRIPT_BUILT = "com/day/cq/wcm/designimporter/CanvasPageBuilder/CANVAS_SCRIPT_BUILT";

    String EVENT_CLIENTLIBS_BUILT = "com/day/cq/wcm/designimporter/CanvasPageBuilder/CLIENTLIBS_BUILT";

    String EVENT_COMPLETE = "com/day/cq/wcm/designimporter/CanvasPageBuilder/COMPLETE";

    /**
     * The name of the OSGi property that defines the name pattern of the HTML file
     * that this CanvasPageBuilder is capable of translating.
     */
    String PN_FILEPATTERN = "filepattern";

    /**
     * Builds the canvas for the input HTML document stream and the import context.
     *
     * @param htmlStream            The InputStream representing the the HTML document that needs
     *                              to be translated to CQ Page
     * @param designImporterContext The context associated with the design package import request
     * @throws RepositoryException   When an exception occurs while writing to the repository
     * @throws DesignImportException When an import exception occurs
     */
    void build(InputStream htmlStream, DesignImporterContext designImporterContext) throws RepositoryException, DesignImportException;

    /**
     * Builds the canvas for the input HTML document stream and the import context.
     *
     * @param htmlStream            The InputStream representing the the HTML document that needs
     *                              to be translated to CQ Page
     * @param designImporterContext The context associated with the design package import request
     * @param buildOptions          Set of option flags to control the way the page(and everything around it) is built
     * @throws RepositoryException   When an exception occurs while writing to the repository
     * @throws DesignImportException When an import exception occurs
     */
    void build(InputStream htmlStream, DesignImporterContext designImporterContext, CanvasBuildOptions buildOptions) throws RepositoryException,
            DesignImportException;

}
