/*
 * Copyright 1997-2008 Day Management AG
 * Barfuesserplatz 6, 4001 Basel, Switzerland
 * All Rights Reserved.
 *
 * This software is the confidential and proprietary information of
 * Day Management AG, ("Confidential Information"). You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Day.
 */
package com.day.cq.wcm.msm.api;

import com.day.cq.commons.JSONItem;

import java.util.List;

/**
 * Defines a live relationship of a synced resource between its blueprint and
 * livecopy. for example, if a site (blueprint) "/content/geometrixx"
 * has a livecopy in "/content/copy", then the resource of "/content/geometrixx/en/jcr:content"
 * and "/content/copy/en/jcr:content" form a relationship.
 *
 * This is also extensible for components within a page, for example the
 * "/content/geometrixx/en/jcr:content/par/title" and
 * "/content/copy/en/jcr:content/par/title" can form a relationship.
 *
 * In contrast to the {@link LiveCopy} the LiveRelationship defines the "dynamic"
 * relation between a source and a synced target. where as the former only
 * defines its configuration.
 *
 * Please note that its not possible to have different relative paths from the
 * roots to the source/target resources. This could happen if a source or
 * target resource are moved or renamed. Having trees that are not in sync
 * hirarchy wise will probably end in chaos sooner or later. Such operations
 * need to be prohibited or the respective changes need to be rolledout to
 * all targets.
 */
public interface LiveRelationship extends JSONItem {

    /**
     * Returns the Live Copy configuration of this relationship
     * @return the Live Copy
     */
    LiveCopy getLiveCopy();

    /**
     * Returns the relative path of this relationship from the sync roots to
     * the actual resources.
     *
     * for example:
     * "/en/jcr:content/par/title" for a paragraph relationship,
     * or "/en/jcr:content" for a page relationship
     *
     * @return the relative path
     */
    String getSyncPath();

    /**
     * Returns the absolute path of the source resource of this relationship.
     * This is a convenience method for
     * <code>lr.getLiveCopy().getBlueprintPath() + lr.getSyncPath()</code>
     *  
     * @return the source path
     */
    String getSourcePath();

    /**
     * Returns the absolute path of the live sync resource.
     * This is a convenience method for
     * <code>lr.getLiveCopy().getPath() + lr.getSyncPath()</code>
     *
     * @return the target path
     */
    String getTargetPath();

    /**
     * Returns the status of this relationship
     * @return the status
     */
    LiveStatus getStatus();

    /**
     * Returns the computed {@link RolloutConfig} of this relationship (blueprint configs + live copy configs)
     * @return the computed configs config
     */
    List<RolloutConfig> getRolloutConfigs();

    /**
     * Returns the computed {@link RolloutConfig} of this relationship (blueprint configs + live copy configs)
     * filtered with the specified trigger.
     * @param trigger Filters the list with the specified trigger
     * @return the computed configs config
     */
    List<RolloutConfig> getRolloutConfigs(RolloutManager.Trigger trigger);
    
/**
     * Returns if the current relationship contains at least one {@link RolloutConfig} with the
     * specified trigger.
     * @param trigger the trigger to check
     * @return true if at least one config as specified trigger, false otherwise.
     */
    boolean isTrigger(RolloutManager.Trigger trigger);
}
