/*
 * ADOBE CONFIDENTIAL
 * __________________
 *
 *  Copyright 2020 Adobe
 *  All Rights Reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe.
 */

package com.adobe.cq.dam.download.api;

import java.util.Calendar;
import java.util.Collection;

import org.osgi.annotation.versioning.ProviderType;

/**
 * <p>
 * The information that will be provided to consumers when they request the status of
 * a download being generated by the {@link DownloadService}.
 * </p>
 * <p>
 * The download status is the primary means of delivering information about the service's
 * output to the consumer. This includes:
 * </p>
 * <ul>
 *  <li>The current progress of the operation that generates download artifacts.</li>
 *  <li>URIs to the download artifacts once they're available.</li>
 *  <li>Error information for any targets that may have failed to process correctly.</li>
 * </ul>
 * <p>
 * See {@link DownloadService#getProgress(String, org.apache.sling.api.resource.ResourceResolver)}
 * for how to retrieve the status of a download.
 * </p>
 * @see DownloadService
 */
@ProviderType
public interface DownloadProgress {

    public enum Status {PROCESSING, SUCCESSFUL, PARTIALLY_SUCCESSFUL, FAILED, POSTPROCESSING};
    
    /**
     * Answer the user who requested the download.
     * @return ID of the user that requested the download.
     */
    String getRequestingUser();
    
    /**
     * Retrieves the current status of the processing job.
     * @return Arbitrary status text.
     */
    Status getStatus();

    /**
     * Retrieves the total number of items that are included in the download.
     * @return Number of download items.
     */
    int getTotalCount();

    /**
     * Retrieves the total size, in bytes, of all items included in the download.
     * @return Size in bytes.
     */
    long getTotalSize();

    /**
     * Retrieves the total number of items that have been processed, and have
     * either succeeded or failed.
     * @return Number of processed download items.
     */
    int getProcessedCount();

    /**
     * Retrieves the current percentage complete value of the download process.
     * @return A percent value between 0 and 100.
     */
    int getProgress();

    /**
     * Of the total number of items in the download, the number of items that failed
     * to process correctly.
     * @return Number of failed download items.
     */
    int getFailureCount();

    /**
     * Of the total number of items in the download, the number of items that processed
     * correctly.
     * @return Number of successful download items.
     */
    int getSuccessCount();
    
    /**
     * Indicates whether or not the entire download creation process has finished.
     * @return <code>true</code> if the process is done, <code>false</code> otherwise.
     */
    boolean isComplete();

    /**
    * Return the date that the archive process started.
    * @return The date that processing started.
    */
    Calendar getStarted();
    
    /**
     * Return the date that the archive process started.
     * @return The date that processing finished.
     */
    Calendar getFinished();

    /**
     * Return the date at which the download will expire. Note that it is possible
     * that a download will be unavailable before the expiry date. This is the
     * latest date that a download could possibly be available
     * @return The expiry date for a download
     */
    Calendar getExpiry();

    /**
     * Retrieves all the artifacts that were the final output of the download process.
     * Artifact information will include the URI where an artifact can be retrieved,
     * or a description of why the artifact failed.
     * @return The artifacts that are the output of the download.
     */
    Collection<DownloadArtifact> getArtifacts();
    
    /**
     * Retrieves the artifacts with the passed id.
     * Artifact information will include the URI where an artifact can be retrieved,
     * or a description of why the artifact failed.
     * @return The artifact represented by the passed id or null if no artifact with this name exists.
     */
    DownloadArtifact getArtifact(String artifactId);
}
